//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Bitmap.h"
#include <Candera/Macros.h>
#include <Candera/System/Mathematics/Math.h>

// Diagnostics is included for reference but commented out because it is not used
// #include <Candera/System/Diagnostics/TraceConstants/TraceConstantsCore.h>
// using namespace Candera::Diagnostics::Core;

namespace Candera {
    using namespace MemoryManagement;


/******************************************************************************
*  Create
******************************************************************************/
Bitmap::SharedPointer Bitmap::Create(UInt16 width, UInt16 height, Bitmap::PixelFormat format, PackAlignment alignment, 
                                     const UInt8* pixels, DisposerFn disposerFn, UInt32 size /*= 0*/, bool isVerticallyFlipped /*= true*/)
{
    return SharedPointer(CANDERA_NEW(Bitmap)(width, height, format, alignment, pixels, disposerFn, size, isVerticallyFlipped));
}

Bitmap::SharedPointer Bitmap::Create(UInt16 width, UInt16 height, Bitmap::PixelFormat format, PackAlignment alignment,
                                     UInt8* pixels, DisposerFn disposerFn, UInt32 size /*= 0*/, bool isMutable /*= true*/, bool isVerticallyFlipped /*= true*/)
{
    return SharedPointer(CANDERA_NEW(Bitmap)(width, height, format, alignment, pixels, disposerFn, size, isMutable, isVerticallyFlipped));
}

Bitmap::SharedPointer Bitmap::Create(UInt16 width, UInt16 height, Int format, PackAlignment alignment,
                                     const UInt8* pixels, DisposerFn disposerFn, UInt32 size /*= 0*/, bool isVerticallyFlipped /*= true*/)
{
    return SharedPointer(CANDERA_NEW(Bitmap)(width, height, format, alignment, pixels, disposerFn, size, isVerticallyFlipped));
}

Bitmap::SharedPointer Bitmap::Create(UInt16 width, UInt16 height, Int format, PackAlignment alignment,
                                     UInt8* pixels, DisposerFn disposerFn, UInt32 size/* = 0*/, bool isMutable/* = true*/, bool isVerticallyFlipped /*= true*/)
{
    return SharedPointer(CANDERA_NEW(Bitmap)(width, height, format, alignment, pixels, disposerFn, size, isMutable, isVerticallyFlipped));
}

Bitmap::SharedPointer Bitmap::Create(UInt16 width, UInt16 height, Bitmap::PixelFormat format, PackAlignment alignment,
                                     const ResourceDataHandle& pixelsResourceHandle, bool isVerticallyFlipped /*= true*/)
{
    return SharedPointer(CANDERA_NEW(Bitmap)(width, height, format, alignment, pixelsResourceHandle, isVerticallyFlipped));
}

Bitmap::SharedPointer Bitmap::Create(UInt16 width, UInt16 height, Int format, PackAlignment alignment,
                                     const ResourceDataHandle& pixelsResourceHandle, bool isVerticallyFlipped /*= true*/)
{
    return SharedPointer(CANDERA_NEW(Bitmap)(width, height, format, alignment, pixelsResourceHandle, isVerticallyFlipped));
}



Bitmap::Bitmap(UInt16 width, UInt16 height, Bitmap::PixelFormat format,
               PackAlignment alignment, const UInt8* pixels, DisposerFn disposerFn,
               UInt32 size, bool isVerticallyFlipped):
               Base(),
               m_isVerticallyFlipped(isVerticallyFlipped),
               m_packAlignment(alignment),
               m_pixelFormat(format),
               m_width(width),
               m_height(height),
               m_pixelsResourceHandle(PixelsResource::CreateHandle(pixels, disposerFn, size)),
               m_ninePatchProperties(),
               m_center()
{
}

Bitmap::Bitmap(UInt16 width, UInt16 height, Bitmap::PixelFormat format,
    PackAlignment alignment, UInt8* pixels, DisposerFn disposerFn,
    UInt32 size, bool isMutable, bool isVerticallyFlipped):
    Base(),
    m_isVerticallyFlipped(isVerticallyFlipped),
    m_packAlignment(alignment),
    m_pixelFormat(format),
    m_width(width),
    m_height(height),
    m_pixelsResourceHandle(PixelsResource::CreateHandle(pixels, disposerFn, size, isMutable)),
    m_ninePatchProperties(),
    m_center()
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(818, CANDERA_LINT_REASON_MUTABLE)
}

Bitmap::Bitmap(UInt16 width, UInt16 height, Int format,
    PackAlignment alignment, const UInt8* pixels, DisposerFn disposerFn,
    UInt32 size, bool isVerticallyFlipped):
    Base(),
    m_isVerticallyFlipped(isVerticallyFlipped),
    m_packAlignment(alignment),
    m_pixelFormat(format),
    m_width(width),
    m_height(height),
    m_pixelsResourceHandle(PixelsResource::CreateHandle(pixels, disposerFn, size)),
    m_ninePatchProperties(),
    m_center()
{
}

Bitmap::Bitmap(UInt16 width, UInt16 height, Int format,
    PackAlignment alignment, UInt8* pixels, DisposerFn disposerFn,
    UInt32 size, bool isMutable, bool isVerticallyFlipped):
    Base(),
    m_isVerticallyFlipped(isVerticallyFlipped),
    m_packAlignment(alignment),
    m_pixelFormat(format),
    m_width(width),
    m_height(height),
    m_pixelsResourceHandle(PixelsResource::CreateHandle(pixels, disposerFn, size, isMutable)),
    m_ninePatchProperties(),
    m_center()
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(818, CANDERA_LINT_REASON_MUTABLE)
}


Bitmap::Bitmap(UInt16 width, UInt16 height, Bitmap::PixelFormat format,
               PackAlignment alignment, const ResourceDataHandle& pixelsResourceHandle, bool isVerticallyFlipped):
               Base(),
               m_isVerticallyFlipped(isVerticallyFlipped),
               m_packAlignment(alignment),
               m_pixelFormat(format),
               m_width(width),
               m_height(height),
               m_pixelsResourceHandle(pixelsResourceHandle),
               m_ninePatchProperties(),
               m_center()
{
}

Bitmap::Bitmap(UInt16 width, UInt16 height, Int format,
               PackAlignment alignment, const ResourceDataHandle& pixelsResourceHandle, bool isVerticallyFlipped):
               Base(),
               m_isVerticallyFlipped(isVerticallyFlipped),
               m_packAlignment(alignment),
               m_pixelFormat(format),
               m_width(width),
               m_height(height),
               m_pixelsResourceHandle(pixelsResourceHandle),
               m_ninePatchProperties(),
               m_center()
{
}

Bitmap::~Bitmap()
{
    DisposePixels();
}

UInt8* Bitmap::GetPixels()
{
    return PixelsResource(GetPixelsResourceHandle()).GetMutableData();
}

bool Bitmap::IsMutable() const
{
    return m_pixelsResourceHandle.m_isMutable;
}

const UInt8* Bitmap::GetPixels() const
{
    return PixelsResource(GetPixelsResourceHandle()).GetData();
}

void Bitmap::SetNext(SharedPointer bitmap)
{
    m_next = bitmap;
}

const Bitmap::SharedPointer& Bitmap::GetNext() const
{
    return m_next;
}

bool Bitmap::HasNext() const
{
    return GetNext() != 0;
}

void Bitmap::SetNinePatchproperties(const FeatStd::Optional<NinePatchProperties>& ninePatchProperties)
{
    m_ninePatchProperties = ninePatchProperties;
}

bool Bitmap::IsNinePatch() const
{
    return (m_ninePatchProperties.IsSet());
}

void Bitmap::SetCenter(const FeatStd::Optional<Candera::Vector2>& center)
{
    m_center = center;
}

void Bitmap::DisposePixels()
{
    PixelsResource::DisposeData(m_pixelsResourceHandle);
}

const Bitmap::PixelFormatProperties Bitmap::c_PixelFormatProperties[] =
{
    // isCompressed
    { false, 16 }, // RgbUnsignedShort565PixelFormat
    { false, 16 }, // RgbaUnsignedShort4444PixelFormat
    { false, 16 }, // RgbaUnsignedShort5551PixelFormat
    { false, 16 }, // DepthUnsignedShortPixelFormat
    { false, 24 }, // RgbUnsignedBytePixelFormat
    { false, 32 }, // RgbaUnsignedBytePixelFormat
    { false,  8 }, // AlphaUnsignedBytePixelFormat
    { false,  8 }, // LuminanceUnsignedBytePixelFormat
    { false, 16 }, // LuminanceAlphaUnsignedBytePixelFormat
    { false, 32 }, // CoverageUnsignedBytePixelFormat, Extension: GL_NV_coverage_sample
    { true,   4 }, // Etc2CompressedRgbPixelFormat
    { true,   4 }, // Etc2CompressedSrgbPixelFormat
    { true,   4 }, // Etc2Alpha1CompressedRgbaPixelFormat
    { true,   4 }, // Etc2Alpha1CompressedSrgbaPixelFormat
    { true,   8 }, // Etc2EacCompressedRgbaPixelFormat
    { true,   8 }, // Etc2EacCompressedSrgbaPixelFormat
    { true,   4 }, // EacCompressedUnsignedRPixelFormat
    { true,   4 }, // EacCompressedSignedRPixelFormat
    { true,   8 }, // EacCompressedUnsignedRGPixelFormat
    { true,   8 }, // EacCompressedSignedRGPixelFormat
    { false, 24 }, // SrgbUnsignedBytePixelFormat
    { false, 32 }, // SrgbaUnsignedBytePixelFormat
    { false, 32 }, // RgbFloatingPointR11G11B10PixelFormat
    { false, 32 }, // RgbFloatingPoint9E5PixelFormat
    { false, 32 }, // RgbaUnsignedIntRev10A2PixelFormat
    { true,   4 }, // Dxt1CompressedRgbFormat
    { true,   4 }, // Dxt1CompressedRgbaFormat
    { true,   8 }, // Dxt3CompressedRgbaFormat
    { true,   8 }  // Dxt5CompressedRgbaFormat
};

Candera::UInt32 Bitmap::GetSize() const
{
    FEATSTD_COMPILETIME_ASSERT((sizeof(c_PixelFormatProperties) / sizeof(PixelFormatProperties)) == PixelFormatCount);

    UInt32 size = m_pixelsResourceHandle.m_size;

    // Calculate size.
    if (0 == m_pixelsResourceHandle.m_size) {
        const PixelFormat pixelFormat = GetPixelFormat();

        if (pixelFormat < PixelFormatCount) {
            size = c_PixelFormatProperties[pixelFormat].BitsPerPixel;

            // compressed formats
            if ((pixelFormat >= Etc2CompressedRgbPixelFormat) && (pixelFormat <= EacCompressedSignedRGPixelFormat)) {
                // All ETC formats store compressed data in 4x4 blocks. I.e. images smaller than 4x4 are stored as 4x4.
                // size calculation: https://www.khronos.org/opengles/sdk/docs/man3/html/glCompressedTexImage2D.xhtml where last value is bytes per block 
                const UInt32 bytesPerBlock = (size * 16/*4x4 block*/) >> 3;
                const UInt16 minCompressedBlockSize = 4;
                const Float width = static_cast<Float>(GetWidth()) / static_cast<Float>(minCompressedBlockSize);
                const Float height = static_cast<Float>(GetHeight()) / static_cast<Float>(minCompressedBlockSize);
                size = (static_cast<UInt32>(Math::Ceil(width) * Math::Ceil(height)) * bytesPerBlock);
            }
            // uncompressed formats
            else {
                size *= static_cast<UInt32>(GetWidth())* static_cast<UInt32>(GetHeight());
                size >>= 3; // divide by 8 to return size in bytes
            }
        }
    }

    return size;
}

Int Bitmap::MapDeprecatedEnumsToPixelFormat(Int format, Type type)
{
    Int result = -1;

    switch(format) {
    case RgbFormat:
        switch(type) {
        case UnsignedShort565Type:
            result = RgbUnsignedShort565PixelFormat;
            break;
        case UnsignedByteType:
            result = RgbUnsignedBytePixelFormat;
            break;
        default:
            //Invalid format / type combination.
            result = -1;
        }
        break;
    case RgbaFormat:
        switch(type) {
        case UnsignedShort4444Type:
            result = RgbaUnsignedShort4444PixelFormat;
            break;
        case UnsignedShort5551Type:
            result = RgbaUnsignedShort5551PixelFormat;
            break;
        case UnsignedByteType:
            result = RgbaUnsignedBytePixelFormat;
            break;
        default:
            //Invalid format / type combination.
            result = -1;
        }
        break;
    case AlphaFormat:
        switch(type) {
        case UnsignedByteType:
            result = AlphaUnsignedBytePixelFormat;
            break;
        default:
            //Invalid format / type combination.
            result = -1;
        }
        break;
    case LuminanceFormat:
        switch(type) {
        case UnsignedByteType:
            result = LuminanceUnsignedBytePixelFormat;
            break;
        default:
            //Invalid format / type combination.
            result = -1;
        }
        break;
    case LuminanceAlphaFormat:
        switch(type) {
        case UnsignedByteType:
            result = LuminanceAlphaUnsignedBytePixelFormat;
            break;
        default:
            //Invalid format / type combination.
            result = -1;
        }
        break;
    case DepthFormat:
        switch(type) {
        case UnsignedShortType:
            result = DepthUnsignedShortPixelFormat;
            break;
        default:
            //Invalid format / type combination.
            result = -1;
        }
        break;
    case CoverageFormat:
        switch(type) {
        case UnsignedByteType:
            result = CoverageUnsignedBytePixelFormat;
            break;
        default:
            //Invalid format / type combination.
            result = -1;
        }
        break;
    default:
        result = format; //Custom format.
    }
    
    return result;
}

Int Bitmap::MapPixelFormatToDeprecatedFormat(Int format)
{
    Int result = 0;

    switch (format) {
    case RgbUnsignedShort565PixelFormat:
        CANDERA_LINT_FALLTHROUGH()
    case RgbUnsignedBytePixelFormat:
        result = RgbFormat;
        break;
    case RgbaUnsignedShort4444PixelFormat:
        CANDERA_LINT_FALLTHROUGH()
    case RgbaUnsignedShort5551PixelFormat:
        CANDERA_LINT_FALLTHROUGH()
    case RgbaUnsignedBytePixelFormat:
        result = RgbaFormat;
        break;
    case AlphaUnsignedBytePixelFormat:
        result = AlphaFormat;
        break;
    case LuminanceUnsignedBytePixelFormat:
        result = LuminanceFormat;
        break;
    case LuminanceAlphaUnsignedBytePixelFormat:
        result = LuminanceAlphaFormat;
        break;
    case DepthUnsignedShortPixelFormat:
        result = DepthFormat;
        break;
    case CoverageUnsignedBytePixelFormat:
        result = CoverageFormat;
        break;
    default:
        result = format; //Custom format.
    }

    return result;
}

Bitmap::Type Bitmap::MapPixelFormatToDeprecatedType(Int format)
{
    Type result = UnsignedByteType; 

    switch (format) {
    case DepthUnsignedShortPixelFormat:
        result = UnsignedShortType;
        break;
    case RgbUnsignedShort565PixelFormat:
        result = UnsignedShort565Type;
        break;
    case RgbaUnsignedShort4444PixelFormat:
        result = UnsignedShort4444Type;
        break;
    case RgbaUnsignedShort5551PixelFormat:
        result = UnsignedShort5551Type;
        break;
    default:
        //In all other cases type is unsigned byte.
        //For custom formats type is don't care.
        result = UnsignedByteType;
    }

    return result;
}

bool Bitmap::IsMipMapChainComplete() const
{
    Bitmap* nextBitmap = m_next.GetPointerToSharedInstance();
    if (0 == nextBitmap) {
        return false;
    }

    const PixelFormat format = GetPixelFormat();
    UInt16 width = GetWidth();
    UInt16 height = GetHeight();
    PackAlignment alignment = GetPackAlignment();

    while (0 != nextBitmap) {
        if (width > 1) {
            width >>= 1;
        }

        if (height > 1) {
            height >>= 1;
        }

        if ((nextBitmap->GetPixelFormat() != format)
            || (nextBitmap->GetPackAlignment() != alignment)
            || (nextBitmap->GetWidth() != width)
            || (nextBitmap->GetHeight() != height)
            || ((!nextBitmap->HasNext()) && ((nextBitmap->GetWidth() != 1) || (nextBitmap->GetHeight() != 1))
            )) {
            return false;
        }
        nextBitmap = nextBitmap->GetNext().GetPointerToSharedInstance();
    }

    return true;
}

} // namespace Candera
