//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_ENGINEBASE_ABSTRACT_NODE_POINTER_H
#define CANDERA_ENGINEBASE_ABSTRACT_NODE_POINTER_H

#include <Candera/EngineBase/Common/CanderaObject.h>

#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Node2D.h>
#endif

#ifdef CANDERA_3D_ENABLED
#include <Candera/Engine3D/Core/Node.h>
#endif

namespace Candera {
    /** @addtogroup CommonBase
    * @{
    */
    /**
    *  @brief  The class AbstractNodePointer is an wrapper class to abstract 2D and 3D scene graph nodes.
    *
    *  The AbstractNodePointer provides an interface of common methods between 2D and 3D. Using this wrapper provides the possibility to 
    *  implement features for both 2D and 3D without separate code as long as only the provided interface is necessary.
    *  It is also possible to distinguish between 2D and 3D node types without RTTI checks. Therefore, if the feature has some differences
    *  between 2D and 3D, they can be implemented without much overhead then.
    */
    class AbstractNodePointer {
    public:
        /**
        * Marker to determine which type the node is without RTTI/casting.
        */
        enum Type
        {
#ifdef CANDERA_2D_ENABLED
            Candera2D, ///< Represents a 2D object
#endif
#ifdef CANDERA_3D_ENABLED
            Candera3D, ///< Represents a 3D object
#endif
            Undefined ///< Represents an unknown object (e.g. null pointer)
        };

        /**
        * Default abstract node pointer!
        * --> Equal to a null pointer with type: Undefined
        */
        AbstractNodePointer();

        /**
        * Creates a copy of an AbstractNodePointer.
        * @param node to be copied.
        */
        AbstractNodePointer(const AbstractNodePointer& node);

        /**
        * Checks whether the underlying pointer is a null pointer and therefore invalid or a valid pointer.
        * @return True when a node is bound to the AbstractNodePointer.
        */
        bool IsValid() const;

        /**
        * Gets the actual AbstractNodePointer type. (e.g. 2D, 3D or undefined)
        * @return The AbstractNodePointer type
        */
        Type GetType() const;

        /**
        * Returns the name of the node.
        * @return the name of the node.
        */
        const Char* GetName() const;

        bool AddChild(const AbstractNodePointer& child) const;
        bool InsertBeforeChild(const AbstractNodePointer& newChild, const AbstractNodePointer& beforeChild) const;
        bool RemoveChild(const AbstractNodePointer& child) const;

        /**
        *  Retrieves the scene graph parent of this Node
        *  @return         Pointer to parent Node, or null if there is no parent available
        */
        AbstractNodePointer GetParent() const;

        /**
        *  Gets the first child node (0 if no children). Use together with GetNextSibling to iterate like this:
        *  for (AbstractNodePointer* child = m_firstChild; child != 0; child = child->GetNextSibling()) {
        *      child->Render();
        *  }
        *  @remark     The order of the returned child is undefined.
        *  @return     Pointer to child node, or null if there is no child available
        */
        AbstractNodePointer GetFirstChild() const;

        /**
        *  Gets the next sibling (0 if no more siblings).
        *  @remark     The order of the returned sibling is undefined.
        *  @return     Pointer to sibling node, or null if there is no sibling available
        */
        AbstractNodePointer GetNextSibling() const;

        /**
        * CanderaObject is the base class which all underlying nodes of an AbstractNodePointer are derived from.
        * @return The node as a CanderaObject
        */
        CanderaObject* ToCanderaObject() const;

        /**
        * Compares a CanderaObject with the AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are equal
        */
        bool operator==(const CanderaObject* node) const;

        /**
        * Compares a CanderaObject with the AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are not equal
        */
        bool operator!=(const CanderaObject* node) const;

        /**
        * Compares two AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are equal
        */
        bool operator==(const AbstractNodePointer& node) const;

        /**
        * Compares two AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are not equal
        */
        bool operator!=(const AbstractNodePointer& node) const;

#ifdef CANDERA_2D_ENABLED
        /**
        * Creates a wrapper object of a Node2D.
        * The wrapper is an abstraction layer between 2D and 3D.
        * @param node which shall be wrapped.
        */
        explicit AbstractNodePointer(Node2D* node);

        /**
        * Tries to cast the node to Node2D. If the node is a 2D type, it will return the 
        * node as Node2D otherwise as null pointer.
        * @return node as Node2D
        */
        Node2D* ToNode2D() const;

        /**
        * Compares a Node2D with the AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are equal
        */
        bool operator==(const Node2D* node) const;

        /**
        * Compares a Node2D with the AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are not equal
        */
        bool operator!=(const Node2D* node) const;
#endif

#ifdef CANDERA_3D_ENABLED
        /**
        * Creates a wrapper object of a Node.
        * The wrapper is an abstraction layer between 2D and 3D.
        * @param node which shall be wrapped.
        */
        explicit AbstractNodePointer(Node* node);

        /**
        * Tries to cast the node to Node. If the node is a 3D type, it will return the
        * node as Node otherwise as null pointer.
        * @return node as Node
        */
        Node* ToNode() const;

        /**
        * Compares a Node (3D) with the AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are equal
        */
        bool operator==(const Node* node) const;

        /**
        * Compares a Node (3D) with the AbstractNodePointer. They are equal when both underlying node pointers are equal.
        * @param node to be compared to
        * @return True when the nodes are not equal
        */
        bool operator!=(const Node* node) const;

#endif

        /**
        * Gets the local position of this object.
        * @return The position as vector
        */
        Vector2 GetPosition() const;

        /**
        * Sets the local position of this object.
        * @param position The coordinate in a plane (e.g. x,y) as vector
        */
        void SetPosition(const Vector2& position) const;

        /**
        * Gets the local scale factor of this object.
        * @return The scale as vector
        */
        Vector2 GetScale() const;

        /**
        * Sets the local scale factor of this object. x and y defines the plane in which the node will be scaled.
        * @param scale    The scale as vector
        */
        void SetScale(const Vector2& scale) const;
      
        /**
        * Gets the local rotation of this object.
        * @return The local rotation value
        */
        Float GetRotation() const;
  
        /**
        * Sets the local rotation of this object.
        * @param rotation The local rotation angle in degrees
        */
        void SetRotation(Float rotation) const;
    
        /**
        * Gets the local pivot point of this object.
        * @return pivotPoint The local pivotPoint.
        */
        Vector2 GetPivotPoint() const;

        /**
        * Sets the local pivot point of this object.
        * @param pivotPoint The local pivotPoint.
        */
        void SetPivotPoint(const Vector2& pivotPoint);

        /**
        * Gets the scene to which the node is bound.
        * If no scene is available or the node is a null pointer, then a nullpointer will be returned.
        * @return The scene to which the node is bound
        */
        AbstractNodePointer GetScene() const;

        /**
        *  Gets the base point in local coordinate 2D space. The base point is typically in the center,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate 2D space.
        */
        void GetBasePoint(Vector2& basePoint) const;

        /**
        *  Retrieve whether rendering is enabled or not in dependency to this node's ancestors settings.
        *  @return      true if this node AND all ancestor nodes have set rendering enabled.
        *              false if either this node OR any ancestor node have set rendering disabled.
        */
        bool IsEffectiveRenderingEnabled() const;

        /**
        *  Retrieve this Node's local setting, if rendering is enabled or disabled.
        *  To compute if effective rendering is enabled in dependency to this Node's ancestors, see
        *  function "IsEffectiveRenderingEnabled" for further details.
        *  @return Whether rendering of this node is enabled (true) or disabled (false).
        */
        bool IsRenderingEnabled() const;

        /**
        *  Defines the local setting if rendering shall be enabled or not.
        *  This boolean value influences this Node's descendants. See function "IsEffectiveRenderingEnabled" for further details.
        *  @param enabled Enables(true) / Disables(false) rendering of this node.
        */
        void SetRenderingEnabled(bool enabled) const;


        /**
        * Checks whether this node is of type typeId.
        * @param typeId type which this node is suspected to be.
        * @return True when the expected type and the actual type of the node are equal.
        */
        bool IsTypeOf(FeatStd::TypeId typeId) const
        {
            return (0 != m_node) ? m_node->IsTypeOf(typeId) : false;
        }

#if defined(CANDERA_LAYOUT_ENABLED)       
        
        /**
        *  \brief  Indicates that the layout is invalid.
        *
        *  To trigger a scene to be laid out only once per Update/Render loop, all children of the scene (also widgets) can invalidate the scene layout by calling
        *  Scene::SetLayoutInvalid(). Before rendering is triggered by the application, Scene::ValidataLayout() can be called to trigger the layout if needed.
        */
        void InvalidateLayout() const;

        /**
        *  Gets the layout strategy for the child nodes.
        *  @return Layouter layout the child nodes.
        */
        Layouter* GetLayouter() const;   

        /**
         *  Sets a layout strategy for the child nodes, e.g. a grid or stack layout.
         *  The attached layouter is Disposed when the node is disposed.
         *  @param layouter Layout strategy for child nodes.
         */
        void SetLayouter(Layouter* layouter) const;

        /**
        * Gets the pivot offset of this object.
        * In 3D there is no actual pivot offset. By default a 3D node returns 0,0.
        * A canvas node returns 0, 0.
        * @return The pivot offset
        */
        CANDERA_DEPRECATED_3_6_0("The pivot offset has been replaced with the pivot point, which represents an actual point of reference for transformations instead of an offset. " \
            "Therefore, the respective pivot point methods should be used.",
        Vector2 GetPivotOffset() const);

        /**
        * Sets the pivot offset of this object.
        * In 3D there is no actual pivot offset. Therefore, this method does nothing for 3D nodes.
        * @param pivotOffset The pivot offset
        */
        CANDERA_DEPRECATED_3_6_0("The pivot offset has been replaced with the pivot point, which represents an actual point of reference for transformations instead of an offset. " \
            "Therefore, the respective pivot point methods should be used.",
        void SetPivotOffset(const Vector2& pivotOffset) const);
#endif

    private:
#if defined(CANDERA_LAYOUT_ENABLED)
        friend class Layouter;

        Vector2 GetPreferredSize() const;
        void SetPreferredSize(const Vector2& preferredSize) const;
#endif

        AbstractNodePointer(Type type, CanderaObject* node);

        Type m_type; //< Type of the node which prevents the class from RTTI (or similar) comparisons
        CanderaObject* m_node; ///< Node which is abstracted by this class
    };
    /** @} */



    inline bool AbstractNodePointer::IsValid() const {
        return 0 != m_node;
    }

    inline AbstractNodePointer::Type AbstractNodePointer::GetType() const
    {
        return m_type;
    }

    inline CanderaObject* AbstractNodePointer::ToCanderaObject() const
    {
        return m_node;
    }

    inline bool AbstractNodePointer::operator==(const CanderaObject* node) const
    {
        return (m_node == node);
    }

    inline bool AbstractNodePointer::operator!=(const CanderaObject* node) const
    {
        return !operator==(node);
    }

    inline bool AbstractNodePointer::operator==(const AbstractNodePointer& node) const
    {
        return (m_type == node.m_type) && (m_node == node.m_node);
    }

    inline bool AbstractNodePointer::operator!=(const AbstractNodePointer& node) const
    {
        return !operator==(node);
    }


#ifdef CANDERA_2D_ENABLED

    inline Node2D* AbstractNodePointer::ToNode2D() const
    {
        return (m_type == Candera2D) ? static_cast<Node2D*>(m_node) : 0;
    }

    inline bool AbstractNodePointer::operator==(const Node2D* node) const
    {
        return (m_type == Candera2D) && (m_node == node);
    }

    inline bool AbstractNodePointer::operator!=(const Node2D* node) const
    {
        return !operator==(node);
    }
#endif

#ifdef CANDERA_3D_ENABLED

    inline Node* AbstractNodePointer::ToNode() const
    {
        return (m_type == Candera3D) ? static_cast<Node*>(m_node) : 0;
    }

    inline bool AbstractNodePointer::operator==(const Node* node) const
    {
        return (m_type == Candera3D) && (m_node == node);
    }

    inline bool AbstractNodePointer::operator!=(const Node* node) const
    {
        return !operator==(node);
    }

#endif
}

#endif //CANDERA_ENGINEBASE_ABSTRACT_NODE_POINTER_H
