//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include "AbstractNodePointer.h"

#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Scene2D.h>
#endif

#ifdef CANDERA_3D_ENABLED
#include <Candera/Engine3D/Core/Scene.h>
#endif

#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/DefaultLayouter.h>
#endif

#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngineBase);

    AbstractNodePointer::AbstractNodePointer() :
        m_type(Undefined),
        m_node(0)
    {
    }

    AbstractNodePointer::AbstractNodePointer(const AbstractNodePointer& node) :
        m_type(node.m_type),
        m_node(node.m_node)
    {
    }

    const Char* AbstractNodePointer::GetName() const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return ToNode2D()->GetName();
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return ToNode()->GetName();
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return "";
    }

    bool AbstractNodePointer::AddChild(const AbstractNodePointer& child) const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return ToNode2D()->AddChild(child.ToNode2D());
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return ToNode()->AddChild(child.ToNode());
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return false;
    }

    bool AbstractNodePointer::InsertBeforeChild(const AbstractNodePointer& newChild, const AbstractNodePointer& beforeChild) const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return ToNode2D()->InsertBeforeChild(newChild.ToNode2D(), beforeChild.ToNode2D());
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return ToNode()->InsertBeforeChild(newChild.ToNode(), beforeChild.ToNode());
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return false;
    }

    bool AbstractNodePointer::RemoveChild(const AbstractNodePointer& child) const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return ToNode2D()->RemoveChild(child.ToNode2D());
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return ToNode()->RemoveChild(child.ToNode());
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return false;
    }

    AbstractNodePointer AbstractNodePointer::GetParent() const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return AbstractNodePointer(ToNode2D()->GetParent());
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return AbstractNodePointer(ToNode()->GetParent());
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return AbstractNodePointer(m_type, 0);
    }

    AbstractNodePointer AbstractNodePointer::GetFirstChild() const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return AbstractNodePointer(ToNode2D()->GetFirstChild());
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return AbstractNodePointer(ToNode()->GetFirstChild());
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return AbstractNodePointer(m_type, 0);
    }

    AbstractNodePointer AbstractNodePointer::GetNextSibling() const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return AbstractNodePointer(ToNode2D()->GetNextSibling());
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return AbstractNodePointer(ToNode()->GetNextSibling());
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return AbstractNodePointer(m_type, 0);
    }


#ifdef CANDERA_2D_ENABLED
    AbstractNodePointer::AbstractNodePointer(Node2D* node) :
        m_type(Candera2D),
        m_node(node)
    {
    }
#endif

#ifdef CANDERA_3D_ENABLED
    AbstractNodePointer::AbstractNodePointer(Node* node) :
        m_type(Candera3D),
        m_node(node)
    {
    }
#endif

    AbstractNodePointer::AbstractNodePointer(Type type, CanderaObject* node) :
        m_type(type),
        m_node(node)
    {
    }
#ifdef CANDERA_3D_ENABLED
    static Vector2 Vector3ToVector2(const Vector3 v) {
        return Vector2(v.GetX(), v.GetY());
    }
#endif

    Vector2 AbstractNodePointer::GetPosition() const
    {
        Vector2 result;
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->GetPosition();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = Vector3ToVector2(ToNode()->GetPosition());
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    void AbstractNodePointer::SetPosition(const Vector2& position) const
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->SetPosition(position);
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                ToNode()->SetPosition(position.GetX(), position.GetY(), ToNode()->GetPosition().GetZ());
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

    Vector2 AbstractNodePointer::GetScale() const
    {
        Vector2 result = Vector2(1.0F, 1.0F);
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->GetScale();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = Vector3ToVector2(ToNode()->GetScale());
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    void AbstractNodePointer::SetScale(const Vector2& scale) const
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->SetScale(scale);
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                ToNode()->SetScale(scale.GetX(), scale.GetY(), ToNode()->GetScale().GetZ());
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

    Float AbstractNodePointer::GetRotation() const
    {
        Float result = 0.0F;
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->GetRotation();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = ToNode()->GetRotation().GetZ();
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    void AbstractNodePointer::SetRotation(Float rotation) const
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->SetRotation(rotation);
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                ToNode()->SetRotation(ToNode()->GetRotation().GetX(), ToNode()->GetRotation().GetY(), rotation);
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }



    Vector2 AbstractNodePointer::GetPivotPoint() const
    {

        Vector2 result(0.0F, 0.0F);
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->GetPivotPoint();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = Vector3ToVector2(ToNode()->GetPivotPoint());
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    FEATSTD_LINT_NONCONST_METHOD(Candera::AbstractNodePointer::SetPivotPoint, "method not made const in order to keep current interface")
    void AbstractNodePointer::SetPivotPoint(const Vector2& pivotPoint)
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->SetPivotPoint(pivotPoint);
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                ToNode()->SetPivotPoint(pivotPoint.GetX(), pivotPoint.GetY(), ToNode()->GetPivotPoint().GetZ());
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

    AbstractNodePointer AbstractNodePointer::GetScene() const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                return AbstractNodePointer(static_cast<Node2D*>(ToNode2D()->GetScene()));
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                return AbstractNodePointer(static_cast<Node*>(ToNode()->GetScene()));
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return AbstractNodePointer(m_type, 0);
    }

    void AbstractNodePointer::GetBasePoint(Vector2& basePoint) const
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, CANDERA_LINT_REASON_RETURNINCASE)
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->GetBasePoint(basePoint);
                return;
#endif
#ifdef CANDERA_3D_ENABLED
#if defined(CANDERA_LAYOUT_ENABLED)
            case Candera3D: {
                    Vector3 basePoint3;
                    ToNode()->GetBasePoint(basePoint3);
                    basePoint.SetX(basePoint3.GetX());
                    basePoint.SetY(basePoint3.GetY());
                }
                return;
#endif
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        basePoint.SetX(0.0F);
        basePoint.SetY(0.0F);
    }

    bool AbstractNodePointer::IsEffectiveRenderingEnabled() const
    {
        bool result = false;
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->IsEffectiveRenderingEnabled();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = ToNode()->IsEffectiveRenderingEnabled();
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    bool AbstractNodePointer::IsRenderingEnabled() const
    {
        bool result = false;
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->IsRenderingEnabled();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = ToNode()->IsRenderingEnabled();
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    void AbstractNodePointer::SetRenderingEnabled(bool enabled) const
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->SetRenderingEnabled(enabled);
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                ToNode()->SetRenderingEnabled(enabled);
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

#if defined(CANDERA_LAYOUT_ENABLED)
    void AbstractNodePointer::InvalidateLayout() const
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D: {
                    const Node2D* current = ToNode2D();
                    const Node2D* layoutNode = ToNode2D();
                    while (0 != current) {
                        layoutNode = current;
                        if (Layouter::IsCached(*current)) {
                            Layouter::SetLayoutValid(const_cast<Node2D&>(*current), false);
                        }
                        current = current->GetParent();
                    }
                    Scene2D* scene = 0;
                    if (0 != layoutNode) {
                        scene = layoutNode->GetScene();
                        if (0 != scene) {
                            scene->SetLayoutInvalid();
                        }
                    }
                }
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D: {
                    const Node* current = ToNode();
                    const Node* layoutNode = ToNode();
                    while (0 != current) {
                        layoutNode = current;
                        if (Layouter::IsCached(*current)) {
                            Layouter::SetLayoutValid(const_cast<Node&>(*current), false);
                        }
                        current = current->GetParent();
                    }
                    Scene* scene = 0;
                    if (0 != layoutNode) {
                        scene = layoutNode->GetScene();
                        if (0 != scene) {
                            scene->SetLayoutInvalid();
                        }
                    }
                }
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

    Layouter* AbstractNodePointer::GetLayouter() const
    {
        Layouter* result = DefaultLayouter::GetInstance();
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->GetLayouter();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = ToNode()->GetLayouter();
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    void AbstractNodePointer::SetLayouter(Layouter* layouter) const
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->SetLayouter(layouter);
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                ToNode()->SetLayouter(layouter);
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

    Vector2 AbstractNodePointer::GetPivotOffset() const
    {
        Vector2 result;
        Vector3 result3D;
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
                result = ToNode2D()->GetPivotOffset();
                FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result3D = ToNode()->GetLayoutStartPosition();
                result.SetX(result3D.GetX());
                result.SetY(result3D.GetY());
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    void AbstractNodePointer::SetPivotOffset(const Vector2& pivotOffset) const
    {
#ifndef CANDERA_2D_ENABLED
        FEATSTD_UNUSED(pivotOffset);
#endif
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
                ToNode2D()->SetPivotOffset(pivotOffset);
                FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

    Vector2 AbstractNodePointer::GetPreferredSize() const
    {
        Vector2 result;
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                result = ToNode2D()->GetPreferredSize();
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                result = ToNode()->GetPreferredSize();
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
        return result;
    }

    void AbstractNodePointer::SetPreferredSize(const Vector2& preferredSize) const
    {
        if (0 != m_node) {
            switch (m_type) {
#ifdef CANDERA_2D_ENABLED
            case Candera2D:
                ToNode2D()->SetPreferredSize(preferredSize);
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Candera3D:
                ToNode()->SetPreferredSize(preferredSize);
                break;
#endif
            case Undefined:
                break;

            default:
                break;
            }
        }
    }

#endif
}
