//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MAPPED_SHARED_OBJECT_CLONE_STRATEGY_H)
#define CANDERA_MAPPED_SHARED_OBJECT_CLONE_STRATEGY_H

#include <Candera/EngineBase/Cloning/SharedObjectCloneStrategy.h>
#include <Candera/System/Container/Map.h>

namespace Candera {

/** @addtogroup CloningBase
 *  @{
 */

/**
 * @brief This class keeps a map of shared objects to resolve sharing.
 *
 * The lifetime of this object should be guaranteed as long as sharing is
 * required. Destroying or clearing the map will prevent sharing with previous
 * objects.
 * @tparam Type The type of the shared object.
 * @tparam Node Unused. Required by SharedObjectCloneStrategy.
 */
template<typename Type, typename Node>
class MappedSharedObjectCloneStrategy : public SharedObjectCloneStrategy<Type, Node> 
{
public:
    typedef typename SharedObjectCloneStrategy<Type, Node>::Result Result;
    typedef typename SharedObjectCloneStrategy<Type, Node>::SharedPointer SharedPointer;

    /**
     * Implements method from SharedObjectCloneStrategy.
     * @param source Object for which the clone is provided.
     * @param root Unused.
     * @return A Result object. isNew is true if the source does not already
     *      appear in the map.
     */
    virtual Result Clone(const SharedPointer& source, 
        const TreeClonePair<Node>& /*root*/) override
    {
        Result result;
        if (source == 0) {
            return result;
        }

        SharedPointer destination;
        SharedPointer* foundItem = m_map.Find(source);
        if (foundItem != 0) {
            destination = *foundItem;
        }
        else {
            destination = source->Clone();
            if ((destination != 0) && m_map.Insert(source, destination)) {
                result.isNew = true;
            }
        }
        result.sharedObject = destination;

        return result;
    }

    /**
     * Clear the contents of the map.
     */
    void Clear()
    {
        m_map.Clear();
    }

private:
    struct SharedPointerComparator
    {
        static Int32 Compare(const SharedPointer& left, const SharedPointer& right) 
        {
            return Candera::Internal::MapLess<Candera::Handle>::Compare(
                reinterpret_cast<Candera::Handle>(left.GetPointerToSharedInstance()),
                reinterpret_cast<Candera::Handle>(right.GetPointerToSharedInstance()));
        }
    };
    Internal::Map<SharedPointer, SharedPointer, SharedPointerComparator> m_map;
};

/** @} */ // end of CloningBase

} // namespace Candera


#endif  // CANDERA_SHARED_OBJECT_CLONE_STRATEGY_H
