//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_KeyframeSequence_H)
#define CANDERA_KeyframeSequence_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Animation/AnimationTimeType.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/MemoryManagement/Disposer.h>

namespace Candera { namespace Animation {
/** @addtogroup AnimationBase
 *  @{
 */

/**
 * @brief Stores a collection of Keyframes.
 *
 * A Keyframe is defined by a sequence time (usually in milliseconds) and an attached array of values.
 */
class KeyframeSequence {
public:
    typedef MemoryManagement::ArrayDisposer<const SequenceTimeType*> TimeTypeDisposer;
    typedef TimeTypeDisposer::DisposerFunction TimeTypeDisposerFn;

    typedef MemoryManagement::ArrayDisposer<const Float*> ValuesDisposer;
    typedef ValuesDisposer::DisposerFunction ValuesDisposerFn;

    FEATSTD_TYPEDEF_SHARED_POINTER(KeyframeSequence);

    struct Keyframe
    {
        SequenceTimeType m_sequencetimeMs;
        const Float* m_valuesBegin;
    };

    /**
     * Named constructor.
     * @return MemoryManagement::SharedPointer to the created KeyframeSequence object.
     */
    FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

    /**
     * Virtual destructor.
     */
    virtual ~KeyframeSequence();

    /**
     * Set keyframes.
     *
     * Keyframes are given by an array of sequence times and an array of values. The KeyframeSequence takes ownership the two arrays
     *  and will dispose them using the supplied disposers, when new keyframes are set or at destruction time.
     *
     * @param numberOfComponents                Number of values that are assigned to each keyframe.
     * @param keyframeCount                     Number of keyframes that compose the sequcene.
     * @param keyframeSequenceTimesArrayBegin   Array of sequence times. Each sequence time is attached to one keyframe.
     *                                          Size of array must be equal with [keyframeCount].
     * @param keyframeSequenceTimesDisposerFn   Disposer function for the sequence times array.
     * @param keyframeValuesArrayBegin          Array of values. Each consecutive group of [numberOfComponents] values
     *                                          are attached to one keyframe. Size of array must be equal with
     *                                          [numberOfComponents] x [keyframeCount].
     * @param keyframeValuesDisposerFn          Disposer function for the values array.
     */
    void SetKeyframes(Int32 numberOfComponents, Int32 keyframeCount,
                      const SequenceTimeType* keyframeSequenceTimesArrayBegin, TimeTypeDisposerFn keyframeSequenceTimesDisposerFn,
                      Float* keyframeValuesArrayBegin, ValuesDisposerFn keyframeValuesDisposerFn);

    /**
     * Set values attached to one keyframe.
     *
     * @param index     Index of the keyframe in sequence, for which the values are set.
     * @param values    Array of values to be set. Size of array must be equal with [numberOfComponents].
     *
     * @return true if successful, false otherwise.
     */
    bool SetKeyframeValues(Int index, const Float* values);

    /**
     * Get number of values attached to each keyframe.
     * @return Number of values attached to each keyframe.
     */
    Int32 GetNumberOfComponents() const { return m_numberOfComponents; }

    /**
     * Get number of keyframes in sequence.
     * @return Number of keyframes in sequence.
     */
    Int32 GetKeyframeCount() const { return m_keyframeCount; }

    /**
     * Get the sequence times array.
     * @return Pointer to the sequence times array.
     */
    const SequenceTimeType* GetKeyframeSequenceTimes() const { return m_keyframeSequenceTimes; }

    /**
     * Get the values array.
     * @return Pointer to keyframe values array.
     */
    Float* GetKeyframeValues() const { 
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1763, Candera::Animation::KeyframeSequence::GetKeyframeValues, CANDERA_LINT_REASON_NONCONST)
    return m_keyframeValues; 
    }

    /**
     * Get a keyframe.
     *
     * @param index     Index of the keyframe in sequence.
     *
     * @return a Keyframe structure containing the sequence time and associated values.
     */
    Keyframe GetKeyframe(Int index) const;

    /**
     * Retrieve the first two consecutive keyframe indexes, for which a given sequence time is placed between the two keyframe
     *  sequence times. Useful for sorted keyframe sequences.
     *
     * @param sequenceTimeMs    The sequence time for which the surrounding keyframes are retrieved.
     * @param indexLeft [out]   The index of the keyframe with a sequence time less or equal with the provided sequence time.
     * @param indexRight [out]  The index of the keyframe with a sequence time greater or equal with the provided sequence time.
     */
    void Find2FramesAroundSequenceTime(SequenceTimeType sequenceTimeMs, Int32& indexLeft, Int32& indexRight);

    /**
    * This method fulfills the interface that is necessary for the binary search algorithm which is used to find two frames around a sequence time.
    */
    static Int Compare(SequenceTimeType key, Int32 index, const SequenceTimeType* collection);

protected:
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Animation::KeyframeSequence::KeyframeSequence, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
    KeyframeSequence();

private:

    /**
    * Sets indexLeft and indexRight depending on i.
    */
    void Set2FrameIndicesAroundSequenceTime(Int32 i, Int32& indexLeft, Int32& indexRight);

    Int32 m_keyframeCount;
    Int32 m_numberOfComponents;
    Float* m_keyframeValues;
    const SequenceTimeType* m_keyframeSequenceTimes;
    TimeTypeDisposerFn m_keyframeSequenceTimesDisposerFn;
    ValuesDisposerFn m_keyframeValuesDisposerFn;
    Int32 m_lastKeyFrameIndex;
    SequenceTimeType m_lastSequenceTimeMs;
    static const Int32 c_linearSearchWidth = 6;

    // Make this class manageable by MemoryManagement::SharedPointer
    CANDERA_SHARED_POINTER_DECLARATION();
};

 /** @} */ // end of AnimationBase
    } // namespace Animation
} // namespace Candera
#endif
