//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "KeyframeSequence.h"

#include <FeatStd/Util/BinarySearch.h>

namespace Candera { namespace Animation {
//    using namespace MemoryManagement;
    using MemoryManagement::SharedPointer;

KeyframeSequence::KeyframeSequence() :
    m_keyframeCount(0),
    m_numberOfComponents(0),
    m_keyframeValues(0),
    m_keyframeSequenceTimes(0),
    m_keyframeSequenceTimesDisposerFn(0),
    m_keyframeValuesDisposerFn(0),
    m_lastKeyFrameIndex(0),
    m_lastSequenceTimeMs(0)
{
}

KeyframeSequence::SharedPointer KeyframeSequence::Create()
{
    return KeyframeSequence::SharedPointer(FEATSTD_NEW(KeyframeSequence)());
}

KeyframeSequence::~KeyframeSequence()
{
    if (m_keyframeSequenceTimesDisposerFn != 0) {
        m_keyframeSequenceTimesDisposerFn(m_keyframeSequenceTimes);
    }
    if (m_keyframeValuesDisposerFn != 0) {
        m_keyframeValuesDisposerFn(m_keyframeValues);
    }

    m_keyframeSequenceTimes = 0;
    m_keyframeSequenceTimesDisposerFn = 0;
    m_keyframeValues = 0;
    m_keyframeValuesDisposerFn = 0;
}

void KeyframeSequence::SetKeyframes(
    Int32 numberOfComponents,
    Int32 keyframeCount,
    const SequenceTimeType* keyframeSequenceTimesArrayBegin,
    TimeTypeDisposerFn keyframeSequenceTimesDisposerFn,
    Float* keyframeValuesArrayBegin,
    ValuesDisposerFn keyframeValuesDisposerFn
)
{
    if ((m_keyframeSequenceTimesDisposerFn != 0) && (keyframeSequenceTimesArrayBegin != m_keyframeSequenceTimes)) {
        m_keyframeSequenceTimesDisposerFn(m_keyframeSequenceTimes);
    }
    if ((m_keyframeValuesDisposerFn != 0) && (keyframeValuesArrayBegin != m_keyframeValues)) {
        m_keyframeValuesDisposerFn(m_keyframeValues);
    }
    m_numberOfComponents = numberOfComponents;
    m_keyframeCount = keyframeCount;
    m_keyframeSequenceTimes = keyframeSequenceTimesArrayBegin;
    m_keyframeSequenceTimesDisposerFn = keyframeSequenceTimesDisposerFn;
    m_keyframeValues = keyframeValuesArrayBegin;
    m_keyframeValuesDisposerFn = keyframeValuesDisposerFn;
}

bool KeyframeSequence::SetKeyframeValues(Int index, const Float* values)
{
    if ((m_keyframeValues == 0) || (values == 0) || (index >= m_keyframeCount)) {
        return false;
    }

    MemoryPlatform::Copy(m_keyframeValues + (index * m_numberOfComponents), values, static_cast<SizeType>(m_numberOfComponents) * sizeof(Float));
    return true;
}

KeyframeSequence::Keyframe KeyframeSequence::GetKeyframe(Int index) const
{
    Keyframe k;
    if ((m_keyframeSequenceTimes == 0) || (m_keyframeValues == 0)) {
        k.m_sequencetimeMs = 0;
        k.m_valuesBegin = 0;
        return k;
    }
    k.m_sequencetimeMs = m_keyframeSequenceTimes[index];
    k.m_valuesBegin = m_keyframeValues + (index * m_numberOfComponents);
    return k;
}

void KeyframeSequence::Find2FramesAroundSequenceTime(SequenceTimeType sequenceTimeMs, Int32& indexLeft, Int32& indexRight)
{
    indexLeft = 0;
    indexRight = 1;

    bool keyframeInLinearSearchFound = false;
    bool playingReversed = sequenceTimeMs < m_lastSequenceTimeMs;

    // First, search linearly in the next few keyframes for the right one (or a few ones before for reversed direction).

    if (playingReversed) {
        for (Int32 i = m_lastKeyFrameIndex; (i > (m_lastKeyFrameIndex - c_linearSearchWidth)) && (i >= 0); --i) {
            const Keyframe currentKeyframe = GetKeyframe(i);
            if (currentKeyframe.m_sequencetimeMs <= sequenceTimeMs) {
                Set2FrameIndicesAroundSequenceTime(i+1, indexLeft, indexRight);
                keyframeInLinearSearchFound = true;
                break;
            }
        }

    }
    else {
        for (Int32 i = m_lastKeyFrameIndex; (i < (m_lastKeyFrameIndex + c_linearSearchWidth)) && (i <= (m_keyframeCount - 1)); ++i) {
            const Keyframe currentKeyframe = GetKeyframe(i);
            if (currentKeyframe.m_sequencetimeMs >= sequenceTimeMs) {
                Set2FrameIndicesAroundSequenceTime(i, indexLeft, indexRight);
                keyframeInLinearSearchFound = true;
                break;
            }
        }
    }

    // No keyframe in the next few ones found: search for it in the remaining keyframes via binary search.

    if (false == keyframeInLinearSearchFound) {

        const SequenceTimeType* sequenceTimeStart = m_keyframeSequenceTimes;
        Int32 toSearchCount = m_keyframeCount;

        // Calculate where to start the search and how many items should be searched to only search in the remaining ones.
        if (!playingReversed) {
            sequenceTimeStart = &m_keyframeSequenceTimes[m_lastKeyFrameIndex];
            toSearchCount = m_keyframeCount - m_lastKeyFrameIndex;
        }
        else {
            toSearchCount = m_lastKeyFrameIndex;
        }

        // The actual binary search.
        Int32 i = FeatStd::Internal::BinarySearchIndexed<KeyframeSequence, SequenceTimeType, Int32>(sequenceTimeMs, sequenceTimeStart, toSearchCount);

        // i == toSearchCount if the element is not found; indexLeft/Right stays 0/1
        if (i == toSearchCount) {
            return;
        }

        /* 
        * i is still related to the sequenceTimeStart calculated above. Hence for non-reversed animations the value has to be increased by
        * the last frame-index stored.
        */
        if (!playingReversed) {
            i += m_lastKeyFrameIndex;
        }

        Set2FrameIndicesAroundSequenceTime(i, indexLeft, indexRight);
    }
    
    m_lastSequenceTimeMs = sequenceTimeMs;
}

void KeyframeSequence::Set2FrameIndicesAroundSequenceTime(Int32 i, Int32& indexLeft, Int32& indexRight)
{
    m_lastKeyFrameIndex = i;
    if (i == 0) {
        indexLeft = 0;
        indexRight = 1;
    }
    else {
        indexLeft = i - 1;
        indexRight = i;
    }
}

Int KeyframeSequence::Compare(SequenceTimeType key, Int32 index, const SequenceTimeType* collection)
{
    if (index <= 0) {
        return 0;
    }

    SequenceTimeType sequenceTimeLeft = collection[index - 1];
    SequenceTimeType sequenceTimeRight = collection[index];

    if ((key > sequenceTimeLeft) && (key <= sequenceTimeRight)) {
        return 0;
    }
    else if (key > sequenceTimeLeft) {
        return 1;
    }
    else {
        return -1;
    }
}


    } // namespace Animation
} // namespace Candera
