//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ExponentialEaseFunction.h"
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera { namespace Animation {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    FEATSTD_RTTI_BASECLASS_DEFINITION(ExponentialEaseFunction)

    static const Float c_e = 2.71828F;

    ExponentialEaseFunction::ExponentialEaseFunction() :
        Base(),
        m_exponent(1.0F),
        m_oneOverDivisor(1.0F / (c_e - 1.0F))
    {
    }

    ExponentialEaseFunction::~ExponentialEaseFunction()
    {
    }

    ExponentialEaseFunction::SharedPointer ExponentialEaseFunction::Create()
    {
        return ExponentialEaseFunction::SharedPointer(CANDERA_NEW(ExponentialEaseFunction));
    }

    Float ExponentialEaseFunction::Resolve(Float inValue) const
    {
        return (Math::Power(c_e, m_exponent * inValue) - 1.0F) * m_oneOverDivisor;
    }

    void ExponentialEaseFunction::SetExponent(Float exponent)
    {
        if (!Math::FloatAlmostEqual(exponent, 0.0F)) {
            m_exponent = exponent;
            m_oneOverDivisor = 1.0F / (Math::Power(c_e, m_exponent) - 1.0F);
        }
        else {
            FEATSTD_LOG_INFO("0 exponent. Value ignored.");
        }
    }

    Float ExponentialEaseFunction::GetExponent() const
    {
        return m_exponent;
    }

    } // namespace Animation
} // namespace Candera
