//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_EaseInterpolationStrategy_H)
#define CANDERA_EaseInterpolationStrategy_H

#include <Candera/EngineBase/Animation/InterpolationStrategy.h>
#include <Candera/EngineBase/Animation/AbstractEasingFunction.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera { namespace Animation {
/** @addtogroup AnimationBase
 *  @{
 */
    class AbstractEasingFunction;

    /**
     * @brief InterpolationStrategy implementation.
     *
     * This interpolation strategy allows the transition between two consecutive key frames
     *  of a sequence to be defined by a mathematical formula (ease function). For a smooth
     *  transition between key frames, the function must follow the rules:
     *  f:[0,1]->(-inf, inf), with f(0)=0 and f(1)=1.
     *
     * Each interpolation calculates the weight of the right and left key frames using the
     *  provided function. A direction can be defined to specify if the function will be
     *  used directly(EaseIn), reversed(EaseOut) or a combination of both(EaseInOut).
     */
    class EaseInterpolationStrategy: public InterpolationStrategy {

        FEATSTD_TYPEDEF_BASE(InterpolationStrategy);

        public:
            typedef MemoryManagement::Disposer<const AbstractEasingFunction*>::DisposerFunction DisposerFn;
            FEATSTD_TYPEDEF_SHARED_POINTER(EaseInterpolationStrategy);

            /**
             * Direction on which the ease function can be applied.
             */
            enum EaseDirection {
                EaseIn,     ///< Ease in only.
                EaseOut,    ///< Ease out only.
                EaseInOut   ///< Ease both in and out.
            };

            FEATSTD_RTTI_DECLARATION();

            /**
             * Creates a EaseInterpolationStrategy object.
             * @return Pointer to the created EaseInterpolationStrategy object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             * Destructs an EaseInterpolationStrategy object.
             */
            virtual ~EaseInterpolationStrategy() override;

            /**
             * Set EaseFunction.
             * @param easeFunction  Pointer to an AbstractEasingFunction object.
             */
            void SetEaseFunction(const AbstractEasingFunction::SharedPointer& easeFunction);

            /**
             * Get EaseFunction.
             * @return pointer to the AstractEasingFunction object.
             */
            AbstractEasingFunction::SharedPointer GetEaseFunction() const {
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1763, Candera::Animation::EaseInterpolationStrategy::GetEaseFunction, CANDERA_LINT_REASON_NONCONST)
                return m_easeFunction;
            }

            /**
             * Set EaseDirection.
             * @param easeDirection  Ease direction to be used in conjunction with the ease function.
             * @see EaseDirection.
             */
            void SetEaseDirection(EaseDirection easeDirection) { m_easeDirection = easeDirection; }

            /**
             * Get EaseDirection.
             * @return ease direction.
             * @see EaseDirection.
             */
            EaseDirection GetEaseDirection() const { return m_easeDirection; }

            /**
             * Calculate the interpolated value of a KeyframeSequence at a given sequence time.
             * @param keyframeSequence              The sequence of keyframes the interpolation strategy should be applied on.
             * @param sequenceTime                  The time at which the interpolated value should be calculated.
             * @param resultValue [out]             The resulted interpolated value.
             */
            virtual void Interpolate(
                const KeyframeSequence* keyframeSequence,
                SequenceTimeType sequenceTime,
                Float* resultValue) const override;

        protected:
            AbstractEasingFunction::SharedPointer m_easeFunction;
            EaseDirection m_easeDirection;

            EaseInterpolationStrategy();
    };

 /** @} */ // end of AnimationBase

    } // namespace Animation
} // namespace Candera

#endif
