//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BounceEaseFunction.h"
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera { namespace Animation {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    FEATSTD_RTTI_BASECLASS_DEFINITION(BounceEaseFunction)

    BounceEaseFunction::BounceEaseFunction() :
        Base(),
        m_bounceCount(0U),
        m_indexCache(0U),
        m_restitutionCoefficient(0.5F)

    {
        m_bounceTimes[0] = -1.0F;
        m_bounceTimes[1] = 1.0F;

        SetRestitutionCoefficient(0.5F);
    }

    BounceEaseFunction::~BounceEaseFunction()
    {
    }

    BounceEaseFunction::SharedPointer BounceEaseFunction::Create()
    {
        return BounceEaseFunction::SharedPointer(CANDERA_NEW(BounceEaseFunction));
    }

    void BounceEaseFunction::SetBounceCount(UInt8 bounceCount)
    {
        if (bounceCount < MaxBounceCount) {
            m_bounceCount = bounceCount;
        }
        else {
            FEATSTD_LOG_INFO("Bounce count exceeds maximum allowed value. Value ignored.");
        }
    }

    void BounceEaseFunction::SetRestitutionCoefficient(Float restitutionCoefficient)
    {
        if ((restitutionCoefficient >= 0.0F) && (restitutionCoefficient <= 1.0F)) {
            m_restitutionCoefficient = restitutionCoefficient;

            for (Int index = 2; index <= MaxBounceCount; index++) {
                m_bounceTimes[index] = m_bounceTimes[index - 1] + (2.0F * restitutionCoefficient);
                restitutionCoefficient *= m_restitutionCoefficient;
            }
        }
        else {
            FEATSTD_LOG_INFO("Restitution coefficient is outside of accepted limits (0.0F..1.0F). Value ignored.");
        }
    }

    Float BounceEaseFunction::Resolve(Float inValue) const
    {
        inValue = (1.0F - inValue) * m_bounceTimes[m_bounceCount + 1];

        while (inValue > m_bounceTimes[m_indexCache + 1]) {
            m_indexCache++;
        }

        while (inValue < m_bounceTimes[m_indexCache]) {
            m_indexCache--;
        }

        return (inValue - m_bounceTimes[m_indexCache]) * (m_bounceTimes[m_indexCache + 1] - inValue);
    }

    } // namespace Animation
} // namespace Candera
