//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationTimeDispatcher.h"
#include <Candera/EngineBase/Animation/AnimationTimeType.h>

#include <Candera/System/Monitor/MonitorPublicIF.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera { namespace Animation {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

AnimationTimeDispatcher::AnimationTimeDispatcher() :
    m_isPlayerMarkedAsRemoved(false),
    m_worldTimeMs(0U),
    m_players()
{
#ifdef FEATSTD_MONITOR_ENABLED
    static_cast<void>(MONITOR_REGISTER_ANIMATION_DISPATCHER(this));
#endif
}

AnimationTimeDispatcher::SharedPointer AnimationTimeDispatcher::Create()
{
    return AnimationTimeDispatcher::SharedPointer(FEATSTD_NEW(AnimationTimeDispatcher));
}

AnimationTimeDispatcher::~AnimationTimeDispatcher()
{
#ifdef FEATSTD_MONITOR_ENABLED
    static_cast<void>(MONITOR_DEREGISTER_ANIMATION_DISPATCHER(this));
#endif
}

void AnimationTimeDispatcher::Update(TimeType worldTimeInMilliseconds, TimeType deltaTimeInMilliseconds)
{
    FEATSTD_UNUSED(deltaTimeInMilliseconds);
    SetWorldTimeMs(worldTimeInMilliseconds);
    DispatchTime();
}

void AnimationTimeDispatcher::SetWorldTimeMs(WorldTimeType worldTimeMs) {
#ifdef FEATSTD_MONITOR_ENABLED
    static_cast<void>(MONITOR_UPDATE_WORLD_TIME(AnimationTimeDispatcher::SharedPointer(this), worldTimeMs));
#endif
    m_worldTimeMs = worldTimeMs;
}

void AnimationTimeDispatcher::DispatchTime()
{
    for (OffsetType i = static_cast<OffsetType>(m_players.Size()) - 1; i >= 0; --i) {
        if ((m_players[i]->IsEnabled()) && (m_players[i]->GetTimeDispatcher() == this)) {
            m_players[i]->ReceiveTime(m_worldTimeMs);
        }
    }
    // Only run through the player list if at least one player has been marked for removal.
    if (m_isPlayerMarkedAsRemoved) {
        RemoveMarkedPlayers();
        m_isPlayerMarkedAsRemoved = false;
    }
}

bool AnimationTimeDispatcher::AddPlayer(MemoryManagement::SharedPointer<AnimationPlayerBase> player)
{
    if (player.PointsToNull()) {
        FEATSTD_LOG_ERROR("Invalid player passed to AddPlayer.");
        return false;
    }

    if ((player->GetTimeDispatcher() != 0) && (player->GetTimeDispatcher() != this)) {
        FEATSTD_LOG_ERROR("AddPlayer failed, player already attached to a time dispatcher.");
        return false;
    }

    //Check if player not already in the player list.
    bool found = false;
    for (UInt i = 0; (i < static_cast<UInt>(m_players.Size())) && (!found); ++i) {
        found = m_players[i] == player;
    }

    if ((!found) && (!m_players.Add(player))) {
        return false;
    }

    player->SetTimeDispatcher(this);

    return true;
}

bool AnimationTimeDispatcher::RemovePlayer(MemoryManagement::SharedPointer<AnimationPlayerBase> player)
{
    m_isPlayerMarkedAsRemoved = true;

    if (player->GetTimeDispatcher() == this) {
        player->SetTimeDispatcher(0);
    }

    return true;
}

UInt32 AnimationTimeDispatcher::GetPlayerCount() const
{
    return static_cast<UInt32>(m_players.Size());
}

MemoryManagement::SharedPointer<AnimationPlayerBase> AnimationTimeDispatcher::GetPlayer(UInt32 index)
{
    return (index < GetPlayerCount()) ? m_players[index] : MemoryManagement::SharedPointer<AnimationPlayerBase>(0);
}

void AnimationTimeDispatcher::RemoveMarkedPlayers()
{
    for (OffsetType i = static_cast<OffsetType>(m_players.Size()) - 1; i >= 0; --i) {
        if (m_players[i]->GetTimeDispatcher() != this) {
            if (!m_players.Remove(i)) {
                FEATSTD_LOG_ERROR("RemoveMarkedPlayers failed, player cannot be removed from dispatcher.");
            }
        }
    }
}

    } // namespace Animation
} // namespace Candera
