//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationPlayerBase.h"
#include <Candera/EngineBase/Animation/AnimationGroupPlayer.h>
#include <Candera/EngineBase/Animation/AnimationTimeDispatcher.h>
#include <Candera/EngineBase/Animation/AnimationPlayerListener.h>
#include <Candera/System/Mathematics/Math.h>

namespace Candera {
    namespace Animation {
        using namespace MemoryManagement;

        FEATSTD_RTTI_DEFINITION(AnimationPlayerBase, CanderaObject)

            AnimationPlayerBase::AnimationPlayerBase() :
            Base(),
            m_timeDispatcher(0),
            m_isEnabled(false),
            m_isPaused(false),
            m_isFinishing(false),
            m_speedFactor(1.0F),
            m_pastTheEndCount(0),
            m_repeatCount(1),
            m_notificationDepth(0),
            m_pauseTime(0U),
            m_pauseDuration(0U),
            m_callbackUserData(0)
        {
                // No instructions.
            }

        AnimationPlayerBase::~AnimationPlayerBase()
        {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540, Candera::AnimationPlayerBase::m_timeDispatcher, CANDERA_LINT_REASON_ASSOCIATION)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540, Candera::AnimationPlayerBase::m_callbackUserData, CANDERA_LINT_REASON_ASSOCIATION)
        }

        bool AnimationPlayerBase::Start()
        {
            if (m_timeDispatcher == 0) {
                return false;
            }

            if (IsEnabled() && ((m_group != 0) || (!Stop()))) {
                return false;
            }

            m_isEnabled = true;
            m_isPaused = false;
            m_isFinishing = false;
            m_pastTheEndCount = 0;

            NotifyListenersOnStart();

            return true;
        }

        bool AnimationPlayerBase::Stop()
        {
            if (m_isEnabled) {
                m_isEnabled = false;
                m_isPaused = false;
                m_group = AnimationGroupPlayer::SharedPointer(0);
                NotifyListenersOnStop();

                return true;
            }

            return false;
        }

        bool AnimationPlayerBase::Finish(Animation::WorldTimeType /*timeToFinishMs*/)
        {
            if (m_timeDispatcher == 0) {
                return false;
            }

            m_isFinishing = true;

            NotifyListenersOnFinish();

            return true;
        }

        bool AnimationPlayerBase::Pause()
        {
            if (m_timeDispatcher == 0) {
                return false;
            }

            if (!IsEnabled()) {
                return false;
            }

            m_pauseTime = m_timeDispatcher->GetWorldTimeMs();
            m_isPaused = true;

            NotifyListenersOnPause();

            return true;
        }

        bool AnimationPlayerBase::Resume()
        {
            if (m_timeDispatcher == 0) {
                return false;
            }

            if ((!m_isEnabled) || (!m_isPaused)) {
                return false;
            }

            m_pauseDuration = m_timeDispatcher->GetWorldTimeMs() - m_pauseTime;
            m_isPaused = false;

            NotifyListenersOnResume();

            return true;
        }

        void AnimationPlayerBase::SetSpeedFactor(Float speedFactor)
        {
            if (speedFactor != 0.0F) {
                m_speedFactor = speedFactor;
            }
        }

        bool AnimationPlayerBase::StartInGroup(const AnimationPlayerBase::SharedPointer& animation)
        {
            if (animation->IsEnabled()) {
                return false;
            }

            animation->m_group = AnimationGroupPlayer::SharedPointer(Dynamic_Cast<AnimationGroupPlayer*>(this));
            if (!animation->Start()) {
                return false;
            }

            return true;
        }

        void AnimationPlayerBase::SendNotifications(Int playCount)
        {
            Int pastTheEndCount = Math::Minimum(m_pastTheEndCount, m_repeatCount);
            for (Int i = (m_pastTheEndCount - playCount) + 1; i <= pastTheEndCount; ++i) {
                NotifyListenersOnPastEnd(i);
            }

            if (!m_isEnabled) {
                m_isFinishing = false;
                if (!m_group.PointsToNull()) {
                    AnimationGroupPlayer::SharedPointer group(m_group);
                    m_group.Release();
                    group->OnPlayerFinished(AnimationPlayerBase::SharedPointer(this));
                }
                m_pastTheEndCount = 0;
            }
        }

        void AnimationPlayerBase::NotifyListenersOnPastEnd(Int completedIterationsCount)
        {
            NotificationLoopBegin();
            for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                AnimationPlayerListener* animationPlayerListener = *it;
                if (0 != animationPlayerListener) {
                    animationPlayerListener->OnPastEnd(this, completedIterationsCount);
                }
            }
            NotificationLoopEnd();
        }


        void AnimationPlayerBase::NotifyListenersOnStart()
        {
            NotificationLoopBegin();
            for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                AnimationPlayerListener* animationPlayerListener = *it;
                if (0 != animationPlayerListener) {
                    animationPlayerListener->OnStartAnimation(this);
                }
            }
            NotificationLoopEnd();
        }

        void AnimationPlayerBase::NotifyListenersOnStop()
        {
            NotificationLoopBegin();
            for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                AnimationPlayerListener* animationPlayerListener = *it;
                if (0 != animationPlayerListener) {
                    animationPlayerListener->OnStopAnimation(this);
                }
            }
            NotificationLoopEnd();
        }

        void AnimationPlayerBase::NotifyListenersOnFinish()
        {
            NotificationLoopBegin();
            for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                AnimationPlayerListener* animationPlayerListener = *it;
                if (0 != animationPlayerListener) {
                    animationPlayerListener->OnFinishAnimation(this);
                }
            }
            NotificationLoopEnd();
        }

        void AnimationPlayerBase::NotifyListenersOnResume()
        {
            NotificationLoopBegin();
            for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                AnimationPlayerListener* animationPlayerListener = *it;
                if (0 != animationPlayerListener) {
                    animationPlayerListener->OnResumeAnimation(this);
                }
            }
            NotificationLoopEnd();
        }

        void AnimationPlayerBase::NotifyListenersOnDirectionChanged(Int direction)
        {
            NotificationLoopBegin();
            for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                AnimationPlayerListener* animationPlayerListener = *it;
                if (0 != animationPlayerListener) {
                    animationPlayerListener->OnDirectionChange(this, direction);
                }
            }
            NotificationLoopEnd();
        }

        void AnimationPlayerBase::NotifyListenersOnPause()
        {
            NotificationLoopBegin();
            for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                AnimationPlayerListener* animationPlayerListener = *it;
                if (0 != animationPlayerListener) {
                    animationPlayerListener->OnPauseAnimation(this);
                }
            }
            NotificationLoopEnd();
        }

        Float AnimationPlayerBase::GetCompositeSpeedFactor() const
        {
            Float result = m_speedFactor;
            if (m_group != 0) {
                result *= m_group->GetCompositeSpeedFactor();
            }

            return result;
        }

        bool AnimationPlayerBase::UpdateIterationsCount(Int additionalIteratonsCount)
        {
            m_pastTheEndCount += additionalIteratonsCount;

            if (((m_repeatCount > 0) && (m_pastTheEndCount >= m_repeatCount)) || (m_isFinishing && (additionalIteratonsCount != 0))) {
                m_isEnabled = false;
            }

            return m_isEnabled;
        }

        bool AnimationPlayerBase::AddAnimationPlayerListener(Animation::AnimationPlayerListener* listener)
        {
            bool isSuccessful = false;
            if (listener != 0) {
                if (IsListenerRemovalAllowed()) {
                    isSuccessful = m_animationPlayerListeners.Append(listener);
                }
                else {
                    isSuccessful = m_pendingAddAnimationPlayerListeners.Append(listener);
                }
            }
            return isSuccessful;
        }

        bool AnimationPlayerBase::RemoveAnimationPlayerListener(Animation::AnimationPlayerListener* listener)
        {
            bool isSuccessful = false;
            if (listener != 0) {
                if (IsListenerRemovalAllowed()) {
                    isSuccessful = m_animationPlayerListeners.Remove(listener);
                }
                else {
                    for (AnimationPlayerListenerContainer::Iterator it = m_animationPlayerListeners.Begin(); it != m_animationPlayerListeners.End(); ++it) {
                        if (listener == *it) {
                            *it = 0;
                            isSuccessful = true;
                        }
                    }
                    static_cast<void>(m_pendingAddAnimationPlayerListeners.Remove(listener));
                }
            }
            return isSuccessful;
        }

        void AnimationPlayerBase::NotificationLoopEnd()
        {
            --m_notificationDepth;
            if (IsListenerRemovalAllowed()) {
                static_cast<void>(m_animationPlayerListeners.Remove(0, true)); // remove listeners which were removed during notification
                // add listeners which were added during notification
                for (AnimationPlayerListenerContainer::Iterator it = m_pendingAddAnimationPlayerListeners.Begin(); it != m_pendingAddAnimationPlayerListeners.End(); ++it) {
                    static_cast<void>(m_animationPlayerListeners.Append(*it));
                }
                m_pendingAddAnimationPlayerListeners.Clear();
            }
        }
    } //namespace Animation
} // namespace Candera
