//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationGroupPlayer.h"
#include <Candera/EngineBase/Animation/AnimationTimeDispatcher.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>

namespace Candera { namespace Animation {
    using namespace MemoryManagement;

    FEATSTD_RTTI_DEFINITION(AnimationGroupPlayer, AnimationPlayerBase)

AnimationGroupPlayer::AnimationGroupPlayer() :
    Base(),
    m_componentCount(0),
    m_finishedComponentCount(0),
    m_firstComponent(0),
    m_finishTime(0U)
{
}

AnimationGroupPlayer::SharedPointer AnimationGroupPlayer::Create()
{
    return AnimationGroupPlayer::SharedPointer(FEATSTD_NEW(AnimationGroupPlayer)());
}

AnimationGroupPlayer::~AnimationGroupPlayer()
{
    RemoveAllPlayers();
}

void AnimationGroupPlayer::ReceiveTime(WorldTimeType /*worldTimeMs*/)
{
    ProcessCommand(m_firstComponent, CCheckTime);
}

bool AnimationGroupPlayer::Start()
{
    if (!Base::Start()) {
        return false;
    }

    m_finishedComponentCount = 0;
    ProcessCommand(m_firstComponent, CPrepare);

    //Ensure that animated values of first key frame are set at the beginning of an animation.
    const_cast<AnimationTimeDispatcher*>(GetTimeDispatcher())->DispatchTime();

    return true;
}

bool AnimationGroupPlayer::Stop()
{
    if (!Base::Stop()) {
        return false;
    }

    ProcessCommand(m_firstComponent, CStop);
    return true;
}

bool AnimationGroupPlayer::Finish(WorldTimeType timeToFinishMs)
{
    if (!Base::Finish(timeToFinishMs)) {
        return false;
    }

    m_finishTime = GetTimeDispatcher()->GetWorldTimeMs() + timeToFinishMs;
    ProcessCommand(m_firstComponent, CFinish);
    return true;
}

bool AnimationGroupPlayer::Pause()
{
    if (!Base::Pause()) {
        return false;
    }

    ProcessCommand(m_firstComponent, CPause);
    return true;
}

bool AnimationGroupPlayer::Resume()
{
    if (!Base::Resume()) {
        return false;
    }

    ProcessCommand(m_firstComponent, CResume);
    return true;
}

void AnimationGroupPlayer::AddPlayer(const AnimationPlayerBase::SharedPointer& player, const AnimationPlayerBase::SharedPointer& predecessor, StartType startType, WorldTimeType delay)
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(593, playerComponent, CANDERA_LINT_REASON_ASSOCIATION)
    Component* playerComponent = 0;
    Component* predecessorComponent = 0;
    if (m_firstComponent != 0) {
        playerComponent = Find(player, true);
        predecessorComponent = Find(predecessor);;
    }

    if ((predecessorComponent == 0) && (predecessor != 0)) {
        AddPlayer(predecessor);
        predecessorComponent = Find(predecessor);;
    }

    if (playerComponent == 0) {
        playerComponent = FEATSTD_NEW(Component)(player);
        m_componentCount++;
    }

    if (playerComponent != 0) {
        playerComponent->m_delay = delay;

        if (predecessorComponent == 0) {
            playerComponent->m_listNode.SetNext(m_firstComponent);
            m_firstComponent = playerComponent;
        }
        else {
            if (startType == WithPrevious) {
                playerComponent->m_listNode.SetNext(predecessorComponent->m_successorsOnStart);
                predecessorComponent->m_successorsOnStart = playerComponent;
            }
            if (startType == AfterPrevious) {
                playerComponent->m_listNode.SetNext(predecessorComponent->m_successorsOnFinish);
                predecessorComponent->m_successorsOnFinish = playerComponent;
            }
            playerComponent->m_predecessor = predecessorComponent;
        }
    }
}

void AnimationGroupPlayer::RemovePlayer(const AnimationPlayerBase::SharedPointer& player, bool removeSuccessors)
{
    Component* component = Find(player, true);

    if (component == 0) {
        return;
    }

    if (!removeSuccessors) {
        Component* successor = component->m_successorsOnStart;
        if (successor != 0) {
            while (successor->m_listNode.GetNext() != 0) {
                successor->m_predecessor = 0;
                successor = successor->m_listNode.GetNext();
            }
            successor->m_listNode.SetNext(m_firstComponent);
            m_firstComponent = component->m_successorsOnStart;
            component->m_successorsOnStart = 0;
        }

        successor = component->m_successorsOnFinish;
        if (successor != 0) {
            while (successor->m_listNode.GetNext() != 0) {
                successor->m_predecessor = 0;
                successor = successor->m_listNode.GetNext();
            }
            successor->m_listNode.SetNext(m_firstComponent);
            m_firstComponent = component->m_successorsOnFinish;
            component->m_successorsOnFinish = 0;
        }
    }

    ProcessCommand(component, CDispose);
}

void AnimationGroupPlayer::RemoveAllPlayers()
{
    ProcessCommand(m_firstComponent, CDispose);
    m_firstComponent = 0;
    m_componentCount = 0;
}

AnimationGroupPlayer::Component* AnimationGroupPlayer::Component::Find(const AnimationPlayerBase::SharedPointer& player, bool removePlayer)
{
    Component* found = 0;
    if (player == m_player) {
        found = this;
    }

    Component* next = m_listNode.GetNext();
    if ((found == 0) && (next != 0)) {
        if (next->m_player == player) {
            found = next;
            if (removePlayer) {
                m_listNode.SetNext(next->m_listNode.GetNext());
                next->m_listNode.SetNext(0);
            }
        }
        else {
            found = m_listNode.GetNext()->Find(player, removePlayer);
        }
    }

    next = m_successorsOnStart;
    if ((found == 0) && (next != 0)) {
        if (next->m_player == player) {
            found = next;
            if (removePlayer) {
                m_successorsOnStart = next->m_listNode.GetNext();
                next->m_listNode.SetNext(0);
            }
        }
        else {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::AnimationGroupPlayer::Component::m_successorsOnStart, CANDERA_LINT_REASON_OPERANDNOTNULL)
            found = m_successorsOnStart->Find(player, removePlayer);
        }
    }

    next = m_successorsOnFinish;
    if ((found == 0) && (next != 0)) {
        if (next->m_player == player) {
            found = next;
            if (removePlayer) {
                m_successorsOnFinish = next->m_listNode.GetNext();
                next->m_listNode.SetNext(0);
            }
        }
        else {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::AnimationGroupPlayer::Component::m_successorsOnFinish, CANDERA_LINT_REASON_OPERANDNOTNULL)
            found = m_successorsOnFinish->Find(player, removePlayer);
        }
    }

    return found;
}

void AnimationGroupPlayer::Component::ProcessCommand(AnimationGroupPlayer* group, ComponentCommand command)
{
    ComponentCommand nextCommand = command;
    ComponentCommand startSuccessorCommand = command;
    ComponentCommand finishSuccessorCommand = command;

    switch (command) {
        case CPause: //pause player
            if (m_player->Pause()) {
                m_state = Paused;
            }
            break;
        case CResume: //resume paused player or update start time if the player didn't start yet
            if (m_state == Paused) {
                static_cast<void>(m_player->Resume());
                m_state = Running;
            }
            else {
                if (m_state == ReadyToStart) {
                    m_startWorldTime += group->GetPauseDuration();
                }
            }
            break;
        case CStop: //stop player
            static_cast<void>(m_player->Stop());
            m_state = Finished;
            break;
        case CFinish: //finish the player in half of the remaining finish time (so successors have time to finish too)
            if (m_state == ReadyToStart) {
                command = CStart;
            }
            else {
                static_cast<void>(m_player->Finish(
                    (group->m_finishTime > group->GetTimeDispatcher()->GetWorldTimeMs()) 
                    ? ((group->m_finishTime - group->GetTimeDispatcher()->GetWorldTimeMs()) / 2U) 
                    : 0U ));
            }
            break;
        case CCheckTime: //check players if their calculated start time is less then current time.
            if ((m_state == ReadyToStart) && (m_startWorldTime <= group->GetTimeDispatcher()->GetWorldTimeMs())) {
                command = CStart;
            }
            break;
        case CDispose:
            break;
        default: //all of the above commands should be propagated to the next in list or each successor player.
            // rest of the commands dispatched below will set their own commands for successors.
            nextCommand = CNone;
            startSuccessorCommand = CNone;
            finishSuccessorCommand = CNone;
    }

    //prepare the component to start by calculating its start time. If the delay is 0, then start the component
    if (command == CPrepare) {
        nextCommand = CPrepare;
        if ((m_delay == static_cast<WorldTimeType>(0)) || group->IsFinishing()) {
            command = CStart;
        }
        else {
            m_startWorldTime = group->GetTimeDispatcher()->GetWorldTimeMs() + (m_delay / static_cast<WorldTimeType>(group->GetCompositeSpeedFactor()));
            m_state = ReadyToStart;
        }
    }

    //start the component and prepare all of its successors that depend on its start time.
    if (command == CStart) {
        startSuccessorCommand = CPrepare;
        if (m_player != 0) {
            m_state = group->StartInGroup(m_player) ? Running : ReadyToStart;
            if (group->IsFinishing()) {
                static_cast<void>(m_player->Finish(
                    (group->m_finishTime > group->GetTimeDispatcher()->GetWorldTimeMs()) 
                    ? ((group->m_finishTime - group->GetTimeDispatcher()->GetWorldTimeMs()) / 2U) 
                    : 0U ));
            }
        }
        else {
            command = CEnd;
        }
    }

    //Finish the component and prepare all of its successors that depend on its finish time.
    //If the component was the last one, then replay the group (if repeat count not reached) or
    //send finish notification.
    if (command == CEnd) {
        finishSuccessorCommand = CPrepare;
        m_state = Finished;
        group->m_finishedComponentCount++;
        if (group->m_finishedComponentCount == group->m_componentCount) {
            if (group->UpdateIterationsCount(1)) {
                group->m_finishedComponentCount = 0;
                group->ProcessCommand(group->m_firstComponent, CPrepare);
            }
            group->SendNotifications(1);
            return;
        }
    }

    //Propagate command to the next in list component.
    if (m_listNode.GetNext() != 0) {
        m_listNode.GetNext()->ProcessCommand(group, nextCommand);
    }

    //Propagate command to the first successor on start component.
    if (m_successorsOnStart != 0) {
        m_successorsOnStart->ProcessCommand(group, startSuccessorCommand);
    }

    //Propagate command to the first successor on finish component.
    if (m_successorsOnFinish != 0) {
        m_successorsOnFinish->ProcessCommand(group, finishSuccessorCommand);
    }

    //Dispose the component.
    if (command == CDispose) {
        FEATSTD_DELETE(this);
    }
}

AnimationGroupPlayer::Component* AnimationGroupPlayer::Find(const AnimationPlayerBase::SharedPointer& player, bool removePlayer)
{
    Component* result = 0;

    if (m_firstComponent != 0) {
        if (m_firstComponent->m_player == player) {
            result = m_firstComponent;
            if (removePlayer) {
                m_firstComponent = m_firstComponent->m_listNode.GetNext();
            }
        }
        else {
            result = m_firstComponent->Find(player, removePlayer);
        }
    }

    if ((result != 0) && (removePlayer)) {
        m_componentCount--;
    }

    return result;
}

void AnimationGroupPlayer::ProcessCommand(Component* component, ComponentCommand command)
{
    if (component != 0) {
        component->ProcessCommand(this, command);
    }
}

void AnimationGroupPlayer::OnPlayerFinished(const AnimationPlayerBase::SharedPointer& player)
{
    ProcessCommand(Find(player), CEnd);
}

bool AnimationGroupPlayer::FindPlayer(const AnimationPlayerBase::SharedPointer& player, AnimationPlayerBase::SharedPointer& predecessor, StartType& startType, WorldTimeType& delay)
{
    Component* component = Find(player, false);

    if (component != 0) {
        if (component->m_predecessor != 0) {
            predecessor = component->m_predecessor->m_player;
            Component* iterator = component->m_predecessor->m_successorsOnStart;
            while ((iterator != 0) && (iterator->m_player != player)) {
                iterator = iterator->m_listNode.GetNext();
            }
            if (iterator != 0) {
                startType = WithPrevious;
            }
            else {
                startType = AfterPrevious;
            }
        }
        else {
            predecessor = AnimationPlayerBase::SharedPointer(0);
            startType = WithPrevious;
        }
        delay = component->m_delay;
        return true;
    }

    return false;
}

const AnimationPlayerBase::SharedPointer AnimationGroupPlayer::GetFirstSuccessor(const AnimationPlayerBase::SharedPointer& player, StartType startType)
{
    Component* component = Find(player, false);
    AnimationPlayerBase::SharedPointer null(0);

    if (component != 0) {
        if (startType == WithPrevious) {
            return (component->m_successorsOnStart != 0) ? component->m_successorsOnStart->m_player : null;
        }
        else {
            return (component->m_successorsOnFinish != 0) ? component->m_successorsOnFinish->m_player : null;
        }
    }

    if ((startType == WithPrevious) && (m_firstComponent != 0)) {
        return m_firstComponent->m_player;
    }

    return AnimationPlayerBase::SharedPointer(0);
}

const AnimationPlayerBase::SharedPointer AnimationGroupPlayer::GetNext(const AnimationPlayerBase::SharedPointer& player)
{
    Component* component = Find(player, false);

    if ((component != 0) && (component->m_listNode.GetNext() != 0)) {
        return component->m_listNode.GetNext()->m_player;
    }

    return AnimationPlayerBase::SharedPointer(0);
}

bool AnimationGroupPlayer::ReplacePlayer(const AnimationPlayerBase::SharedPointer& player, const AnimationPlayerBase::SharedPointer& newPlayer)
{
    Component* component = Find(player, false);

    if ((component != 0) && (component->m_player == player)) {
        component->m_player = newPlayer;
        return true;
    }

    return false;
}

    } // namespace Animation
} // namespace Candera
