//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_AnimationAggregatedProperty_H
#define CANDERA_AnimationAggregatedProperty_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Animation/AnimationTimeType.h>
#include <Candera/EngineBase/Animation/AnimationKeyframeSequence.h>
#include <Candera/EngineBase/Animation/AnimationPropertySetter.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera { namespace Animation {
/** @addtogroup AnimationBase
 *  @{
 */

/**
 *  @brief Associates several keyframe sequences with the property they affect
 *  (so the values in the keyframe sequence have a destination where they will be used).
 *  Aggregates the outputs (the interpolated values) produced by the keyframe sequences,
 *  and assigns the result to the property.
 */
class AnimationBlendedProperty {
    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(AnimationBlendedProperty);

        /**
         *  Creates a AnimationBlendedProperty object.
         *  @return MemoryManagement::SharedPointer to the created AnimationBlendedProperty object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        ~AnimationBlendedProperty();

        /**
         *  Adds a keyframe sequence.
         *  @param kfs The keyframe sequence that is added.
         *  @return true if sucessfull, false if not.
         */
        bool AddKeyframeSequence(const MemoryManagement::SharedPointer<AnimationKeyframeSequence>& kfs);

        /**
         *  Retrieves the number of  the keyframe sequence.
         *  @return The number of the keyframe sequence.
         */
        SizeType GetNumberOfKeyframeSequences() const;

        /**
         *  Retrieves a keyframe sequence at a specified index.
         *  @param index The index of the keyframe sequence.
         *  @return MemoryManagement::SharedPointer to the AnimationKeyframeSequence.
         */
        MemoryManagement::SharedPointer<AnimationKeyframeSequence> GetKeyframeSequence(SizeType index) const;

        /**
         *  Sets the animation property setter needed for animating the animated value.
         *  @param animationPropertySetter AnimationBlendedProperty takes ownership and will Dispose it when no longer needed
         */
        void SetAnimationPropertySetter(const AnimationPropertySetter::SharedPointer& animationPropertySetter);

        /**
         *  Retrieves the animation property setter needed for animating the animated value.
         *  @return The animation property setter.
         */
        AnimationPropertySetter::SharedPointer GetAnimationPropertySetter() const;

        /**
         *  Updates the animated property based on the given time
         *  @param sequenceTimeMs Animation time given.
         */
        void Animate(SequenceTimeType sequenceTimeMs);

    protected:
        /**
         *  Aggregates (combines) the outputs of the individual keyframe sequences into one value.
         *  The current aggregation strategy is unweighted sum; make configurable if more is needed.
         *  @param sequenceTimeMs Current animation time to be used for interpolating keyframe values.
         *  @param dest           Pointer to a pre-allocated array to write all aggregated outputs of each keyframe sequence into.
         */
        void Aggregate(SequenceTimeType sequenceTimeMs, Float* dest) const;

    private:
        Int32 m_numberOfComponents;
        typedef Internal::Vector<MemoryManagement::SharedPointer<AnimationKeyframeSequence> > KfsContainer;
        KfsContainer m_keyframeSequences;
        AnimationPropertySetter::SharedPointer m_propertySetter;

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::AnimationBlendedProperty::AnimationBlendedProperty, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        AnimationBlendedProperty();

        AnimationBlendedProperty(const AnimationBlendedProperty&);
        AnimationBlendedProperty& operator=(const AnimationBlendedProperty&);
};

 /** @} */ // end of AnimationBase
    }   // namespace Animation
} // namespace Candera

#endif
