//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationBlendedProperty.h"

namespace Candera { namespace Animation {

AnimationBlendedProperty::SharedPointer AnimationBlendedProperty::Create()
{
    return AnimationBlendedProperty::SharedPointer(FEATSTD_NEW(AnimationBlendedProperty));
}

bool AnimationBlendedProperty::AddKeyframeSequence(const MemoryManagement::SharedPointer<AnimationKeyframeSequence>& kfs)
{
    if (!m_keyframeSequences.Empty()) {
        if (kfs->GetNumberOfComponents() != m_keyframeSequences[0]->GetNumberOfComponents()) {
            //xxx trace
            return false;
        }
    }
    return m_keyframeSequences.Add(kfs);
}

SizeType AnimationBlendedProperty::GetNumberOfKeyframeSequences() const
{
    return m_keyframeSequences.Size();
}

MemoryManagement::SharedPointer<AnimationKeyframeSequence> AnimationBlendedProperty::GetKeyframeSequence(SizeType index) const
{
    return m_keyframeSequences[index];
}

void AnimationBlendedProperty::SetAnimationPropertySetter(const AnimationPropertySetter::SharedPointer& animationPropertySetter)
{
    m_propertySetter = animationPropertySetter;
}

AnimationPropertySetter::SharedPointer AnimationBlendedProperty::GetAnimationPropertySetter() const
{
    return m_propertySetter;
}

static const Int32 s_maxNumberOfComponents = 32;

void AnimationBlendedProperty::Animate(Candera::Animation::SequenceTimeType sequenceTimeMs)
{
    if (m_keyframeSequences.Size() == 0) {
        return;
    }

    m_numberOfComponents = m_keyframeSequences[0]->GetNumberOfComponents();

    Float aggregatedResult[s_maxNumberOfComponents];
    Aggregate(sequenceTimeMs, aggregatedResult);

    m_propertySetter->Set(aggregatedResult);
}

void AnimationBlendedProperty::Aggregate(Animation::SequenceTimeType sequenceTimeMs, Float* dest) const
{
    for (Int32 j = 0; j < m_numberOfComponents ; ++j) {
        dest[j] = 0.0F;
    }

    const SizeType numberOfKfs = m_keyframeSequences.Size();
    for (SizeType i = 0; i < numberOfKfs; ++i) {
        MemoryManagement::SharedPointer<AnimationKeyframeSequence> const keyframeSequence = m_keyframeSequences[i];
        Float interpolatedValues[s_maxNumberOfComponents];
        keyframeSequence->Interpolate(sequenceTimeMs, interpolatedValues);
        for (Int32 j = 0; j < m_numberOfComponents; ++j) {
            dest[j] += interpolatedValues[j];
        }
    }
}

AnimationBlendedProperty::AnimationBlendedProperty():
    m_numberOfComponents(0),
    m_propertySetter(0)
{
}

AnimationBlendedProperty::~AnimationBlendedProperty()
{
}

    }  // namespace Animation
} // namespace Candera
