//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GenericShaderParamSetter.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/MorphingMesh.h>
#include <Candera/Engine3D/Core/PointSprite.h>
#if defined (CANDERA_3D_CANVAS_ENABLED)
#include <Candera/Engine3D/Canvas/CanvasRenderable.h>
#endif
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/System/Mathematics/Matrix3.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>


#ifdef FEATSTD_LOG_ENABLED
// Logging Enabled

#define CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(uniformName) \
    { \
    FEATSTD_LOG_WARN("Uniform \"%s\" does not exist for current shader.", (uniformName == 0) ? "" : uniformName); \
    }

#else
// Release build.
#define CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(uniformName) FEATSTD_UNUSED(uniformName)

#endif


namespace Candera {
    using namespace Diagnostics;
    using namespace Candera::Internal;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    Shader::UniformCacheHandle GenericShaderParamSetter::s_uniformCacheHandle;

    enum UniformCacheIndices {
        ModelMatrix3 = 0,
        ModelMatrix4,
        NormalModelMatrix3,
        ModelViewMatrix3,
        ModelViewMatrix4,
        NormalModelViewMatrix3,
        ModelViewProjectionMatrix4,
        ProjectionMatrix4,
        CameraLookAtVector,
        CameraPosition,
        LightEnabled,
        LightEnabled1,
        LightEnabled2,
        LightEnabled3,
        LightEnabled4,
        LightEnabled5,
        LightEnabled6,
        LightEnabled7,
        PointSpriteSize,
        MaterialDiffuse,
        MorphWeightArray,
        CanvasPivot,
        CanvasSize,
        UniformCacheIndicesCount
    };

    static const ShaderParamNames::UniformSemantic UniformCacheParams[UniformCacheIndicesCount] = {
        ShaderParamNames::ModelMatrix3,
        ShaderParamNames::ModelMatrix4,
        ShaderParamNames::NormalModelMatrix3,
        ShaderParamNames::ModelViewMatrix3,
        ShaderParamNames::ModelViewMatrix4,
        ShaderParamNames::NormalModelViewMatrix3,
        ShaderParamNames::ModelViewProjectionMatrix4,
        ShaderParamNames::ProjectionMatrix4,
        ShaderParamNames::CameraLookAtVector,
        ShaderParamNames::CameraPosition,
        ShaderParamNames::LightEnabled,
        ShaderParamNames::LightEnabled1,
        ShaderParamNames::LightEnabled2,
        ShaderParamNames::LightEnabled3,
        ShaderParamNames::LightEnabled4,
        ShaderParamNames::LightEnabled5,
        ShaderParamNames::LightEnabled6,
        ShaderParamNames::LightEnabled7,
        ShaderParamNames::PointSpriteSize,
        ShaderParamNames::MaterialDiffuse,
        ShaderParamNames::MorphWeightArray,
        ShaderParamNames::CanvasPivot,
        ShaderParamNames::CanvasSize
    };

    const GenericShaderParamSetter::ActivateFn* GenericShaderParamSetter::GetActivationFunctionList()
    {
        static const ActivateFn activateFnList[] = {
            &GenericShaderParamSetter::ActivateModelMatrix3,
            &GenericShaderParamSetter::ActivateModelMatrix4,
            &GenericShaderParamSetter::ActivateNormalModelMatrix,
            &GenericShaderParamSetter::ActivateModelViewMatrix3,
            &GenericShaderParamSetter::ActivateModelViewMatrix4,
            &GenericShaderParamSetter::ActivateNormalModelViewMatrix,
            &GenericShaderParamSetter::ActivateModelViewProjectionMatrix,
            &GenericShaderParamSetter::ActivateProjectionMatrix,
            &GenericShaderParamSetter::ActivateCameraLookAtVector,
            &GenericShaderParamSetter::ActivateCameraPosition,
            &GenericShaderParamSetter::ActivateLights,
            &GenericShaderParamSetter::ActivateMaterial,
            &GenericShaderParamSetter::ActivateMorphingMesh,
            &GenericShaderParamSetter::ActivatePointSprite,
            &GenericShaderParamSetter::ActivateTextures,
            &GenericShaderParamSetter::ActivateCanvasRenderable
        };

        return activateFnList;
    }

    GenericShaderParamSetter::GenericShaderParamSetter():
        Base(),
        m_isModelMatrix4Enabled(false),
        m_isModelMatrix3Enabled(false),
        m_isNormalModelMatrix3Enabled(false),
        m_isModelViewMatrix4Enabled(false),
        m_isModelViewMatrix3Enabled(false),
        m_isNormalModelViewMatrix3Enabled(false),
        m_isModelViewProjectionMatrix4Enabled(true),
        m_isProjectionMatrix4Enabled(false),
        m_isCameraLookAtVectorEnabled(false),
        m_isCameraPositionEnabled(false),
        m_isLightsActivationEnabled(true),
        m_isMaterialActivationEnabled(true),
        m_isMorphingMeshActivationEnabled(false),
        m_isPointSpriteActivationEnabled(false),
        m_isTextureActivationEnabled(true),
        m_isCanvasActivationEnabled(false),
        m_lightCoordinateSpace(Light::Object)
    {
        const ActivateFn initialSequence[] = {
            &GenericShaderParamSetter::ActivateTextures,
            &GenericShaderParamSetter::ActivateMaterial,
            &GenericShaderParamSetter::ActivateLights,
            &GenericShaderParamSetter::ActivateModelViewProjectionMatrix
        };
        UInt32 size = sizeof(initialSequence) / sizeof(initialSequence[0]);
        if (m_autoUniforms.Reserve(size)) {
            while(size-- > 0) {
                // Container is large enough to fit.
                static_cast<void>(m_autoUniforms.Add(initialSequence[size]));
            }
        }
    }

    GenericShaderParamSetter::GenericShaderParamSetter(const GenericShaderParamSetter& rhs):
        Base(rhs),
        m_isModelMatrix4Enabled(rhs.m_isModelMatrix4Enabled),
        m_isModelMatrix3Enabled(rhs.m_isModelMatrix3Enabled),
        m_isNormalModelMatrix3Enabled(rhs.m_isNormalModelMatrix3Enabled),
        m_isModelViewMatrix4Enabled(rhs.m_isModelViewMatrix4Enabled),
        m_isModelViewMatrix3Enabled(rhs.m_isModelViewMatrix3Enabled),
        m_isNormalModelViewMatrix3Enabled(rhs.m_isNormalModelViewMatrix3Enabled),
        m_isModelViewProjectionMatrix4Enabled(rhs.m_isModelViewProjectionMatrix4Enabled),
        m_isProjectionMatrix4Enabled(rhs.m_isProjectionMatrix4Enabled),
        m_isCameraLookAtVectorEnabled(rhs.m_isCameraLookAtVectorEnabled),
        m_isCameraPositionEnabled(rhs.m_isCameraPositionEnabled),
        m_isLightsActivationEnabled(rhs.m_isLightsActivationEnabled),
        m_isMaterialActivationEnabled(rhs.m_isMaterialActivationEnabled),
        m_isMorphingMeshActivationEnabled(rhs.m_isMorphingMeshActivationEnabled),
        m_isPointSpriteActivationEnabled(rhs.m_isPointSpriteActivationEnabled),
        m_isTextureActivationEnabled(rhs.m_isTextureActivationEnabled),
        m_isCanvasActivationEnabled(rhs.m_isCanvasActivationEnabled),
        m_lightCoordinateSpace(rhs.m_lightCoordinateSpace),
        m_autoUniforms(rhs.m_autoUniforms)
    {
    }

    GenericShaderParamSetter::SharedPointer GenericShaderParamSetter::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER);
        GenericShaderParamSetter* ptr = FEATSTD_NEW(GenericShaderParamSetter)();
        if (ptr == 0) {
            return GenericShaderParamSetter::SharedPointer(0);
        }

        GenericShaderParamSetter::SharedPointer sharedPointer(ptr);
        return sharedPointer;
    }

    AbstractShaderParamSetter::SharedPointer GenericShaderParamSetter::Clone() const
    {
        return GenericShaderParamSetter::SharedPointer(FEATSTD_NEW(GenericShaderParamSetter)(*this));
    }

    bool GenericShaderParamSetter::Activate(Node& node, const Appearance::SharedPointer& appearance, Int instanceIndex)
    {
        bool isSuccess = true;

        if (appearance == 0) {
            return false;
        }

        MemoryManagement::SharedPointer<Shader> shader = appearance->GetShader();
        if (shader == 0) {
            return false;
        }

        const Appearance& appearanceInstance = appearance.GetSharedInstance();
        Shader& shaderInstance = shader.GetSharedInstance();

        const void* accessorId = UniformCacheParams;
        s_uniformCacheHandle = shaderInstance.GetUniformCacheHandle(accessorId);
        if (!s_uniformCacheHandle.IsValid()) {
            if (!shaderInstance.RegisterUniformCacheAccessor(accessorId, UniformCacheParams, UniformCacheIndicesCount, s_uniformCacheHandle)) {
                FEATSTD_LOG_ERROR("GenericShaderParamSetter::Activate failed, RegisterUniformCacheHandle failed.");
                return false;
            }
        }

        //Iterate over stored function pointers and invoke their Activation.
        for(AutoUniformContainer::Iterator it = m_autoUniforms.Begin(); it != m_autoUniforms.End(); ++it)
        {
            if(!(this->*(*it))(node, appearanceInstance, shaderInstance, instanceIndex)){
                isSuccess = false;
            }
        }

        //Activate constant attributes
        if (!ActivateConstantAttributeList(*shader)) {
            isSuccess = false;
        }

        //Activate manually set uniforms.
        if (!ActivateUniformList(*shader, instanceIndex)) {
            isSuccess = false;
        }

        return isSuccess;
    }

    void GenericShaderParamSetter::SetAutoUniformEnabled(bool& currentEnableState, bool newEnabledState, ActivateFn activateFunction)
    {
        if (currentEnableState != newEnabledState) {
            if (newEnabledState) {
                SizeType insertIndex = 0;
                for (const ActivateFn* fnEntireList = GetActivationFunctionList(); (insertIndex < m_autoUniforms.Size()) && (*fnEntireList != activateFunction); ++fnEntireList)
                {
                    if (*fnEntireList == m_autoUniforms[insertIndex]) {
                        ++insertIndex;
                    }
                }
                currentEnableState = m_autoUniforms.Insert(insertIndex, activateFunction);
            }
            else {
                for (SizeType removeIndex = 0; (removeIndex < m_autoUniforms.Size()) && (currentEnableState); removeIndex++) {
                    if (m_autoUniforms[removeIndex] == activateFunction) {
                        currentEnableState = !(m_autoUniforms.Remove(removeIndex));
                    }
                }
            }
        }
    }

    void GenericShaderParamSetter::SetModelMatrix4Enabled(bool enableModelMatrix4)
    {
        SetAutoUniformEnabled(
            m_isModelMatrix4Enabled,
            enableModelMatrix4,
            &GenericShaderParamSetter::ActivateModelMatrix4);
    }

    void GenericShaderParamSetter::SetModelMatrix3Enabled(bool enableModelMatrix3)
    {
        SetAutoUniformEnabled(
            m_isModelMatrix3Enabled,
            enableModelMatrix3,
            &GenericShaderParamSetter::ActivateModelMatrix3);
    }

    void GenericShaderParamSetter::SetNormalModelMatrix3Enabled(bool enableNormalModelMatrix3)
    {
        SetAutoUniformEnabled(
            m_isNormalModelMatrix3Enabled,
            enableNormalModelMatrix3,
            &GenericShaderParamSetter::ActivateNormalModelMatrix);
    }

    void GenericShaderParamSetter::SetModelViewMatrix4Enabled(bool enableModelViewMatrix4)
    {
        SetAutoUniformEnabled(
            m_isModelViewMatrix4Enabled,
            enableModelViewMatrix4,
            &GenericShaderParamSetter::ActivateModelViewMatrix4);
    }

    void GenericShaderParamSetter::SetModelViewMatrix3Enabled(bool enableModelViewMatrix3)
    {
        SetAutoUniformEnabled(
            m_isModelViewMatrix3Enabled,
            enableModelViewMatrix3,
            &GenericShaderParamSetter::ActivateModelViewMatrix3);
    }

    void GenericShaderParamSetter::SetNormalModelViewMatrix3Enabled(bool enableNormalModelViewMatrix3)
    {
        SetAutoUniformEnabled(
            m_isNormalModelViewMatrix3Enabled, 
            enableNormalModelViewMatrix3, 
            &GenericShaderParamSetter::ActivateNormalModelViewMatrix);
    }

    void GenericShaderParamSetter::SetModelViewProjectionMatrix4Enabled(bool enableModelViewProjectionMatrix4)
    {
        SetAutoUniformEnabled(
            m_isModelViewProjectionMatrix4Enabled,
            enableModelViewProjectionMatrix4,
            &GenericShaderParamSetter::ActivateModelViewProjectionMatrix);
    }

    void GenericShaderParamSetter::SetProjectionMatrix4Enabled(bool enableProjectionMatrix4)
    {
        SetAutoUniformEnabled(
            m_isProjectionMatrix4Enabled,
            enableProjectionMatrix4,
            &GenericShaderParamSetter::ActivateProjectionMatrix);
    }

    void GenericShaderParamSetter::SetCameraLookAtVectorEnabled(bool enableCameraLookAtVector)
    {
        SetAutoUniformEnabled(
            m_isCameraLookAtVectorEnabled,
            enableCameraLookAtVector,
            &GenericShaderParamSetter::ActivateCameraLookAtVector);
    }

    void GenericShaderParamSetter::SetCameraPositionEnabled(bool enableCameraPosition)
    {
        SetAutoUniformEnabled(
            m_isCameraPositionEnabled,
            enableCameraPosition,
            &GenericShaderParamSetter::ActivateCameraPosition);
    }

    void GenericShaderParamSetter::SetLightActivationEnabled(bool enableLightActivation)
    {
        SetAutoUniformEnabled(
            m_isLightsActivationEnabled,
            enableLightActivation,
            &GenericShaderParamSetter::ActivateLights);
    }

    void GenericShaderParamSetter::SetMaterialActivationEnabled(bool enableMaterialActivation)
    {
        SetAutoUniformEnabled(
            m_isMaterialActivationEnabled,
            enableMaterialActivation,
            &GenericShaderParamSetter::ActivateMaterial);
    }

    void GenericShaderParamSetter::SetMorphingMeshActivationEnabled(bool enableMorphingMeshActivation)
    {
        SetAutoUniformEnabled(
            m_isMorphingMeshActivationEnabled,
            enableMorphingMeshActivation,
            &GenericShaderParamSetter::ActivateMorphingMesh);
    }

    void GenericShaderParamSetter::SetPointSpriteActivationEnabled(bool enablePointSpriteActivation)
    {
        SetAutoUniformEnabled(
            m_isPointSpriteActivationEnabled,
            enablePointSpriteActivation,
            &GenericShaderParamSetter::ActivatePointSprite);
    }

    void GenericShaderParamSetter::SetTextureActivationEnabled(bool enableTextureActivation)
    {
        SetAutoUniformEnabled(
            m_isTextureActivationEnabled,
            enableTextureActivation,
            &GenericShaderParamSetter::ActivateTextures);
    }

    void GenericShaderParamSetter::SetCanvasActivationEnabled(bool enableCanvasActivation)
    {
        SetAutoUniformEnabled(
            m_isCanvasActivationEnabled,
            enableCanvasActivation,
            &GenericShaderParamSetter::ActivateCanvasRenderable);
    }

    bool GenericShaderParamSetter::ActivateModelMatrix3(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, shader, CANDERA_LINT_REASON_NONCONST)

        bool areMatricesSet = true;
        Int uniformLocation;

        if (m_isModelMatrix3Enabled) {
            if (shader.GetUniformLocation(s_uniformCacheHandle, ModelMatrix3, uniformLocation)) {
                Matrix3 modelMatrix3InversedTransposed = node.GetWorldTransform();
                modelMatrix3InversedTransposed.Inverse();
                modelMatrix3InversedTransposed.Transpose();
                areMatricesSet = Renderer::SetUniform<Shader::FloatMat3>(uniformLocation, modelMatrix3InversedTransposed.GetData(), instanceIndex);
            }
            else {
                const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::ModelMatrix3);
                CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
            }
        }

        return areMatricesSet;
    }

    bool GenericShaderParamSetter::ActivateModelMatrix4(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, shader, CANDERA_LINT_REASON_NONCONST)

        bool isMatrixSet = true;
        Int uniformLocation;

        const Matrix4& modelMatrix = node.GetWorldTransform();
        if (shader.GetUniformLocation(s_uniformCacheHandle, ModelMatrix4, uniformLocation)) {
            isMatrixSet = Renderer::SetUniform<Shader::FloatMat4>(uniformLocation, modelMatrix.GetData(), instanceIndex);
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::ModelMatrix4);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isMatrixSet;
    }

    bool GenericShaderParamSetter::ActivateNormalModelMatrix(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        Int uniformLocation;
        bool isMatrixSet = true;

        if (shader.GetUniformLocation(s_uniformCacheHandle, NormalModelMatrix3, uniformLocation)) {
            Matrix3 normalModelMatrix3InversedTransposed = node.GetWorldScale() * node.GetWorldRotation();
            normalModelMatrix3InversedTransposed.Inverse();
            normalModelMatrix3InversedTransposed.Transpose();
            isMatrixSet = Renderer::SetUniform<Shader::FloatMat3>(uniformLocation, normalModelMatrix3InversedTransposed.GetData(), instanceIndex);
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::NormalModelMatrix3);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isMatrixSet;
    }

    bool GenericShaderParamSetter::ActivateModelViewMatrix3(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        bool isMatrixSet = true;
        Int uniformLocation;

        if (shader.GetUniformLocation(s_uniformCacheHandle, ModelViewMatrix3, uniformLocation)) {
            Matrix3 modelViewMatrixInversedTransposed = node.GetWorldTransform() * RenderDevice::GetActiveCamera()->GetViewMatrix();
            modelViewMatrixInversedTransposed.Inverse();
            modelViewMatrixInversedTransposed.Transpose();
            isMatrixSet = Renderer::SetUniform<Shader::FloatMat3>(uniformLocation, modelViewMatrixInversedTransposed.GetData(), instanceIndex);
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::ModelViewMatrix3);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isMatrixSet;
    }

    bool GenericShaderParamSetter::ActivateModelViewMatrix4(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        bool isMatrixSet = true;
        Int uniformLocation;

        if (shader.GetUniformLocation(s_uniformCacheHandle, ModelViewMatrix4, uniformLocation)) {
            const Matrix4& modelViewMatrix = node.GetWorldTransform() * RenderDevice::GetActiveCamera()->GetViewMatrix();
            isMatrixSet = Renderer::SetUniform<Shader::FloatMat4>(uniformLocation, modelViewMatrix.GetData(), instanceIndex);
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::ModelViewMatrix4);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isMatrixSet;
    }

    bool GenericShaderParamSetter::ActivateNormalModelViewMatrix(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        Int uniformLocation;
        bool isMatrixSet = true;
        if (shader.GetUniformLocation(s_uniformCacheHandle, NormalModelViewMatrix3, uniformLocation)) {
            Matrix3 normalModelViewMatrix3InversedTransposed = (node.GetWorldScale() * node.GetWorldRotation()) * RenderDevice::GetActiveCamera()->GetViewMatrix();
            normalModelViewMatrix3InversedTransposed.Inverse();
            normalModelViewMatrix3InversedTransposed.Transpose();
            isMatrixSet = Renderer::SetUniform<Shader::FloatMat3>(uniformLocation, normalModelViewMatrix3InversedTransposed.GetData(), instanceIndex);
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::NormalModelViewMatrix3);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isMatrixSet;
    }

    bool GenericShaderParamSetter::ActivateModelViewProjectionMatrix(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        Int uniformLocation;
        bool isMatrixSet = true;

        if (shader.GetUniformLocation(s_uniformCacheHandle, ModelViewProjectionMatrix4, uniformLocation)) {
            const Matrix4& mvpMatrix = node.GetWorldTransform() * RenderDevice::GetActiveCamera()->GetViewProjectionMatrix();
            isMatrixSet = Renderer::SetUniform<Shader::FloatMat4>(uniformLocation, mvpMatrix.GetData(), instanceIndex);
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::ModelViewProjectionMatrix4);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isMatrixSet;
    }

    bool GenericShaderParamSetter::ActivateProjectionMatrix(const Node& /*node*/, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        Int uniformLocation;
        bool isMatrixSet = true;

        if (shader.GetUniformLocation(s_uniformCacheHandle, ProjectionMatrix4, uniformLocation)) {
            // if instanceIndex >= 1, this uniform (which is shared amongst instances) has already been set
            if ((instanceIndex < 1) && (shader.GetUniformLocation(s_uniformCacheHandle, ProjectionMatrix4, uniformLocation))) {
                const Matrix4& pMatrix = RenderDevice::GetActiveCamera()->GetProjection()->GetProjectionMatrix();
                isMatrixSet = Renderer::SetUniformInRenderDevice<Shader::FloatMat4>(uniformLocation, pMatrix.GetData());
            }
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::ProjectionMatrix4);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isMatrixSet;
    }

    bool GenericShaderParamSetter::ActivateCameraLookAtVector(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        // if instanceIndex >= 1, this uniform (which is shared amongst instances) has already been set
        if (instanceIndex >= 1) {
            FEATSTD_DEBUG_ASSERT(IsInstancingSupported());
            return true;
        }

        Int uniformLocation;
        bool isCameraVectorSet = true;

        if (shader.GetUniformLocation(s_uniformCacheHandle, CameraLookAtVector, uniformLocation)) {
            //Transform camera direction into object space.
            Vector3 camVec;
            Matrix4 viewMat = RenderDevice::GetActiveCamera()->GetViewMatrix();
            //Get look-at-vector of camera.
            camVec.SetX(viewMat.Get(0,2));
            camVec.SetY(viewMat.Get(1,2));
            camVec.SetZ(viewMat.Get(2,2));
            if (m_lightCoordinateSpace == Light::Object) {
                Matrix4 inverseWorldRotation = node.GetWorldRotation();
                inverseWorldRotation.Inverse();
                camVec.TransformCoordinate(inverseWorldRotation);
            }
            static_cast<void>(camVec.Normalize());
            isCameraVectorSet = Renderer::SetUniformInRenderDevice<Shader::FloatVec3>(uniformLocation, camVec.GetData());
        }
        else {
            const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::CameraLookAtVector);
            CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        }

        return isCameraVectorSet;
    }

    bool GenericShaderParamSetter::ActivateCameraPosition(const Node& /*node*/, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        Int uniformLocation;
        bool isCameraVectorSet = true;

        // if instanceIndex >= 1, this uniform (which is shared amongst instances) has already been set
        if (instanceIndex < 1) {
            if (shader.GetUniformLocation(s_uniformCacheHandle, CameraPosition, uniformLocation)) {
                Vector3 cameraPosition = RenderDevice::GetActiveCamera()->GetWorldPosition();
                isCameraVectorSet = Renderer::SetUniformInRenderDevice<Shader::FloatVec3>(uniformLocation, cameraPosition.GetData());
            }
            else {
                const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::CameraPosition);
                CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
            }
        }

        return isCameraVectorSet;
    }

    bool GenericShaderParamSetter::ActivateLights(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        // if instanceIndex > 0, lights (which are shared amongst instances) have already been set
        if (instanceIndex > 0) {
            return true;
        }

        bool areLightsActivated = true;

        Light::SetCoordinateSpace(m_lightCoordinateSpace);

        UInt index = 0;
        const Light* light = RenderDevice::GetFirstActiveLight();
        while((light != 0) && (index < CANDERA_MAX_LIGHTS_COUNT))
        {
            if (light->IsEffectiveRenderingEnabled() && node.IsInScopeOf(light->GetScopeMask())) {
                if (!light->Activate(shader, node, index)) {
                    areLightsActivated = false;
                }
                else {
                    ++index;
                }
            }
            light = RenderDevice::GetNextActiveLight();
        }
        Int uniformLocation;
        // if the number of lights changes in Candera, the UniformCacheIndex enums need to be adapted accordingly
        FEATSTD_COMPILETIME_ASSERT(CANDERA_MAX_LIGHTS_COUNT == ((LightEnabled7 - LightEnabled) + 1));
        for (; index < CANDERA_MAX_LIGHTS_COUNT; ++index) {
            if (shader.GetUniformLocation(s_uniformCacheHandle, static_cast<UInt>(LightEnabled) + index, uniformLocation)) {
                static const Int value = 0;
                areLightsActivated = areLightsActivated && Renderer::SetUniformInRenderDevice<Shader::Bool>(uniformLocation, &value);
                Renderer::SetLight(0, index);
            }
        }

        return areLightsActivated;
    }

    bool GenericShaderParamSetter::ActivateMaterial(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const
    {
        bool isMaterialActivated = true;  //Also returns true if material doesn't exist, this is no error. Just returns false if activation of material failed.

        Material::SharedPointer material = appearance.GetMaterial();
        if (material != 0) {
            FEATSTD_LINT_NEXT_EXPRESSION(948, "(&node == 0) should never happen (it may though). code left as is to ensure backward compatibility")
                isMaterialActivated = material->Activate(shader, ((&node) == 0) ? 1.0F : node.GetEffectiveAlphaValue(), instanceIndex);
            if (!isMaterialActivated) {
                FEATSTD_LOG_WARN("Activation of material failed.");
            }
        }

        return isMaterialActivated;
    }

    bool GenericShaderParamSetter::ActivateMorphingMesh(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        FEATSTD_DEBUG_ASSERT(instanceIndex <= 0);
        FEATSTD_UNUSED(instanceIndex);

        const MorphingMesh* morphingMesh = Dynamic_Cast<const MorphingMesh*>(&node);
        if (0 == morphingMesh) {
            return false;
        }

        Int uniformLocation;
        if (shader.GetUniformLocation(s_uniformCacheHandle, MorphWeightArray, uniformLocation)) {
            // Uniform may not exist.
            static_cast<void>(Renderer::SetUniformInRenderDevice<Shader::Float>(uniformLocation, &(morphingMesh->m_morphWeight[0]),
                static_cast<UInt>(MorphingMesh::MaxMorphWeightCount)));
        }

        return true;
    }

    bool GenericShaderParamSetter::ActivatePointSprite(const Node& node, const Appearance& /*appearance*/, Shader& shader, Int instanceIndex) const
    {
        const PointSprite* pointSprite = Dynamic_Cast<const PointSprite*>(&node);
        if (0 == pointSprite) {
            return false;
        }

        Int uniformLocation;
        if (shader.GetUniformLocation(s_uniformCacheHandle, PointSpriteSize, uniformLocation)) {
            const Camera* camera = RenderDevice::GetActiveCamera();
            const Float distanceToCamera = pointSprite->GetWorldPosition().GetDistanceTo(camera->GetWorldPosition());
            const Float sizeFactorAbc = Math::SquareRoot(1.0F /
                (pointSprite->GetPointSizeScaleA() + (pointSprite->GetPointSizeScaleB() * distanceToCamera) + (pointSprite->GetPointSizeScaleC() * distanceToCamera * distanceToCamera)));
            Float finalSize = pointSprite->GetPointSize() * sizeFactorAbc;
           
            return Renderer::SetUniform<Shader::Float>(uniformLocation, &finalSize, instanceIndex);
        }

        const Char* missingUniform = ShaderParamNames::GetUniformName(ShaderParamNames::PointSpriteSize);
        CANDERA_GENERICSHADERPARAMSETTER_WARN_MISSING_UNIFORM(missingUniform);
        return false;
    }

    bool GenericShaderParamSetter::ActivateTextures(const Node& /*node*/, const Appearance& appearance, Shader& shader, Int instanceIndex) const
    {
        bool areTexturesActivated = true; //Also returns true if no textures exist, this is no error. Just returns false if activation of textures failed.

        // if instanceIndex >= 1, textures (which are shared amongst instances) have already been set
        if (instanceIndex < 1) {
            for (UInt unit = 0; unit < CANDERA_MAX_TEXTURE_UNIT_COUNT; unit++) {
                Texture::SharedPointer tex = appearance.GetTexture(unit);
                if (tex != 0) {
                    bool activated = tex->Activate(shader, unit);
                    if (!activated) {
                        FEATSTD_LOG_WARN("Activation of texture[%d] failed.", unit);
                        areTexturesActivated = false;
                    }
                }
            }
        }

        return areTexturesActivated;
    }

    bool GenericShaderParamSetter::ActivateCanvasRenderable(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const
    {
        
#if defined(CANDERA_3D_CANVAS_ENABLED)

        bool isCanvasActivated = true;

        const CanvasRenderable* canvasRenderable = Dynamic_Cast<const CanvasRenderable*>(&node);
        if (canvasRenderable != 0) {

            Int pivotUniformLocation;
            Int sizeUniformLocation;
            if (shader.GetUniformLocation(s_uniformCacheHandle, CanvasPivot, pivotUniformLocation) &&
                shader.GetUniformLocation(s_uniformCacheHandle, CanvasSize, sizeUniformLocation)) {

                Rectangle renderRectangle = canvasRenderable->GetVertexRectangle();
                Vector2 pivot = renderRectangle.GetPosition();
                Vector2 size = canvasRenderable->GetActualDimension(appearance);

                isCanvasActivated = Renderer::SetUniform<Shader::FloatVec2>(pivotUniformLocation, pivot.GetData(), instanceIndex) &&
                    Renderer::SetUniform<Shader::FloatVec2>(sizeUniformLocation, size.GetData(), instanceIndex);
            }
        }

        return isCanvasActivated;
#else
        FEATSTD_UNUSED(node);
        FEATSTD_UNUSED(appearance);
        FEATSTD_UNUSED(shader);
        FEATSTD_UNUSED(instanceIndex);
        return true;
#endif
        
    }

    FEATSTD_RTTI_DEFINITION(GenericShaderParamSetter, ShaderParamSetter)
}// namespace
