//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderStateOrderCriterion.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Renderer.h>

namespace Candera {

    using namespace Diagnostics;
    using namespace Internal;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    FEATSTD_RTTI_DEFINITION(RenderStateOrderCriterion, RenderOrderCriterion)

    //Hash for mapping custom OpenGL handles to values with a restricted number of bits.
    static const UInt32 c_hash = 2654435761U;

    void RenderStateOrderCriterion::PrepareRenderOrderCriterionValue(Candera::Node &n) const
    {
        RenderState renderState = {};

        //Initialize the most significant bits from the orderCriterionValue with a hash generated from the Shader and Texture's handle.
        const SharedPointer<Appearance>& nodeAppearance = n.GetAppearance();
        if (!nodeAppearance.PointsToNull()) {
            const SharedPointer<Shader>& nodeShader = nodeAppearance->GetShader();
            if (!nodeShader.PointsToNull()) {
                UInt32 handle = static_cast<UInt32>(nodeShader->GetProgramHandle());
                FEATSTD_LINT_NEXT_EXPRESSION(1960, "lowest significant bits sufficient to identify the shader")
                renderState.m_bits.m_shaderHash =  handle * c_hash;
            }

            //Use 6 bits for Texture handle hash.
            const SharedPointer<Texture>& nodeTexture = nodeAppearance->GetTexture();
            if (!nodeTexture.PointsToNull()) {
                const SharedPointer<TextureImage>& nodeTextureImage = nodeTexture->GetTextureImage();
                if (!nodeTextureImage.PointsToNull()) {
                    UInt32 handle = static_cast<UInt32>(nodeTextureImage->GetVideoMemoryHandle());
                    FEATSTD_LINT_NEXT_EXPRESSION(1960, "lowest significant bits sufficient to identify the texture")
                    renderState.m_bits.m_textureHash = static_cast<UInt32>(handle * c_hash);
                }
            }
        }

        // Retrieve the base RenderMode, which is either the RenderMode applied to active Camera or the DefaultRenderMode.
        // Render state values of BaseRenderMode are taken if Node's RenderMode has inheritance bit set for certain RenderState.
        SharedPointer<RenderMode> baseRenderMode;
        const Camera* activeCamera = RenderDevice::GetActiveCamera();
        if (activeCamera != 0) {
            const SharedPointer<Appearance>& cameraAppearance = activeCamera->GetAppearance();
            if (!cameraAppearance.PointsToNull()) {
                baseRenderMode = cameraAppearance->GetRenderMode();
            }
        }

        if (baseRenderMode.PointsToNull()) {
            baseRenderMode = Renderer::GetDefaultRenderMode();
        }

        //Retrieve the Node's renderMode.
        SharedPointer<RenderMode> nodeRenderMode;
        if (!nodeAppearance.PointsToNull()) {
            nodeRenderMode = nodeAppearance->GetRenderMode();
        }

        //If node has no RenderMode use the base one.
        if (nodeRenderMode.PointsToNull()) {
            nodeRenderMode = baseRenderMode;
        }

        if (!nodeRenderMode.PointsToNull()) {
            bool inherit = nodeRenderMode->IsInherited(RenderMode::BlendingEnabledBit);
            bool blendingEnabled = (inherit && baseRenderMode->IsBlendingEnabled()) || ((!inherit) && nodeRenderMode->IsBlendingEnabled());
            renderState.m_bits.m_rmBlendingEnabled = static_cast<UInt32>(blendingEnabled);

            inherit = nodeRenderMode->IsInherited(RenderMode::BlendModeBit);
            RenderMode::BlendFactor srcFactor = inherit ? baseRenderMode->GetBlendMode().sourceAlphaFactor : nodeRenderMode->GetBlendMode().sourceAlphaFactor;
            RenderMode::BlendOperation blendOp = inherit ? baseRenderMode->GetBlendMode().operationAlpha : nodeRenderMode->GetBlendMode().operationAlpha;
            RenderMode::BlendFactor dstFactor = inherit ? baseRenderMode->GetBlendMode().destAlphaFactor : nodeRenderMode->GetBlendMode().destAlphaFactor;
            renderState.m_bits.m_rmAlphaBlendModeSrcFactor = static_cast<UInt32>(srcFactor);
            renderState.m_bits.m_rmAlphaBlendModeOperation = static_cast<UInt32>(blendOp);
            renderState.m_bits.m_rmAlphaBlendModeDstFactor = static_cast<UInt32>(dstFactor);

            inherit = nodeRenderMode->IsInherited(RenderMode::DepthWriteEnabledBit);
            bool depthWriteEnabled = (inherit && baseRenderMode->IsDepthWriteEnabled()) || ((!inherit) && nodeRenderMode->IsDepthWriteEnabled());
            renderState.m_bits.m_rmDepthWriteEnabled = static_cast<UInt32>(depthWriteEnabled);

            inherit = nodeRenderMode->IsInherited(RenderMode::DepthWriteEnabledBit);
            bool depthTestEnabled = (inherit && baseRenderMode->IsDepthTestEnabled()) || ((!inherit) && nodeRenderMode->IsDepthTestEnabled());
            renderState.m_bits.m_rmDepthTestEnabled = static_cast<UInt32>(depthTestEnabled);

            inherit = nodeRenderMode->IsInherited(RenderMode::DepthWriteEnabledBit);
            bool stencilTestEnabled = (inherit && baseRenderMode->IsStencilTestEnabled()) || ((!inherit) && nodeRenderMode->IsStencilTestEnabled());
            renderState.m_bits.m_rmStencilTestEnabled = static_cast<UInt32>(stencilTestEnabled);

            inherit = nodeRenderMode->IsInherited(RenderMode::DepthWriteEnabledBit);
            bool colorWriteRedEnabled = (inherit && baseRenderMode->IsColorWriteRedEnabled()) || ((!inherit) && nodeRenderMode->IsColorWriteRedEnabled());
            bool colorWriteGreenEnabled = (inherit && baseRenderMode->IsColorWriteGreenEnabled()) || ((!inherit) && nodeRenderMode->IsColorWriteGreenEnabled());
            bool colorWriteBlueEnabled = (inherit && baseRenderMode->IsColorWriteBlueEnabled()) || ((!inherit) && nodeRenderMode->IsColorWriteBlueEnabled());
            bool colorWriteAlphaEnabled = (inherit && baseRenderMode->IsColorWriteAlphaEnabled()) || ((!inherit) && nodeRenderMode->IsColorWriteAlphaEnabled());
            renderState.m_bits.m_rmRWriteEnabled = static_cast<UInt32>(colorWriteRedEnabled);
            renderState.m_bits.m_rmGWriteEnabled = static_cast<UInt32>(colorWriteGreenEnabled);
            renderState.m_bits.m_rmBWriteEnabled = static_cast<UInt32>(colorWriteBlueEnabled);
            renderState.m_bits.m_rmAWriteEnabled = static_cast<UInt32>(colorWriteAlphaEnabled);

            inherit = nodeRenderMode->IsInherited(RenderMode::WindingModeBit);
            RenderMode::Winding windingMode = inherit ? baseRenderMode->GetWinding() : nodeRenderMode->GetWinding();
            renderState.m_bits.m_rmWindingMode = static_cast<UInt32>(windingMode);

            inherit = nodeRenderMode->IsInherited(RenderMode::CullingModeBit);
            RenderMode::Culling cullMode = (inherit ? baseRenderMode->GetCulling() : nodeRenderMode->GetCulling());
            renderState.m_bits.m_rmCullingMode = static_cast<UInt32>(cullMode);
        }

        SetRenderOrderCriterionValue(n, OrderCriterionValue(renderState.m_value));
    }

    bool RenderStateOrderCriterion::IsBefore(const Node& a, const Node& b) const
    {
        return GetRenderOrderCriterionValue(a).GetUInt32() < GetRenderOrderCriterionValue(b).GetUInt32();
    }

    bool RenderStateOrderCriterion::PrepareRenderOrderCriterionValues(RenderOrderBin::NodeContainer& nodeContainer,
        RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const
    {
        if (!Reserve<UInt32>(sortingContainer, nodeContainerSize, 0)) {
            FEATSTD_LOG_ERROR("Prepare render order criterion value failed, could not reserve space in bins.");
            return false;
        }

        UInt16 indexOfNode = 0;
        for (UInt i = 0; i < static_cast<UInt>(nodeContainerSize); ++i) {
            Node& n = *(nodeContainer[i]);

            // Due to the complexity of preparing the individual render state order criterion, there's no use to inline it.
            PrepareRenderOrderCriterionValue(n);

            sortingContainer[indexOfNode] = RenderOrderBin::SortingElement(GetRenderOrderCriterionValue(n).GetUInt32(), indexOfNode);
            indexOfNode++;
        }

        return true;
    }

    struct RenderStateOrderComparator
    {
        bool operator()(const RenderOrderBin::SortingElement& a, const RenderOrderBin::SortingElement& b) const {
            return a.IsBeforeUInt32(b);
        }
    };

    void RenderStateOrderCriterion::Sort(RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const
    {
        RenderStateOrderComparator comparator;
        sortingContainer.Sort(comparator, 0, containerSize - 1);
    }

    bool RenderStateOrderCriterion::HasOrderChanged(const RenderOrderBin::SortingBin& currentBin,
        const RenderOrderBin::SortingBin& previousBin) const
    {
        RenderStateOrderComparator comparator;
        RenderOrderBin::SortingElement startingKey(FeatStd::Internal::NativeTypeLimit<UInt32>::Min(), 0);
        return RenderOrderCriterion::HasOrderChanged(currentBin, previousBin, comparator, startingKey);
    }
}
