//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RENDERORDERCRITERION_H)
#define CANDERA_RENDERORDERCRITERION_H

#include <Candera/Engine3D/RenderOrder/OrderCriterion.h>
#include <Candera/Engine3D/RenderOrder/RenderOrderBin.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */

/**
 *  @brief  The abstract class RenderOrderCriterion defines an interface to prepare the
 *          order criterion values (i.e. sorting keys) of a RenderOrderBin and sort it.
 */
class RenderOrderCriterion : public OrderCriterion
{
    FEATSTD_TYPEDEF_BASE(OrderCriterion);

public:

    /**
    *  Constructor
    */
    RenderOrderCriterion() {}

    /**
    *  Destructor
    */
    virtual ~RenderOrderCriterion() {}

    /**
     *  Calculates the order criterion values (i.e. sorting keys) for the given nodes.
     *  @param nodeContainer     The container of nodes to compute the sorting keys for.
     *  @param sortingContainer  The container to store the sorting keys and node indices.
     *  @param nodeContainerSize The number of nodes in use in the nodeContainer. This can be
     *                           smaller than the actual size of the container
     *  @return True if all order criterion values could be prepared, False otherwise.
     */
    virtual bool PrepareRenderOrderCriterionValues(Internal::RenderOrderBin::NodeContainer& nodeContainer,
        Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const = 0;

    /**
     *  Sort the container of the RenderOrderBin
     *  @param sortingContainer  The container to be sorted.
     *  @param containerSize     The size of the container to be sorted. This can be smaller than
     *                           the actual size of the container to sort a subsection.
     */
    virtual void Sort(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const = 0;

    /**
     *  Determine if the render order has changed from the previous pass to the current one.
     *  If the order has not changed, the render order bin will use the result of the previous pass,
     *  instead of sorting the current pass.
     *  @param currentBin   The sorting bin of the previous pass.
     *  @param previousBin  The sorting bin of the current pass.
     *  @return True, if the bins are different or the order has changed. False, otherwise.
     */
    virtual bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
        const Internal::RenderOrderBin::SortingBin& previousBin) const;

    FEATSTD_RTTI_DECLARATION();

protected:

    /**
     *  If necessary, reserve space for nodeContainerSize elements in the given sortingContainer.
     *  @param sortingContainer  The container to reserve space in.
     *  @param nodeContainerSize The number of elements to reserve space for.
     *  @param initialKey        The initial key for validating the order. Depending on the comparator and the
     *                           nature of the RenderOrderBin::SortingElement, this is a minimum or maximum value.
     *  @return True, if the sortingContainer was big enough or space could be reserved. False, otherwise.
     */
    template<typename SortingKey>
    bool Reserve(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize,
        SortingKey initialKey) const;

    /**
     *  Determine if the render order has changed from the previous pass to the current one.
     *  If the order has not changed, the render order bin will use the result of the previous pass,
     *  instead of sorting the current pass.
     *  @param currentBin   The sorting bin of the previous pass.
     *  @param previousBin  The sorting bin of the current pass.
     *  @param comparator   The comparator to validate the order.
     *  @param key          The initial key for validating the order. Depending on the comparator and the
     *                      nature of the RenderOrderBin::SortingElement, this is a minimum or maximum value.
     *  @return True, if the bins are different or the order has changed. False, otherwise.
     */
    template<typename Comparator>
    bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
        const Internal::RenderOrderBin::SortingBin& previousBin, Comparator& comparator,
        Internal::RenderOrderBin::SortingElement& key) const;
};

/** @} */ // end of RenderOrder3D

template<typename Comparator>
bool RenderOrderCriterion::HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
    const Internal::RenderOrderBin::SortingBin& previousBin, Comparator& comparator,
    Internal::RenderOrderBin::SortingElement& key) const
{
    if (currentBin.GetSize() != previousBin.GetSize()) {
        return true;
    }

    const Internal::RenderOrderBin::NodeContainer& currentNodeContainer = currentBin.GetNodeContainer();
    const Internal::RenderOrderBin::SortingContainer& currentSortingContainer = currentBin.GetSortingContainer();

    const Internal::RenderOrderBin::NodeContainer& previousNodeContainer = previousBin.GetNodeContainer();
    const Internal::RenderOrderBin::SortingContainer& previousSortingContainer = previousBin.GetSortingContainer();

    for (Int32 i = 0; i < previousBin.GetSize(); ++i) {
        UInt16 sortedIndexOfNode = previousSortingContainer[i].GetIndexOfNode();
        if (previousNodeContainer[sortedIndexOfNode] != currentNodeContainer[sortedIndexOfNode]) {
            return true;
        }

        if (comparator(currentSortingContainer[sortedIndexOfNode], key)) {
            return true;
        }

        key = currentSortingContainer[sortedIndexOfNode];
    }

    return false;
}


template<typename SortingKey>
bool RenderOrderCriterion::Reserve(Internal::RenderOrderBin::SortingContainer& sortingContainer,
    const SizeType nodeContainerSize, SortingKey initialKey) const
{
    if (nodeContainerSize > 0xFFff) {
        return false;
    }

    if (sortingContainer.GetCapacity() < nodeContainerSize) {
        if (!sortingContainer.Reserve(nodeContainerSize)) {
            return false;
        }
    }

    while (sortingContainer.Size() < nodeContainerSize) {
        if (!sortingContainer.Add(Internal::RenderOrderBin::SortingElement(initialKey, static_cast<UInt16>(sortingContainer.Size())))) {
            return false;
        }
    }

    return true;
}

}

#endif // CANDERA_RENDERORDERCRITERION_H
