//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderOrderBin.h"

#include <Candera/Engine3D/RenderOrder/RenderOrderCriterion.h>

namespace Candera {
namespace Internal {

/**
 * SortHelper is a wrapping function object to compare Nodes by polymorphic OrderCriterions.
 */
struct SortHelper
{
    const Candera::OrderCriterion* orderCriterion;
    bool operator()(const Node* a, const Node* b) const
    {
        return orderCriterion->IsBefore(*a, *b);
    }
};


RenderOrderBin::RenderOrderBin(Int32 maxNodeCount) :
    Base(),
    m_nameHash(0),
    m_rank(0),
    m_orderCriterion(0),
    m_currentBinIndex(0),
    m_isSortingEnabled(true),
    m_isRenderOrderCriterionAndSorted(false)
{
    static_cast<void>(m_bins[0].GetNodeContainer().Reserve(maxNodeCount));
}

RenderOrderBin::RenderOrderBin(const RenderOrderBin& src) :
Base(src),
m_nameHash(src.m_nameHash),
m_rank(src.m_rank),
m_orderCriterion(src.m_orderCriterion),
m_currentBinIndex(src.m_currentBinIndex),
m_isSortingEnabled(src.m_isSortingEnabled),
m_isRenderOrderCriterionAndSorted(src.m_isRenderOrderCriterionAndSorted)
{
    static_cast<void>(m_bins[0].GetNodeContainer().Reserve(src.m_bins[0].GetNodeContainer().GetCapacity()));
    static_cast<void>(m_bins[0].GetSortingContainer().Reserve(src.m_bins[0].GetSortingContainer().GetCapacity()));
    static_cast<void>(m_bins[1].GetNodeContainer().Reserve(src.m_bins[1].GetNodeContainer().GetCapacity()));
    static_cast<void>(m_bins[1].GetSortingContainer().Reserve(src.m_bins[1].GetSortingContainer().GetCapacity()));
}

RenderOrderBin::~RenderOrderBin()
{
    m_orderCriterion = 0;
}

void RenderOrderBin::SetOrderCriterion(const OrderCriterion* orderCriterion)
{
    const RenderOrderCriterion* renderOrderCriterion = Dynamic_Cast<const RenderOrderCriterion*>(orderCriterion);
    if (0 != renderOrderCriterion) {
        const SizeType capacity = m_bins[0].GetNodeContainer().GetCapacity();
        if (m_bins[0].GetSortingContainer().GetCapacity() < capacity) {
            static_cast<void>(m_bins[0].GetSortingContainer().Reserve(capacity));
            static_cast<void>(m_bins[1].GetNodeContainer().Reserve(capacity));
            static_cast<void>(m_bins[1].GetSortingContainer().Reserve(capacity));
        }
    }

    m_orderCriterion = orderCriterion;
}

void RenderOrderBin::Clear()
{
    const RenderOrderCriterion* renderOrderCriterion = Dynamic_Cast<const RenderOrderCriterion*>(m_orderCriterion);
    if (0 != renderOrderCriterion) {
        SwapSortingBins();
    }

    m_bins[m_currentBinIndex].Clear();

    m_isRenderOrderCriterionAndSorted = false;
}

bool RenderOrderBin::AddNode(Node* node)
{
    SortingBin& bin = m_bins[m_currentBinIndex];
    return bin.AddNode(node);
}

void RenderOrderBin::Sort()
{
    if (!IsSortingEnabled()) {
        return;
    }

    if (m_orderCriterion == 0) {
        return;
    }

    if (GetSize() == 0) {
        return;
    }

    const RenderOrderCriterion* renderOrderCriterion = Dynamic_Cast<const RenderOrderCriterion*>(m_orderCriterion);
    SortingBin& bin = m_bins[m_currentBinIndex];

    if (0 != renderOrderCriterion) {
        NodeContainer& nodeContainer = bin.GetNodeContainer();
        SortingContainer& sortingContainer = bin.GetSortingContainer();
        if (renderOrderCriterion->PrepareRenderOrderCriterionValues(nodeContainer, sortingContainer, GetSize())) {
            if (renderOrderCriterion->HasOrderChanged(bin, m_bins[1 - m_currentBinIndex])) {
                renderOrderCriterion->Sort(bin.GetSortingContainer(), GetSize());
            }
            else {
                // Order has not changed, so the previous sorting container can be switched to current.
                SwapSortingBins();
            }

            m_isRenderOrderCriterionAndSorted = true;
        }
    }
    else {
        for (SizeType i = 0; i < GetSize(); ++i) {
            m_orderCriterion->PrepareRenderOrderCriterionValue(*(bin.GetNodeContainer()[i]));
        }

        SortHelper sortHelper;
        sortHelper.orderCriterion = m_orderCriterion;
        bin.GetNodeContainer().Sort(sortHelper, 0, GetSize()-1);
    }
}

Node* RenderOrderBin::GetNode(Int32 index) const
{
    const SortingBin& bin = m_bins[m_currentBinIndex];

    if (m_isRenderOrderCriterionAndSorted) {
        return bin.GetNodeContainer()[bin.GetSortingContainer()[index].GetIndexOfNode()];
    }

    return bin.GetNodeContainer()[index];
}

}

} // namespace Candera
