//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ORDERCRITERIONVALUE_H)
#define CANDERA_ORDERCRITERIONVALUE_H

#include <Candera/Environment.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */

/**
 * @brief  OrderCriterionValue class encapsulates a 32bit Float or Int value.
 *
 * The encapsulated value can be set, retrieved and compared by various OrderCriterions.
 * Note: OrderCriterion shall use the same value type for all operations.
 */
class OrderCriterionValue
{
    public:
        OrderCriterionValue(UInt32 value) { SetUInt32(value); }
        OrderCriterionValue(Float value) { SetFloat(value); }

        /**
         * Sets 32-bit integer value.
         * Note: UInt32 value can be used as a bitfield to encode multiple sorting criteria and their cascading sorting order. Each criterion value can be encoded in
         * certain bits, where higher bits are more significant than lower bits when sorted. E.g. if render order shall be sorted by material and then by texture,
         * the higher 16 bit encode a material hash and the lower 16 bit encode the texture hash. A OrderCriterion, which sorts by an integer OrderCriterionValue
         * then implicitly sorts by material and then by texture. With that approach a custom cascading sort order can be realized.
         * @param value defines 32-bit unsigned integer value.
         */
        void SetUInt32(UInt32 value) { m_value.intValue = value; }

        /**
         * Gets integer value.
         * @return 32-bit unsigned integer value.
         */
        UInt32 GetUInt32() const { return m_value.intValue; }

        /**
         * Sets value as float.
         * @param value defines a float value.
         */
        void SetFloat(Float value) { m_value.floatValue = value; }

        /**
         * Gets float value.
         * @return float value.
         */
        Float GetFloat() const { return m_value.floatValue; }

    private:
        union {
            UInt32 intValue;
            Float floatValue;
        } m_value;
};

 /** @} */ // end of RenderOrder3D

}

#endif // CANDERA_ORDERCRITERIONVALUE_H
