//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ORDERCRITERION_H)
#define CANDERA_ORDERCRITERION_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/Node.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */
 
/**
 *  @brief  Abstract class OrderCriterion defines an interface that compares two Nodes according to a specific sort order criterion.
 *          Concrete implementations of the OrderCriterion interface can be used to control the order of containers like e.g. RenderOrderBin.
 */
class OrderCriterion
{
    public:
        /**
         *  Constructor
         */
        OrderCriterion() {}

        /**
         *  Destructor
         */
        virtual ~OrderCriterion() {}

        /**
         *  Compares two Nodes according to a specific sort order criterion.
         *  @param a First Node whose order criterion is compared to Node b.
         *  @param b Second Node whose order criterion is compared to Node a.
         *  @return true if first Node is in the order before the second Node given.
         */
        virtual bool IsBefore(const Node& a, const Node& b) const = 0;

        /**  Calculates information that makes the comparison simpler / cheaper, and stores it in the Node.
         *   This is called immediately before sorting and rendering, so in particular,
         *   Example: distance-based criteria can pre-calculate the distance to the camera. 
         *   @param n Node to calculate order criterion from.
         */
        virtual void PrepareRenderOrderCriterionValue(Node& n) const {
            FEATSTD_UNUSED(n);
        }

        FEATSTD_RTTI_DECLARATION();

    protected:
        /**
         *  Sets render order sorting number prepared by this class' derived function PrepareRenderOrderCriterionValue.
         *  @param n: The render order sort number is applied to the Node given.
         *  @param criterionValue specifies the prepared sort number used for comparison in this class' derived function IsBefore.
         */
        void SetRenderOrderCriterionValue(Node &n, OrderCriterionValue criterionValue) const { n.SetRenderOrderCriterionValue(criterionValue); }

        /**
         *  Retrieves the render order criterion value, specifies the prepared sort number used for comparison in this 
         *  class' derived function IsBefore.
         *  @param n: The render order sort number is applied to the Node given.
         *  @return The render order criterion value.
         */
        OrderCriterionValue GetRenderOrderCriterionValue(const Node &n) const { return n.GetRenderOrderCriterionValue(); }
};
 
 /** @} */ // end of RenderOrder3D
 
}

#endif // CANDERA_ORDERCRITERION_H
