//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_DistanceToCameraCriterion_h)
#define Candera_DistanceToCameraCriterion_h

#include <Candera/Engine3D/RenderOrder/RenderOrderCriterion.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */

// forward declaration.
class Node;

/**
 *  @brief  Abstract class CameraOrderCriterion implements different distance functions to determine the distance from camera to any node.
 *          The class CameraOrderCriterion is derived by DistanceToCameraOrderCriterion and ReverseDistanceToCameraOrderCriterion.  A render order bin that uses either
 *          DistanceToCameraOrderCriterion or a ReverseDistanceToCameraOrderCriterion sorts its objects by distance to camera.
 *
 *          Following distance functions are supported:
 *          + CameraPositionToBoundingSphere: Calculates the distance from camera position to the bounding sphere of another node. This is the default function. Fastest
 *            function, but less precise in comparison to bounding box functions.
 *
 *          + CameraPositionToBoundingBox: Calculates the distance from camera position to the world oriented bounding box of another node. This function is slower but more
 *            precise in comparison to the bounding sphere functions.
 *
 *          + ViewPlaneToBoundingSphere: Calculates the distance from the camera's plane perpendicular to the the look-at vector to the bounding sphere of another node. Faster but
 *            less precise in comparison to bounding box functions. Use this function if objects shall rather be sorted by distance to viewing plane than to position of camera. Typical
 *            use case is e.g. Billboards parallel to viewing plane.
 *
 *          + ViewPlaneToBoundingBox: Calculates the distance from the camera's plane perpendicular to the the look-at vector to the world oriented bounding box of another node.
 *            This function is slower but more precise in comparison to bounding sphere functions. Use this function if objects shall rather be sorted by distance to viewing plane than
 *            to position of camera. Typical use case is e.g. Billboards parallel to viewing plane.

 */
class CameraOrderCriterion: public RenderOrderCriterion
{
    FEATSTD_TYPEDEF_BASE(RenderOrderCriterion);

    public:
        /**
         *  DistanceFunction determines how the distance from camera to any node is calculated.
         */
        enum DistanceFunction{
            CameraPositionToBoundingSphere, ///< Distance from camera's position to bounding sphere of a node.
            CameraPositionToBoundingBox,    ///< Distance from camera's position to world oriented bounding box of a node.
            ViewPlaneToBoundingSphere,      ///< Distance from camera's viewing plane to bounding sphere of a node. Note: Plane is fix at origin of coordinate system.
            ViewPlaneToBoundingBox          ///< Distance from camera's viewing plane to world oriented bounding box of a node. Note: Plane is fix at origin of coordinate system.
        };

        /**
         *  Constructor of abstract class CameraOrderCriterion. Sets default distance function CameraPositionToBoundingSphere.
         */
        CameraOrderCriterion() : m_distFunc(CameraPositionToBoundingSphere) {}

        /**
         *  Destructor of abstract class CameraOrderCriterion.
         */
        virtual ~CameraOrderCriterion() override {}

        /**
         *  Selects distance function to be used to calculate the distance from the camera to a node.
         *  @param distFunc determines the distance function to be used.
         */
        void SetDistanceFunction(DistanceFunction distFunc) { m_distFunc = distFunc; }

        /**
         *  Retrieves the DistanceFunction used to calculate the distance from the camera to a node.
         *  @return DistanceFunction currently selected.
         */
        DistanceFunction GetDistanceFunction() const { return m_distFunc; }

        /**
         *  Overrides OrderCriterion::PrepareRenderOrderCriterionValue. Calculates distance to active camera, using the selected distance function.
         *  @param n Node to calculate distance to activeCamera from.
         */
        virtual void PrepareRenderOrderCriterionValue(Node& n) const override;

        /**
         *  Calculates the order criterion values (i.e. sorting keys) for the given nodes.
         *  @param nodeContainer     The container of nodes to compute the sorting keys for.
         *  @param sortingContainer  The container to store the sorting keys and node indices.
         *  @param nodeContainerSize The number of nodes in use in the nodeContainer. This can be
         *                           smaller than the actual size of the container
         *  @return True if all order criterion values could be prepared, False otherwise.
         */
        virtual bool PrepareRenderOrderCriterionValues(Internal::RenderOrderBin::NodeContainer& nodeContainer,
            Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const override;

        FEATSTD_RTTI_DECLARATION();

    private:
        DistanceFunction m_distFunc;    // Distance function used to determine the distance from camera to a node.
};


/**
 *  @brief  class DistanceToCameraOrderCriterion compares two Node's distances to the camera.
 *          A render order bin using DistanceToCameraOrderCriterion can sort its nodes from "Front to Back" according to
 *          selected distance function. (@see CameraOrderCriterion)
 */
class DistanceToCameraOrderCriterion: public CameraOrderCriterion
{
    FEATSTD_TYPEDEF_BASE(CameraOrderCriterion);

    public:
        /**
         *  Constructs a DistanceToCameraOrderCriterion object.
         */
        DistanceToCameraOrderCriterion() {}

        /**
         *  Destructs a DistanceToCameraOrderCriterion object.
         */
        virtual ~DistanceToCameraOrderCriterion() override {}

        /**
         *  Compares two Node's distances to camera.
         *  @param a First Node whose distance is compared to Node b.
         *  @param b Second Node whose distance is compared to Node a.
         *  @return true if first Node's distance to camera is less than second Node's distance to camera.
         */
        virtual bool IsBefore(const Node& a, const Node& b) const override;

        /**
         *  Sort the container of the RenderOrderBin
         *  @param sortingContainer  The container to be sorted.
         *  @param containerSize     The size of the container to be sorted. This can be smaller than
         *                           the actual size of the container to sort a subsection.
         */
        virtual void Sort(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const override;

        /**
         *  Determine if the render order has changed from the previous pass to the current one.
         *  If the order has not changed, the render order bin will use the result of the previous pass,
         *  instead of sorting the current pass.
         *  @param currentBin   The sorting bin of the previous pass.
         *  @param previousBin  The sorting bin of the current pass.
         *  @return True, if the bins are different or the order has changed. False, otherwise.
         */
        virtual bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
            const Internal::RenderOrderBin::SortingBin& previousBin) const override;

        FEATSTD_RTTI_DECLARATION();
};

/**
 *  @brief  class ReverseDistanceToCameraOrderCriterion compares two Node's distances to the camera.
 *          A render order bin using ReverseDistanceToCameraOrderCriterion can sort its Nodes from "Back to Front" according to
 *          selected distance function. (@see CameraOrderCriterion)
 */
class ReverseDistanceToCameraOrderCriterion: public CameraOrderCriterion
{
    FEATSTD_TYPEDEF_BASE(CameraOrderCriterion);

    public:
        /**
         *  Constructs a ReverseDistanceToCameraOrderCriterion object.
         */
        ReverseDistanceToCameraOrderCriterion() {}

        /**
         *  Destructs a ReverseDistanceToCameraOrderCriterion object.
         */
        virtual ~ReverseDistanceToCameraOrderCriterion() override {}

        /**
         *  Compares two Node's distances to camera.
         *  @param a First Node whose distance is compared to Node b.
         *  @param b Second Node whose distance is compared to Node a.
         *  @return true if first Node's distance to camera is greater than second Node's distance to camera.
         */
        virtual bool IsBefore(const Node& a, const Node& b) const override;

        /**
         *  Sort the container of the RenderOrderBin
         *  @param sortingContainer  The container to be sorted.
         *  @param containerSize     The size of the container to be sorted. This can be smaller than
         *                           the actual size of the container to sort a subsection.
         */
        virtual void Sort(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const override;

        /**
         *  Determine if the render order has changed from the previous pass to the current one.
         *  If the order has not changed, the render order bin will use the result of the previous pass,
         *  instead of sorting the current pass.
         *  @param currentBin   The sorting bin of the previous pass.
         *  @param previousBin  The sorting bin of the current pass.
         *  @return True, if the bins are different or the order has changed. False, otherwise.
         */
        virtual bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
            const Internal::RenderOrderBin::SortingBin& previousBin) const;

        FEATSTD_RTTI_DECLARATION();
};

 /** @} */ // end of RenderOrder3D

}

#endif // Candera_DistanceToCameraCriterion_h
