//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_BATCHORDERCRITERION_H
#define CANDERA_BATCHORDERCRITERION_H

#include <Candera/Engine3D/RenderOrder/DistanceToCameraOrderCriterion.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */

/**
 *  @brief  class BatchOrderCriterion produces a render order that contains batches of nodes with equal properties.
 *          Depending on the mode, these properties include Shader, RenderMode or Appearance.
 */
class BatchOrderCriterion : public RenderOrderCriterion
{
    FEATSTD_TYPEDEF_BASE(RenderOrderCriterion);

public:
    /**
     *  Mode determines what the order is being sorted by.
     */
    enum Mode {
        ShaderAndRenderMode, ///< Creates batches of shared Shaders and RenderModes using the same VertexBuffer
        Appearance           ///< Creates batches of shared Appearances using the same VertexBuffer
    };

    /**
     *  Constructor
     */
    BatchOrderCriterion() : m_mode(ShaderAndRenderMode) {}

    /**
     *  Destructor
     */
    virtual ~BatchOrderCriterion() override {}

    /**
     *  Selects the mode used to create batches.
     *  @param mode determines the mode to creates batches.
     */
    void SetMode(Mode mode) { m_mode = mode; }

    /**
     *  Retrieves the mode used to create batches.
     *  @return Mode  Returns the currently selected mode used to create batches.
     */
    Mode GetMode() const { return m_mode; }

    /**
     *  Empty implementation of legacy interface.
     */
    virtual bool IsBefore(const Node& a, const Node& b) const { FEATSTD_UNUSED(a); FEATSTD_UNUSED(b); return true; }

    /**
     *  Calculates the order criterion values (i.e. sorting keys) from the rank for the given nodes.
     *  @param nodeContainer     The container of nodes to compute the sorting keys for.
     *  @param sortingContainer  The container to store the sorting keys and node indices.
     *  @param nodeContainerSize The number of nodes in use in the nodeContainer. This can be
     *                           smaller than the actual size of the container
     *  @return True if all order criterion values could be prepared, False otherwise.
     */
    virtual bool PrepareRenderOrderCriterionValues(Internal::RenderOrderBin::NodeContainer& nodeContainer,
        Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const override;

    /**
     *  Sort the container of the RenderOrderBin
     *  @param sortingContainer  The container to be sorted.
     *  @param containerSize     The size of the container to be sorted. This can be smaller than
     *                           the actual size of the container to sort a subsection.
     */
    virtual void Sort(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const override;

    /**
     *  Determine if the render order has changed from the previous pass to the current one.
     *  If the order has not changed, the render order bin will use the result of the previous pass,
     *  instead of sorting the current pass.
     *  @param currentBin   The sorting bin of the previous pass.
     *  @param previousBin  The sorting bin of the current pass.
     *  @return True, if the bins are different or the order has changed. False, otherwise.
     */
    virtual bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
        const Internal::RenderOrderBin::SortingBin& previousBin) const;

    FEATSTD_RTTI_DECLARATION();

private:
    Mode m_mode;
};

/** @} */ // end of RenderOrder3D

}

#endif // Candera_RankOrderCriterion_h
