//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_AbstractRenderOrder_H)
#define CANDERA_AbstractRenderOrder_H

#include <Candera/Environment.h>

namespace Candera {

/**  @addtogroup RenderOrder3D
 *   @{
 */
 
//Forward declaration.
class Node;

/**
 *  @brief The class AbstractRenderOrder defines an interface how to add Nodes to and how to iterate and clear RenderOrderBins
 *  managed by any conrete RenderOrder implementation that collaborates with class Renderer.
 *  Nodes are assigned to render order bins; nodes assigned to an "earlier" bin are rendered before nodes assigned to a "later" bin.
 *  Sorting criterias can be applied to RenderOrderBins to define the order of Nodes within RenderOrderBins.
 */
class AbstractRenderOrder
{
    public:
        /**
         *  Constructor
         */
        AbstractRenderOrder() {}
        
        /**
         *  Destructor
         */
        virtual ~AbstractRenderOrder() {}

        /**
         *  Set iterator of this RenderOrder to first Node in the first bin.
         */
        virtual void IterationBegin() = 0;

        /**
         *  Returns if there are more Nodes to retrieve. This is true, as long as the last Node in the last bin
         *  has not been retrieved.
         *  @return True if there are more Nodes to retrieve, False otherwise.
         */
        virtual bool IterationHasMoreNodes() = 0;

        /**
         *  Move iterator to next Node in this render order.
         */
        virtual void IterationMoveToNextNode() = 0;

        /**
         *  Retrieve the Node the iterator is pointing to.
         *  @return Node the iterator is pointing to.
         */
        virtual Node* IterationGetNode() = 0;

        /**
         *  Assign the Node given to a bin defined in class Node. For more details see function Node::SetRenderOrderBinAssignment.
         *  @param node The node to be assigned to the bin.
         *  @return True if bin was assigned successfully, false if node is 0, node cannot be rendered or assigned bin doesn't exist.
         */
        virtual bool AssignNodeToBin(Node* node) = 0;

        /**
         *  Assign the Node given to a bin defined in class Node. For more details see function Node::SetRenderOrderBinAssignment.
         *  The given Node has to be verified first according to the needs of the concrete AbstractRenderOrder that implements this interface.
         *  This can be used to assign nodes from one AbstractRenderOrder to another without the need to re-verify each node.
         *  If the concrete AbstractRenderOrder implementation does not override this function, it simply re-routes to AssignNodeToBin.
         *  @param node The node to be assigned to the bin.
         *  @return True if bin was assigned successfully, false if node is 0, node cannot be rendered or assigned bin doesn't exist.
         */
        virtual bool AssignVerifiedNodeToBin(Node* node) { return AssignNodeToBin(node); }

        /**
         *  Sort all bins of this RenderOrder.
         */
        virtual void SortBins() = 0;

        /**
         *  Clear all bins applied to this render order.
         */
        virtual void Clear() = 0;

        /**
         *  Creates a clone of this instance.
         *  @return A pointer to the clone.
         */
        virtual AbstractRenderOrder* Clone() const = 0;

        /**
         *  Destroys a class instance
         */
        virtual void Dispose() = 0;

    protected:
        // Copying is allowed for Cloning.
        AbstractRenderOrder(const AbstractRenderOrder&) {}
    private:
        // Assignment forbidden until plausible reason provided
        AbstractRenderOrder& operator=(const AbstractRenderOrder&);        
};
 
 /**  @} */ // end of RenderOrder3D
 
} // namespace Candera

#endif    // CANDERA_AbstractRenderOrder_H

