//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VERTEXBUFFEROPTIMIZER_H)
#define CANDERA_VERTEXBUFFEROPTIMIZER_H

#include <Candera/Environment.h>

namespace Candera {
    namespace Internal {
/** @addtogroup Mathematics3D
 *  @{
 */
    
    /**
     * Vertex data used for caching.
     */
    struct VertexCacheData {
        bool isCached;
        UInt32* triangles;
        UInt32 triangleCount;
        VertexCacheData():isCached(false),triangles(0),triangleCount(0) {}
    };

    /**
     * Class for simulating a FIFO vertex cache.
     */
    class VertexCache {
    public:
        /**
         * Constructor.
         */
        VertexCache();

        /**
         * Destructor.
         */
        ~VertexCache();

        /**
         * Initialize the cache for a given capacity.
         * 
         * @param capacity Cache capacity.
         * @return true if initialization succeeds, false otherwise.
         */
        bool Init(UInt8 capacity);

        /**
         * Initialize the cache with another cache. The internal state of the other cache is copied.
         * 
         * @param vertexCache Vertex cache.
         * @return true if initialization succeeds, false otherwise.
         */
        bool Init(const VertexCache& vertexCache);

        /**
         * Push vertex in cache.
         * 
         * @param vertex Vertex to be pushed in cache.
         */
        void Push(UInt32 vertex);

        /**
         * Pop vertex out of the cache.
         * 
         * @return Popped vertex.
         */
        UInt32 Pop();

        /**
         * Return cache size.
         * 
         * @return Cache size.
         */
        UInt8 GetSize() const;

        /**
         * Get vertex from cache.
         * 
         * @param index Index of the vertex (must not exceed cache size).
         * @return Vertex found at given index in cache.
         */
        UInt32 GetVertex(UInt8 index) const;

        /**
         * Check if cache is full (i.e. cache size has reached its capacity).
         * 
         * @return true if cache is full, false otherwise.
         */
        bool IsFull() const { return m_capacity == m_size; }

    private:
        UInt32* m_indices;
        UInt8 m_head;
        UInt8 m_size;
        UInt8 m_capacity;
    };

 /** @} */ // end of Mathematics3D
    } // namespace Internal
} // namespace Candera

#endif  // CANDERA_VERTEXBUFFEROPTIMIZER_H
