//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexCache.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {
    namespace Internal {

        VertexCache::VertexCache()
            :m_indices(0), 
            m_head(0), 
            m_size(0), 
            m_capacity(0)
        {

        }

        VertexCache::~VertexCache()
        {
            if (m_indices != 0) {
                FEATSTD_DELETE_ARRAY(m_indices);
                m_indices = 0;
            }
        }

        bool VertexCache::Init(UInt8 capacity)
        {
            if (m_capacity != capacity) {
                m_capacity = capacity;
                if (m_indices != 0) {
                    FEATSTD_DELETE_ARRAY(m_indices);
                    m_indices = 0;
                }
                m_indices = FEATSTD_NEW_ARRAY(UInt32, capacity);
            }

            m_size = 0;
            m_head = 0;

            return (m_indices != 0);
        }

        bool VertexCache::Init(const VertexCache& vertexCache)
        {
            bool isInitialized = Init(vertexCache.m_capacity);

            if (isInitialized) {
                m_size = vertexCache.m_size;
                m_head = vertexCache.m_head;
                MemoryPlatform::Copy(m_indices, vertexCache.m_indices, sizeof(UInt32) * static_cast<SizeType>(m_capacity));
            }

            return isInitialized;
        }

        void VertexCache::Push(UInt32 vertex)
        {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::Internal::VertexCache::m_indices, CANDERA_LINT_REASON_OPERANDNOTNULL)
            if (m_capacity > 0) {
                m_indices[(m_head + m_size++) % m_capacity] = vertex;
            }
        }

        UInt32 VertexCache::Pop()
        {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::Internal::VertexCache::m_indices, CANDERA_LINT_REASON_OPERANDNOTNULL)
            UInt32 vertex = m_indices[m_head];
            if (m_capacity > 0) {
                m_head = (m_head + 1) % m_capacity;
                --m_size;
            }
            return vertex;
        }

        UInt8 VertexCache::GetSize() const
        {
            return m_size;
        }

        UInt32 VertexCache::GetVertex(UInt8 index) const
        {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::Internal::VertexCache::m_indices, CANDERA_LINT_REASON_OPERANDNOTNULL)
            if (m_capacity == 0) {
                return 0;
            }
            return m_indices[(m_head + index) % m_capacity];
        }

    } // namespace Internal
} // namespace Candera
