//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATH3D_H)
#define CANDERA_MATH3D_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
/** @addtogroup Mathematics3D
 *  @{
 */

    //Forward declarations to prevent reciprocal dependencies
    class Matrix3;
    class Matrix4;
    class Vector2;
    class Vector3;
    class Vector4;
    class Plane;
    class Rectangle;
    class Line;
    class Node;
    class Camera;
    class RenderTarget;
    class VertexBuffer;
    class Light;

    class Mesh;
    class LineList;

    /**
    * @brief The Math3D class for 3D transformation and helper functions.
    */
    class Math3D
    {
    public:

        /**
         * Describes target primitives to convert a vertex buffer to.
         */
        enum PrimitiveTypeConversion {
            LineConversion,                     ///< Create lines from triangles.
            PrimitiveConversionCount            ///< Used for iteration.
        };

         /**
         * Describes attributes to be generated in vertex buffer given.
         */
        enum AttributeGeneration {
            TangentAndBinormalGeneration,       ///< Create tangents and bi-normals.
            AttributeGenerationCount            ///< Used for iteration.
        };

         /**
         * Describes target precision qualifier for vertex buffer given.
         */
        enum PrecisionConversionTarget {
            FloatConversion,                    ///< Convert from half float to float.
            HalfFloatConversion,                ///< Convert from float to half float.
            PrecisionConversionCount            ///< Used for iteration.
        };

        /**
        * Sets the Matrix4 to a shadowed Matrix4. The shadowed Matrix
        * can be used to flatten geometry into a plane. (e.g. like a shadow)
        * @param [in,out]    matrix          The Matrix4 on which the operation will be performed. It is at first initialized to an identity Matrix4
        * and then the values are set.
        * @param [in]        light           The light that casts the shadows.
        * @param [in]        plane           The plane to which the resulting Matrix4 should transform.
        * @return   The Matrix4 that was handed to the function will also be returned with the new values set for easy use.
        */
        static Matrix4& SetShadow(Matrix4& matrix, const Light& light, const Plane& plane);

        /**
        * Sets the Matrix4 to a reflected Matrix4 around a given Plane.
        * @param [in,out]    matrix    The Matrix4 on which the operation will be performed. It is at first initialized to an identity Matrix4
        * and then the values are set.
        * @param [in]        plane     The reflection-axis plane.
        * @return   The Matrix4 that was handed to the function will also be returned with the new values set for easy use.
        */
        static Matrix4& SetReflection(Matrix4& matrix, const Plane& plane);
        
        /**
        * Sets the Matrix4 to a perspective field of view based
        * on the right-hand-side view of the 3D-coordinate system.
        * @param [in,out]    matrix      The Matrix4 on which the operation will be performed. It is at first initialized to an identity Matrix4
        * and then the values are set.
        * @param [in]        nearZ       The Z-value of the nearest viewplane.
        * @param [in]        farZ        The Z-value of the farthest viewplane.
        * @param [in]        fovY        The Y size of the view field in degrees.
        * @param [in]        aspectRatio The view width in pixels divided by the view height in pixels.
        * @return   The Matrix4 that was handed to the function will also be returned with the new values set for easy use.
        */
        static Matrix4& SetPerspectiveProjection(Matrix4& matrix, Float nearZ, Float farZ, Float fovY, Float aspectRatio);

        /**
        * Sets the Matrix4 to a orthographic projection view
        * @param [in,out]    matrix      The Matrix4 on which the operation will be performed. It is at first initialized to an identity Matrix4
        * and then the values are set.
        * @param [in]        nearZ       The Z-value of the nearest viewplane.
        * @param [in]        farZ        The Z-value of the farthest viewplane.
        * @param [in]        left        The left delimiter.
        * @param [in]        right       The right delimiter.
        * @param [in]        bottom      The bottom delimiter.
        * @param [in]        top         The top delimiter.
        * @return   The Matrix4 that was handed to the function will also be returned with the new values set for easy use.
        */
        static Matrix4& SetOrthographicProjection(Matrix4& matrix, Float nearZ, Float farZ, Float left, Float right, Float bottom, Float top);

        /**
        * Sets the Matrix4 to an offset, asymmetric perspective field of view based
        * on the right-hand-side view of the 3D-coordinate system.
        * @param [in,out]    matrix      The Matrix4 on which the operation will be performed. It is at first initialized to an identity Matrix4
        * and then the values are set.
        * @param [in]        nearZ               The Z-value of the nearest viewplane.
        * @param [in]        farZ                The Z-value of the farthest viewplane.
        * @param [in]        fovY                The Y size of the view field in degrees.
        * @param [in]        aspectRatio         The view width in pixels divided by the view height in pixels.
        * @param [in]        eyeSeparation       Eye separation used to calculate the offset for the left and right plane.
        * @param [in]        convergenceDistance Distance at which two stereoscopic projections shall be the same.
        *                                        Used to calculate the offset for left and right plane.
        * @return   The Matrix4 that was handed to the function will also be returned with the new values set for easy use.
        */
        static Matrix4& SetStereoProjection(Matrix4& matrix, Float nearZ, Float farZ, Float fovY, Float aspectRatio, Float eyeSeparation, Float convergenceDistance);

        /**
        * Sets the Matrix4 to a Frustum.
        * @param [in,out]    matrix      The Matrix4 on which the operation will be performed. It is at first initialized to an identity Matrix4
        * and then the values are set.
        * @param [in]        left        The left delimiter.
        * @param [in]        right       The right delimiter.
        * @param [in]        bottom      The bottom delimiter.
        * @param [in]        top     The top delimiter.
        * @param [in]        nearZ       The Z-value of the nearest viewplane.
        * @param [in]        farZ        The Z-value of the farthest viewplane.
        * @return   The Matrix4 that was handed to the function will also be returned with the new values set for easy use.
        */
        static Matrix4& SetFrustum(Matrix4& matrix, Float left, Float right, Float bottom, Float top, Float nearZ, Float farZ);

        /**
        * Tests if a given line intersects a given triangle.
        * @param triangleVertex1                Vertex 1 of the triangle to test.
        * @param triangleVertex2                Vertex 2 of the triangle to test.
        * @param triangleVertex3                Vertex 3 of the triangle to test.
        * @param line                           The line to test with.
        * @param hitPosition [out]              The hit position if there is an intersection.
        *                                       If there is no intersection this param stays unchanged.
        * @return                               Success if an intersection is found, otherwise false.
        */
        static bool TriangleLineIntersection(const Vector3& triangleVertex1,
                                             const Vector3& triangleVertex2,
                                             const Vector3& triangleVertex3,
                                             const Line& line,
                                             Vector3& hitPosition /*out*/);

        /**
        * Tests if the distance between two lines is shorter then a given distance (lines 'almost' intersect)
        * @param line1                          The first test line.
        * @param line2                          The second test line.
        * @param maxDistance                    The maximum distance accepted between the two lines
        * @param hitPosition [out]              The point on the second line where the distance is minimal.
        *                                       If the minimal distance is greater than maxDistance, this param stays unchanged.
        * @return                               Success if an intersection is found, otherwise false.
        */
        static bool LineLineIntersection(const Line& line1,
                                         const Line& line2,
                                         Float maxDistance,
                                         Vector3 &hitPosition /*out*/);

        /**
        * Tests if a given line intersects a given bounding sphere.
        * @param sphereCenter                   The sphere center.
        * @param sphereRadius                   The sphere radius.
        * @param line                           The line to test with.
        * @param hitPosition [out]              The hit position if there is an intersection.
        *                                       If there is no intersection this param stays unchanged.
        * @return                               Success if an intersection is found, otherwise false.
        */
        static bool SphereLineIntersection(const Line& line,
                                           const Vector3& sphereCenter,
                                           Float sphereRadius,
                                           Vector3& hitPosition /*out*/);

        /**
        * Creates a LineList object from a Mesh Object. All the mesh properties are assigend to the LineList object.
        * A new VertexBuffer is created for the LineList which contains 3 lines for each triangle in the original VertexBuffer.
        *
        * @param mesh                           The original Mesh.
        * @return                               The new LineList object if the creation of the LineList from the Mesh succeded.
        *                                       0 if the creation of the LineList failed.
        */
        static LineList* CreateLineListFromMesh(const Mesh* mesh);

        /** Calculates a Line which is cast from the Camera origin through
        * a specified point on the near plane to the far plane.
        * The resulting Line starts at the near plane and ends on the far plane.
        * x and y are absolute screen space coordinates starting from top-left.
        * @param camera     The corresponding Camera.
        * @param x          The x value of the point on the near plane as an absolute screen space coordinate in pixels.
        * @param y          The y value of the point on the near plane as an absolute screen space coordinate in pixels.
        * @param line [out] The calculated line as out parameter.
        * @return           true if everything went ok, otherwise false.
        */
        static bool CalculatePickLine(const Camera& camera, Float x, Float y, Line& line /*[out]*/);

        /**
         * Common algorithm to test if geometry of a node intersects with a pick in screen coordinates.
         * @param node               The node to test if a pick intersects the geometry.
         * @param camera             The view camera. (i.e. the current camera)
         * @param x                  The x value as a screen coordinate in pixels starting from top-left.
         * @param y                  The y value as a screen coordinate in pixels starting from top-left.
         * @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
         *                           If no intersection is found then this param stays unchanged.
         * @return                   True if an intersection is found with the given node, otherwise false.
         */
        static bool IsPickIntersectingGeometry(const Node& node, const Camera& camera, Int x, Int y, Float& distance /*out*/);

        /**
        * Converts absolute screen space coordinates through the inverse Viewport transformation,
        * to a right handed 3D space coordinates.
        * @param camera                         The corresponding Camera.
        * @param screenspaceCoordinates         x and y are absolute screen space coordinates in pixels with origin top-left.
        *                                       In case of x and y a cast from an integer type to a Float is necessary.
        *                                       A z value of 0.0 means z is on the near plane of the Camera.
        *                                       A z value of 1.0 means z is on the far plane of the Camera.
        * @param unprojectedCoordinates [out]   The calculated 3D coordinates as out parameter.
        * @return                               Success if everything went ok, otherwise false.
        */
        static bool ConvertViewportCoordinates(const Camera& camera,
                                               const Vector3& screenspaceCoordinates,
                                               Vector3& unprojectedCoordinates /*[out]*/);

        /**
        * Computes axis-aligned bounding box (AABB) in local coordinate space.
        * @param vertexBuffer is an in parameter describing the vertex geometry
        * @param minBounds    is an out parameter describing the returned lower-left vertex of the bounding box.
        * @param maxBounds    is an out parameter describing the returned upper-right vertex of the bounding box.
        * @param index        Index to which Position element inside the vertex buffer shall be taken, default 0. Multiple positions
        *              can occur for example if morphing meshes are used.
        * @return true if bounding box was computed successfully and false if not because of missing geometry.
        */
        static bool ComputeBoundingBox(const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer,
                                                Vector3& minBounds,
                                                Vector3& maxBounds,
                                                UInt8 index = 0);

       /**
        * Computes the axis-aligned bounding box (AABB) of another axis-aligned box
        * in a provided space.
        * @param srcMinBounds lower-left vertex of the source box.
        * @param srcMaxBounds upper-right vertex of the source box.
        * @param transform    space transformation matrix.
        * @param minBounds    is an out parameter describing the returned lower-left vertex of the bounding box.
        * @param maxBounds    is an out parameter describing the returned upper-right vertex of the bounding box.
        */
        static void TransformBoundingBox(const Vector3& srcMinBounds,
                                         const Vector3& srcMaxBounds,
                                         const Matrix4& transform,
                                         Vector3& minBounds,
                                         Vector3& maxBounds);

       /**
        * Transforms a point in world space into a point in viewport space.
        * @param point            vertex of the point which has to be transformed.
        * @param camera           defines the view-projection matrix on which the point is projected.
        * @param transformedPoint [out] normalized point in viewport space.
        */
        static void TransformPointToViewport(const Vector3& point,
            const Camera* camera,
            Vector3& transformedPoint);

       /**
        * Transforms a bounding box in world space into an axis-aligned bounding rectangle in viewport space.
        * @param bbBounds          8 vertices corresponding to the eight corners of the bounding box.
        * @param camera            defines the view-projection matrix on which the bounding box is projected.
        * @param boundingRectangle [out] normalized bounding rectangle in viewport space.
        */
        static void TransformBoundingBoxToViewport(const Vector3* bbBounds,
            const Camera* camera,
            Rectangle& boundingRectangle);

        /**
        * Transforms a point in world space into a point surface space
        * @param point            vertex of the world point.
        * @param camera           defines the view-projection matrix and RenderTarget on which the point is projected.
        * @param transformedPoint [out] point in screen space.
        * @return false if no render target is assigned to camera, true otherwise
        */
        static bool TransformPointToSurface(const Vector3 point,
            const Camera* camera,
            Vector2& transformedPoint);

       /**
        * Transforms a bounding box in world space into an axis-aligned bounding rectangle in surface space
        * @param bbBounds          8 vertices corresponding to the eight corners of the world oriented bounding box.
        * @param camera            defines the view-projection matrix and RenderTarget on which the bounding box is projected.
        * @param boundingRectangle [out] bounding rectangle in screen space.
        * @return false if no render target is assigned to camera, true otherwise
        */
        static bool TransformBoundingBoxToSurface(const Vector3* bbBounds,
            const Camera* camera,
            Rectangle& boundingRectangle);


       /**
        * Transforms a point in world space into a point in screen space
        * @param point vertex of the world point.
        * @param camera defines the view-projection matrix and RenderTarget on which the point is projected.
        * @param transformedPoint [out] point in screen space.
        * @return false if no render target is assigned to camera or an invalid window is assigned to the render target, true otherwise
        */
        static bool TransformPointToScreen(const Vector3 point,
            const Camera* camera,
            Vector2& transformedPoint);

       /**
        * Transforms a bounding box in world space into an axis-aligned bounding rectangle in screen space
        * @param bbBounds          8 vertices corresponding to the eight corners of the world oriented bounding box.
        * @param camera            defines the view-projection matrix and RenderTarget on which the bounding box is projected.
        * @param boundingRectangle [out] bounding rectangle in screen space.
        * @return false if no render target is assigned to camera or an invalid window is assigned to the render target, true otherwise
        */
        static bool TransformBoundingBoxToScreen(const Vector3* bbBounds,
            const Camera* camera,
            Rectangle& boundingRectangle);

        /**
        * Computes euler angles from the given rotation matrix.
        * @param rotationMatrix     The rotation matrix that is to be converted.
        * @param[out] eulerAngles   The resulting euler angles (pitch, yaw, roll) in degrees.
        */
        static void ConvertRotationMatrixToEulerAngles(const Matrix4& rotationMatrix, Vector3& eulerAngles);

        /**
         * Sort the index array for minimum vertex cache misses during vertex processing.
         * 
         * It is important that the vertexCacheSize parameter matches as much as possible the actual target vertex cache size.
         * Giving a smaller value will result in a less optimized index array, while giving a greater value introduces the 
         * possibility of cache trashing. If the exact vertex cache size is unknown, it is better to underestimate the size of it.
         * 
         * @param vertexBuffer
         * @param vertexCacheSize Size of the VertexCache.
         * @return Number of cache misses.
         */
        static UInt32 OptimizeVertexCache(MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer, UInt8 vertexCacheSize);

#ifndef FEATSTD_FIXED_POINT_ARITHMETIC
        /**
         * Converts a 32 bit float value into a 16-bit half float value.
         * @param floatValue The float value to convert.
         * @return The 16-bit float value, stored inside a 16-bit integer.
         */
        static UInt16 ConvertFloatToHalfFloat(Float floatValue);

        /**
         * Converts a 16 bit half float value into a 32-bit float value.
         * @param halfFloatValue The half float value to convert.
         * @return The 32-bit float value.
         */
        static Float ConvertHalfFloatToFloat(UInt16 halfFloatValue);
#endif
        
        /**
         * Creates a vertex buffer containing the same vertices than the given vertex buffer, but with indices removed.
         * Duplicates vertices if necessary.
         * @param indexedVb The indexed vertex buffer to remove the index from.
         * @return The unindexed vertex buffer, which will contain duplicated vertices. 0 if indexedVb was not indexed or any other error case.
         */
        static MemoryManagement::SharedPointer<VertexBuffer> CreateVertexBufferWithIndexRemoved(const MemoryManagement::SharedPointer<VertexBuffer> indexedVb);

        /**
         * Creates a vertex buffer containing the same vertices than the given vertex buffer, but with indices added. 
         * Duplicated vertices will be merged.
         * @param nonIndexedVb The nonindexed vertex buffer to add the index to.
         * @param indexType Type of indices in the index buffer. This is just a hint: if 8bit index is specified but there are more than 256 vertices,
         *  16bit indices will be created.
         * @return The indexed vertex buffer. 0 if indexedVb was already indexed or any other error case.
         */
        static MemoryManagement::SharedPointer<VertexBuffer> CreateVertexBufferWithIndexGenerated(const MemoryManagement::SharedPointer<VertexBuffer> nonIndexedVb, VertexGeometry::BufferType indexType = VertexGeometry::IndexedArrayBuffer);

        /**
         * Creates a vertex buffer with vertices of a certain primitive from another vertex buffer.
         * @param vb The source vertex buffer to create the new primitive vertex buffer.
         * @param target The target primitive to convert to.
         *               Currently the following conversions are supported:
         *               <ul>
         *                <li>
         *                 Lines: Converts a vertex buffer of any triangle type (indexed or not) to indexed lines.#
         *                        Conversions from other line types or points are yet not supported.
         *               </li>
         *               <li>
         *               NoConversion: No conversion is done, can be used as delimiter to iterate over all conversions.
         *               </li>
         *               </ul>
         * @return A vertex buffer holding primtives of the specified target, computed from a given vertex buffer.
         *         0 in errornous conditions, if vb == 0, vb->GetVertexGeometry() == 0 or if the passed vertex
         *         buffer doesn't hold appropriate primitives.
         */
        static MemoryManagement::SharedPointer<VertexBuffer> CreateVertexBufferByPrimitiveTypeConversion(const MemoryManagement::SharedPointer<VertexBuffer>& vb, PrimitiveTypeConversion target);

        /**
         * Creates a vertex buffer that additionally holds the specified attributes, which was generated.
         * @param vb Original vertex buffer. Based on the given data a new one with calculated attributes will be created.
         * @param attributes Specifies which attributes are created, in most cases has requirements in the source vertex buffer.
         *                   Currently supported:
         *                   <ul>
         *                   <li> 
         *                     TangentsAndBinormals: Creates tangents and binormals, used for e.g. normal mapping.
         *                     Tangents and binormals are computed out of the vertices position, normal and texture coordinates, if they don't exist
         *                     then 0 is returned,
         *                     If tangents and binormals already exist in the vertex buffer then again 0 is returned. If either one exists, it will be
         *                     overwritten, just to ensure that calculated tangents and binormals are consistent in terms of their handedness.
         *                     Tangents and binormals of shared vertices are accumulated and then normalized.
         *                     If multiple attributes of one kind exist in the vertex buffer, then all compatible tuples get updated with tangents and binormals.
         *                     Positions, normals, tangents and bitangents have to contain at least 3 floats (x, y, z). Texture coordinates at least 2 floats (u,v).
         *                    </li>
         *                    <li>
         *                      NoGeneration: No attribute generation is done, can be used as delimiter to iterate over all attributes to generate.
         *                    </li>
         *                   </ul>
         * @return  A pointer to a vertex buffer containing the specified attributes, if the requirements to calculate them are fulfilled.
         *          Otherwise 0 is returned.
         */
        static MemoryManagement::SharedPointer<VertexBuffer> CreateVertexBufferWithAttributeGenerated(const MemoryManagement::SharedPointer<VertexBuffer>& vb, AttributeGeneration attributes);

        /**
         * Creates a vertex buffer with the specified attribute removed.
         * @param vb The source vertex buffer to remove the attribute from.
         * @param usage The attribute to remove. See Candera::VertexGeometry.
         * @param usageIndex The index of the attribute to remove. See Candera::VertexGeometry.
         * @return A copy of the previous vertex buffer, with the specified attribute removed. 0 in erroneous conditions.
         */
        static MemoryManagement::SharedPointer<VertexBuffer> CreateVertexBufferWithAttributeRemoved(const MemoryManagement::SharedPointer<VertexBuffer>& vb, VertexGeometry::VertexUsage usage, UInt8 usageIndex);
        
         /**
         * Creates a vertex buffer with the specified attributes precision changed.
         * @param vb The source vertex buffer to convert the attribute.
         * @param precision The target precision, FloatConversion to convert from half float to float, HalfFloatConversion to
         *                  convert from float to half float and NoPrecisionConversion just as delimiter for iterations.
         * @param usage The attribute to change the precision. See Candera::VertexGeometry.
         * @param usageIndex The index of the attribute to change the precision. See Candera::VertexGeometry.
         * @return A copy of the previous vertex buffer, with the specified attributes precision changed. 0 in erroneous conditions.
         */
        static MemoryManagement::SharedPointer<VertexBuffer> CreateVertexBufferByPrecisionConversion(const MemoryManagement::SharedPointer<VertexBuffer>& vb, PrecisionConversionTarget precision, VertexGeometry::VertexUsage usage, UInt8 usageIndex);

        /**
         * Normalize the given rectangle from the render target space.
         * @param renderTarget The render target for which the rectangle has to be normalized.
         * @param renderTargetSpaceRectangle The rectangle in the render target space.
         * @param normalizedRectangle Contains the normalized rectangle after the call.
         */
        static void NormalizeRectangle(const RenderTarget& renderTarget, const Rectangle& renderTargetSpaceRectangle, Rectangle& normalizedRectangle);
};

 /** @} */ // end of Mathematics3D
} // namespace Candera

#endif  // CANDERA_MATH3D_H
