//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Math3D.h"
#include <Candera/Engine3D/Mathematics/VertexCache.h>
#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/Engine3D/Core/LineList.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/Engine3D/Core/VertexGeometryBuilder.h>
#include <Candera/Engine3D/Core/VertexGeometryAccessors.h>
#include <Candera/Engine3D/Core/Light.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Line.h>
#include <Candera/System/Mathematics/Plane.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/Window.h>
#include <FeatStd/Util/FeatLimits.h>
#include <FeatStd/MemoryManagement/Heap.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;
    using namespace Candera::Internal;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    static MemoryManagement::SharedPointer<VertexBuffer> CreateLinesFromTrianglesInternal(const MemoryManagement::SharedPointer<VertexBuffer>& vb);
    MemoryManagement::SharedPointer<VertexBuffer> CreateTangentsAndBinormalsFromVertexBufferInternal(MemoryManagement::SharedPointer<VertexBuffer> vb);

    Matrix4& Math3D::SetShadow(Matrix4& matrix, const Light& light, const Plane& plane)
    {
        if (light.GetType() != Light::Ambient) {
            Plane normPlane(plane);
            normPlane.Normalize();
            Vector4 lightVector;

            if ((light.GetType() == Light::Point) || (light.GetType() == Light::Spot)) {
                lightVector = Vector4(light.GetWorldPosition());
            }
            else if (light.GetType() == Light::Directional) {
                lightVector = -Vector4(light.GetWorldDirection());
                lightVector.SetW(0.0F);
            }
            else {
                //do nothing
            }

            Float dot = normPlane.GetDotProduct(lightVector);
            // Calculate row 0
            matrix.Set(0, 0, dot - (normPlane.GetDirection().GetX() * lightVector.GetX()));
            matrix.Set(0, 1, -normPlane.GetDirection().GetX() * lightVector.GetY());
            matrix.Set(0, 2, -normPlane.GetDirection().GetX() * lightVector.GetZ());
            matrix.Set(0, 3, -normPlane.GetDirection().GetX() * lightVector.GetW());

            // Calculate row 1
            matrix.Set(1, 0, -normPlane.GetDirection().GetY() * lightVector.GetX());
            matrix.Set(1, 1, dot - (normPlane.GetDirection().GetY() * lightVector.GetY()));
            matrix.Set(1, 2, -normPlane.GetDirection().GetY() * lightVector.GetZ());
            matrix.Set(1, 3, -normPlane.GetDirection().GetY() * lightVector.GetW());

            // Calculate row 2
            matrix.Set(2, 0, -normPlane.GetDirection().GetZ() * lightVector.GetX());
            matrix.Set(2, 1, -normPlane.GetDirection().GetZ() * lightVector.GetY());
            matrix.Set(2, 2, dot - (normPlane.GetDirection().GetZ() * lightVector.GetZ()));
            matrix.Set(2, 3, -normPlane.GetDirection().GetZ() * lightVector.GetW());

            // Calculate row 3
            matrix.Set(3, 0, -normPlane.GetDistance() * lightVector.GetX());
            matrix.Set(3, 1, -normPlane.GetDistance() * lightVector.GetY());
            matrix.Set(3, 2, -normPlane.GetDistance() * lightVector.GetZ());
            matrix.Set(3, 3, dot - (normPlane.GetDistance() * lightVector.GetW()));
        }
        else {
            matrix.SetIdentity();
        }

        return matrix;
    }

    Matrix4& Math3D::SetReflection(Matrix4& matrix, const Plane& plane)
    {
        matrix.SetIdentity();
        Plane normPlane(plane);
        normPlane.Normalize();

        // Calculate row 0
        matrix.Set(0, 0, 1.0F - (2.0F * normPlane.GetDirection().GetX() * normPlane.GetDirection().GetX()));
        matrix.Set(0, 1, -2.0F * normPlane.GetDirection().GetX() * normPlane.GetDirection().GetY());
        matrix.Set(0, 2, -2.0F * normPlane.GetDirection().GetX() * normPlane.GetDirection().GetZ());

        // Calculate row 1
        matrix.Set(1, 0, -2.0F * normPlane.GetDirection().GetX() * normPlane.GetDirection().GetY());
        matrix.Set(1, 1, 1.0F - (2.0F * normPlane.GetDirection().GetY() * normPlane.GetDirection().GetY()));
        matrix.Set(1, 2, -2.0F * normPlane.GetDirection().GetY() * normPlane.GetDirection().GetZ());

        // Calculate row 2
        matrix.Set(2, 0, -2.0F * normPlane.GetDirection().GetZ() * normPlane.GetDirection().GetX());
        matrix.Set(2, 1, -2.0F * normPlane.GetDirection().GetZ() * normPlane.GetDirection().GetY());
        matrix.Set(2, 2, 1.0F - (2.0F * normPlane.GetDirection().GetZ() * normPlane.GetDirection().GetZ()));

        // Calculate row 3
        matrix.Set(3, 0, -2.0F * normPlane.GetDistance() * normPlane.GetDirection().GetX());
        matrix.Set(3, 1, -2.0F * normPlane.GetDistance() * normPlane.GetDirection().GetY());
        matrix.Set(3, 2, -2.0F * normPlane.GetDistance() * normPlane.GetDirection().GetZ());
        return matrix;
    }

    Matrix4& Math3D::SetPerspectiveProjection(Matrix4& matrix, Float nearZ, Float farZ, Float fovY, Float aspectRatio)
    {
        matrix.SetIdentity();
        Float frustumW;
        Float frustumH;
        frustumH = Math::Tangent((fovY / 360.0F) * Math::Pi()) * nearZ;
        frustumW = frustumH * aspectRatio;

        return SetFrustum(matrix, -frustumW, frustumW, -frustumH, frustumH, nearZ, farZ);
    }

    Matrix4& Math3D::SetOrthographicProjection(Matrix4& matrix, Float nearZ, Float farZ, Float left, Float right, Float bottom, Float top)
    {
        matrix.SetIdentity();
        matrix.Set(0, 0, 2.0F / (right - left));
        matrix.Set(1, 1, 2.0F / (top - bottom));
        // this is a sign change (compared to the "usual" matrix) that corresponds to looking down negative Z, but having nearZ and farZ as positive values
        matrix.Set(2, 2, -2.0F / (farZ - nearZ));
        matrix.Set(0, 3, -(right + left) / (right - left));
        matrix.Set(1, 3, -(top + bottom) / (top - bottom));
        matrix.Set(2, 3, -(farZ + nearZ) / (farZ - nearZ));
        matrix.Transpose();
        return matrix;
    }

    Matrix4& Math3D::SetStereoProjection(Matrix4& matrix, Float nearZ, Float farZ, Float fovY, Float aspectRatio, Float eyeSeparation, Float convergenceDistance)
    {
        matrix.SetIdentity();
        Float frustumW;
        Float frustumH;
        frustumH = Math::Tangent((fovY / 360.0F) * Math::Pi()) * nearZ;
        frustumW = frustumH * aspectRatio;
        Float frustumXOffset = eyeSeparation * (nearZ / convergenceDistance);

        return SetFrustum(matrix, -frustumW - frustumXOffset, frustumW - frustumXOffset, -frustumH, frustumH, nearZ, farZ);
    }

    Matrix4& Math3D::SetFrustum(Matrix4& matrix, Float left, Float right, Float bottom, Float top, Float nearZ, Float farZ)
    {
        Float deltaX = right - left;
        Float deltaY = top - bottom;
        Float deltaZ = farZ - nearZ;

        if ((nearZ <= 0.0F) || (farZ <= 0.0F) || (deltaX <= 0.0F) || (deltaY <= 0.0F) || (deltaZ <= 0.0F)) {
            return matrix;
        }

        matrix.Set(0, 0, (2.0F * nearZ) / deltaX);
        matrix.Set(0, 1, 0.0F);
        matrix.Set(0, 2, 0.0F);
        matrix.Set(0, 3, 0.0F);

        matrix.Set(1, 1, (2.0F * nearZ) / deltaY);
        matrix.Set(1, 0, 0.0F);
        matrix.Set(1, 2, 0.0F);
        matrix.Set(1, 3, 0.0F);

        matrix.Set(2, 0, (right + left) / deltaX);
        matrix.Set(2, 1, (top + bottom) / deltaY);
        matrix.Set(2, 2, -(nearZ + farZ) / deltaZ);
        matrix.Set(2, 3, -1.0F);

        matrix.Set(3, 2, (-2.0F * nearZ * farZ) / deltaZ);
        matrix.Set(3, 0, 0.0F);
        matrix.Set(3, 1, 0.0F);
        matrix.Set(3, 3, 0.0F);
        return matrix;
    }

    bool Math3D::TriangleLineIntersection(const Vector3& triangleVertex1,
                                          const Vector3& triangleVertex2,
                                          const Vector3& triangleVertex3,
                                          const Line& line,
                                          Vector3& hitPosition /*out*/)
    {
        Vector3 normal;
        Vector3 intersectPos;

        //find triangle normal
        normal = (triangleVertex2 - triangleVertex1).GetCrossProduct(triangleVertex3 - triangleVertex1);
        if (normal.Normalize() <= 0.F) {
            return false; // Degenerated triangle.
        }

        //find distance from rayStart and rayEnd to the plane defined by the triangle
        Float distance1 = (line.GetStart() - triangleVertex1).GetDotProduct(normal);
        Float distance2 = (line.GetEnd() - triangleVertex1).GetDotProduct(normal);
        if ((distance1 * distance2) >= 0.0F) {
            return false;  // line doesn't cross the triangle.
        }
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING);
        if (distance1 == distance2) {
            return false;// line and plane are parallel
        }

        // find point on the line that intersects with the plane
        intersectPos = line.GetStart() + ((line.GetEnd() - line.GetStart()) * (-distance1 / (distance2 - distance1)));

        // find if the intersection point lies inside the triangle by testing it against all edges
        Vector3 vTest;
        vTest = normal.GetCrossProduct(triangleVertex2 - triangleVertex1);
        if (vTest.GetDotProduct(intersectPos - triangleVertex1) < 0.0F) {
            return false;
        }
        vTest = normal.GetCrossProduct(triangleVertex3 - triangleVertex2);
        if (vTest.GetDotProduct(intersectPos - triangleVertex2) < 0.0F) {
            return false;
        }
        vTest = normal.GetCrossProduct(triangleVertex1 - triangleVertex3);
        if (vTest.GetDotProduct(intersectPos - triangleVertex1) < 0.0F) {
            return false;
        }

        hitPosition = intersectPos;
        return true;
    }

    bool Math3D::LineLineIntersection(const Line& line1,
                                      const Line& line2,
                                      Float maxDistance,
                                      Vector3& hitPosition /*out*/)
    {
        Vector3 d1 = line1.GetEnd() - line1.GetStart(); // d1(s): d1*s + line1.GetStart() = 0
        Vector3 d2 = line2.GetEnd() - line2.GetStart(); // d2(t): d2*t + line2.GetStart() = 0
        Vector3 d0 = line1.GetStart() - line2.GetStart(); // d(s,t): d1*s - d2*t + d = 0

        // the shortest segment between L1 and L2 is perpendicular on both L1 and L2
        // we have to find sc and tc where: d1 * d = 0 and d2 * d = 0
        // d1 * d1 * s - d1 * d2 * t = -d1 * d
        // d1 * d2 * s - d2 * d2 * t = -d2 * d
        // => sc = (d1*d2 * d2*d - d2*d2 * d1*d) / (d1*d1 * d2*d2 - d1*d2 * d1*d2)
        // => tc = (d1*d1 * d2*d - d1*d2 * d1*d) / (d1*d1 * d2*d2 - d1*d2 * d1*d2)
        Float a = d1.GetSquaredLength();
        Float b = d1.GetDotProduct(d2);
        Float c = d2.GetSquaredLength();
        Float d = d1.GetDotProduct(d0);
        Float e = d2.GetDotProduct(d0);

        // => sc = (b*e - c*d) / (a*c - b*b)
        // => tc = (a*e - b*d) / (a*c - b*b)

        Float D = (a * c) - (b * b); // D= d1*d1 * d2*d2 - d1*d2 * d1*d2 =
        //  = |d1|^2 * |d2|^2 - (|d1|*|d2|*cos(teta))^2 = (|d1|*|d2|*sin(teta))^2
        Float s1 = 0.0F;
        Float s2 = D;
        Float t1 = 0.0F;
        Float t2 = D;

        // compute the line parameters of the two closest points
        if (D < Math::EpsilonFloat3D()) { // sin(teta) ~= 0 => line almost parallel
            s1 = 0.0F;
            s2 = 1.0F;
            t1 = e;
            t2 = c;
        }
        else { // get the closest points on the infinite lines
            s1 = (b * e) - (c * d);
            t1 = (a * e) - (b * d);

            // check if point on first line is in the interior of the segment
            if (s1 < 0.0) { // point on first line is outside of segment
                s1 = 0.0F; // the actual point on segment where the distance is minimal is the extremity
                t1 = e; // sc = 0, tc = e/c
                t2 = c;
            }
            else if (s1 > s2) { // point on first line is outside of segment
                s1 = s2; // the actual point on segment where the distance is minimal is the extremity
                t1 = e + b; // sc = 1, tc = (e+b)/c
                t2 = c;
            }
            else {
                //do nothing
            }
        }

        // check if point on second line is in the interior of the segment
        if (t1 < 0.0) { // point on second line is outside of segment
            t1 = 0.0F; // the actual point on segment where the distance is minimal is the extremity
            // recompute sc
            if (-d < 0.0F) {
                s1 = 0.0F; // the actual point on first segment where the distance is minimal is the extremity
            }
            else if (-d > a) {
                s1 = s2; // the actual point on first segment where the distance is minimal is the extremity
            }
            else {
                s1 = -d;
                s2 = a;
            }
        }
        else if (t1 > t2) { // point on second line is outside of segment
            t1 = t2; // the actual point on segment where the distance is minimal is the extremity
            // recompute sc for this edge
            if ((-d + b) < 0.0F) {
                s1 = 0.0F; // the actual point on first segment where the distance is minimal is the extremity
            }
            else if ((-d + b) > a) {
                s1 = s2; // the actual point on first segment where the distance is minimal is the extremity
            }
            else {
                s1 = (-d + b);
                s2 = a;
            }
        }
        else {
            //do nothing
        }

        Float sc = (Math::Absolute(s1) < Math::EpsilonFloat3D()) ? 0.0F : (s1 / s2);
        Float tc = (Math::Absolute(t1) < Math::EpsilonFloat3D()) ? 0.0F : (t1 / t2);

        // get the difference of the two closest points
        Vector3 dFinal = (d0 + (d1 * sc)) - (d2 * tc);

        bool hit = false;
        if (dFinal.GetLength() < (maxDistance)) {
            hitPosition = d2 * tc;
            hit = true;
        }
        return hit;
    }

    bool Math3D::SphereLineIntersection(const Line& line,
                                        const Vector3& sphereCenter,
                                        Float sphereRadius,
                                        Vector3& hitPosition /*out*/)
    {
        Vector3 lineDirection = line.GetEnd() - line.GetStart();
        if (lineDirection.Normalize() <= 0.F) {
            return false; // Degenerate line.
        }

        const Vector3 sphereToLineStart = sphereCenter - line.GetStart();
        Float squaredLength = sphereToLineStart.GetSquaredLength();
        Float angleCosinus = sphereToLineStart.GetDotProduct(lineDirection);
        Float squaredRadius = sphereRadius * sphereRadius;

        //sphere behind line start
        if ((squaredLength > squaredRadius) && (angleCosinus < 0.0F)) {
            return false;
        }

        //squared distance from the sphere center to projection
        Float projectionDistanceSquared = (squaredLength - (angleCosinus * angleCosinus));
        //if greater than radius -> no intersection
        if (projectionDistanceSquared > squaredRadius) {
            return false;
        }

        //distance along the line to the intersection point
        Float factor = 0.0F;
        Float squaredDistance = Math::SquareRoot(squaredRadius - projectionDistanceSquared);
        if (squaredLength > squaredRadius) {
            factor = angleCosinus - squaredDistance;
        }
        else {
            factor = angleCosinus + squaredDistance;
        }

        const Vector3 intersection = lineDirection * factor;
        hitPosition = line.GetStart() + intersection;

        return true;
    }

    LineList* Math3D::CreateLineListFromMesh(const Mesh* mesh)
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_INSTANCESOBTAINABLE);

        LineList* lineList = LineList::Create();
        if (lineList == 0) {
            return 0;
        }
        static_cast<Transformable&>(*lineList) = *mesh;
        lineList->SetRenderingEnabled(mesh->IsRenderingEnabled());
        lineList->SetAppearance(mesh->GetAppearance());
        lineList->SetAlphaValue(mesh->GetAlphaValue());
        lineList->SetCenter(mesh->GetCenter());
        lineList->SetRadius(mesh->GetRadius());
        Vector3 minBox;
        Vector3 maxBox;
        mesh->GetBoundingBox(minBox, maxBox);
        lineList->SetBoundingBox(minBox, maxBox);
        lineList->SetRenderOrderBinAssignment(mesh->GetRenderOrderBinAssignment());
        lineList->SetRenderOrderRank(mesh->GetRenderOrderRank());
        lineList->SetRenderBenchmark(mesh->GetRenderBenchmark());
        lineList->SetIntersectionTestEnabled(mesh->IsIntersectionTestEnabled());
        lineList->SetScopeMask(mesh->GetScopeMask());

        lineList->SetVertexBuffer(CreateLinesFromTrianglesInternal(mesh->GetVertexBuffer()));
        lineList->SetLineType(LineList::Lines);
        lineList->SetWidth(1.0F);

        return lineList;
    }

    MemoryManagement::SharedPointer<VertexBuffer> CreateTangentsAndBinormalsFromVertexBufferInternal(MemoryManagement::SharedPointer<VertexBuffer> vb)
    {
        using FeatStd::Internal::PointerAdd;
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_INSTANCESOBTAINABLE);
        SharedPointer<VertexBuffer> nullBuffer(0);

        if (vb == 0) {
            return nullBuffer;
        }

        if (vb->GetVertexGeometry() == 0) {
            return nullBuffer;
        }

        const VertexGeometry* vg = vb->GetVertexGeometry();

        Vector<UInt8> consideredUsageIndices; //Only usage indices, that contain the needed attributes (position, normal, texCoord) but
        //don't contain tangents and binormals are considered for further calculations.

        ElementFormatAccessor elementFormatAccessor(*vg);
        bool additionsSuccessfull = true;
        //Collect usage indices where tangents and binormals have to be created.
        for (UInt8 i = 0; (i <= vg->GetMaxUsageIndex()) && additionsSuccessfull; i++) {
            const VertexGeometry::VertexElementFormat* element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::Position, i);
            bool position = (element != 0) && (VertexGeometry::GetVertexElementSize(element->Type) >= VertexGeometry::GetVertexElementSize(VertexGeometry::Float32_3));

            element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::Normal, i);
            bool normal = (element != 0) && (VertexGeometry::GetVertexElementSize(element->Type) >= VertexGeometry::GetVertexElementSize(VertexGeometry::Float32_3));

            element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::TextureCoordinate, i);
            bool texCoord = (element != 0) && (VertexGeometry::GetVertexElementSize(element->Type) >= VertexGeometry::GetVertexElementSize(VertexGeometry::Float32_2));

            element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::Tangent, i);
            bool tangent = (element != 0) && (VertexGeometry::GetVertexElementSize(element->Type) >= VertexGeometry::GetVertexElementSize(VertexGeometry::Float32_3));

            element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::BiNormal, i);
            bool binormal = (element != 0) && (VertexGeometry::GetVertexElementSize(element->Type) >= VertexGeometry::GetVertexElementSize(VertexGeometry::Float32_3));

            if ((position) && (normal) && (texCoord) && ((!tangent) || (!binormal))) {
                additionsSuccessfull = consideredUsageIndices.Add(i);
            }
        }

        if ((!additionsSuccessfull) || consideredUsageIndices.Empty() || (vg->GetVertexCount() == 0U)) {
            return nullBuffer;
        }

        VertexAccessor vertexAccessor(*vg);
        //Initialize arrays for tangents and binormals.
        UInt32 arraySize = vg->GetVertexCount() * static_cast<UInt32>(consideredUsageIndices.Size() * sizeof(Vector3));
        Vector3* tangents = FeatStd::Internal::PointerToPointer<Vector3*>(FEATSTD_ALLOC(arraySize));
        Vector3* binormals = FeatStd::Internal::PointerToPointer<Vector3*>(FEATSTD_ALLOC(arraySize));

        //Initialize tangent array.
        for (UInt32 i = 0; i < (vg->GetVertexCount() * static_cast<UInt32>(consideredUsageIndices.Size())); i++) {
            tangents[i] = Vector3(0.0F, 0.0F, 0.0F);
            binormals[i] = Vector3(0.0F, 0.0F, 0.0F);
        }

        const UInt32 size = static_cast<UInt32>(consideredUsageIndices.Size());

        for (VertexBuffer::PrimitiveIterator triangleIterator = vb->GetPrimitiveIterator(); triangleIterator.IsValid(); ++triangleIterator) {
            const VertexBuffer::Primitive& triangle = *triangleIterator;
            for (Int i = 0; i < static_cast<Int>(size); i++)  {
                //Compute tangents for each vertex, according to http://www.terathon.com/code/tangent.html.

                Vector3 tangent = Vector3(0.0F, 0.0F, 0.0F);
                Vector3 binormal = Vector3(0.0F, 0.0F, 0.0F);

                UInt16 posOffset = 0;
                UInt16 texOffset = 0;

                const VertexGeometry::VertexElementFormat* element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::Position, consideredUsageIndices[i]);
                if (element != 0) {
                    posOffset = element->Offset;
                }

                element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::TextureCoordinate, consideredUsageIndices[i]);
                if (element != 0) {
                    texOffset = element->Offset;
                }

                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                const Float* pos0Pointer = FeatStd::aligned_cast<const Float*>(PointerAdd<const Float*>(vertexAccessor.GetVertex(triangle.m_index[0]), posOffset));
                Vector3 position0 = Vector3(*(pos0Pointer), *(pos0Pointer + 1), *(pos0Pointer + 2));
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                const Float* texCoord0Pointer = FeatStd::aligned_cast<const Float*>(PointerAdd(vertexAccessor.GetVertex(triangle.m_index[0]), texOffset));
                Vector2 texCoord0 = Vector2(*(texCoord0Pointer), *(texCoord0Pointer + 1));

                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                const Float* pos1Pointer = FeatStd::aligned_cast<const Float*>(PointerAdd(vertexAccessor.GetVertex(triangle.m_index[1]), posOffset));
                Vector3 position1 = Vector3(*(pos1Pointer), *(pos1Pointer + 1), *(pos1Pointer + 2));
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                const Float* texCoord1Pointer = FeatStd::aligned_cast<const Float*>(PointerAdd(vertexAccessor.GetVertex(triangle.m_index[1]), texOffset));
                Vector2 texCoord1 =  Vector2(*(texCoord1Pointer), *(texCoord1Pointer + 1));

                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                const Float* pos2Pointer = FeatStd::aligned_cast<const Float*>(PointerAdd(vertexAccessor.GetVertex(triangle.m_index[2]), posOffset));
                Vector3 position2 = Vector3(*(pos2Pointer), *(pos2Pointer + 1), *(pos2Pointer + 2));
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                const Float* texCoord2Pointer = FeatStd::aligned_cast<const Float*>(PointerAdd(vertexAccessor.GetVertex(triangle.m_index[2]), texOffset));
                Vector2 texCoord2 = Vector2(*(texCoord2Pointer), *(texCoord2Pointer + 1));

                Vector3 edge1 = position1 - position0;
                Vector3 edge2 = position2 - position0;
                Vector2 edge1UV = texCoord1 - texCoord0;
                Vector2 edge2UV = texCoord2 - texCoord0;

                Float constantDivisor = (edge1UV.GetY() * edge2UV.GetX()) - (edge1UV.GetX() * edge2UV.GetY());

                if (constantDivisor != 0.0F) {
                    Float constant = 1.0F / constantDivisor;
                    tangent = ((edge1 * -edge2UV.GetY()) + (edge2 * edge1UV.GetY())) * constant;
                    binormal = ((edge1 * -edge2UV.GetX()) + (edge2 * edge1UV.GetX())) * constant;

                    //Tangents and BiNormals are added to already calculated ones in the case of shared vertices.
                    tangents[(triangle.m_index[0] * size) + static_cast<UInt32>(i)] += tangent;
                    tangents[(triangle.m_index[1] * size) + static_cast<UInt32>(i)] += tangent;
                    tangents[(triangle.m_index[2] * size) + static_cast<UInt32>(i)] += tangent;

                    binormals[(triangle.m_index[0] * size) + static_cast<UInt32>(i)] += binormal;
                    binormals[(triangle.m_index[1] * size) + static_cast<UInt32>(i)] += binormal;
                    binormals[(triangle.m_index[2] * size) + static_cast<UInt32>(i)] += binormal;
                }
            }
        }

        if (tangents == 0) {
            //Abort if no tangents were calculated.
            FEATSTD_FREE(tangents);
            FEATSTD_FREE(binormals);
            return nullBuffer;
        }

        //Orthogonalize and Normalize the summed up tangents and binormals.
        for (UInt32 i = 0; i < vg->GetVertexCount(); i++) {
            for (UInt32 j = 0; j < size; j++) {
                Vector3 normal(0.0F, 0.0F, 0.0F);

                const VertexGeometry::VertexElementFormat* element = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::Normal, consideredUsageIndices[j]);
                if (element != 0) {
                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                    const Float* normalData = FeatStd::aligned_cast<const Float*>(PointerAdd(vertexAccessor.GetVertex(i), element->Offset));
                    normal.SetX(normalData[0]);
                    normal.SetY(normalData[1]);
                    normal.SetZ(normalData[2]);
                }

                //Orthogonalize tangent using Gram-Schmidt orthogonalization.
                Vector3 T = tangents[(i * size) + j];  //To shorten Code.
                tangents[(i * size) + j] -= normal * (normal.GetDotProduct(T));
                T = tangents[(i * size) + j];
                //Orthogonalize binormal using Gram-Schmidt orthogonalization.
                binormals[(i * size) + j] = normal.GetCrossProduct(T);

                //Normalize tangents.
                static_cast<void>(tangents[(i * size) + j].Normalize());
                static_cast<void>(binormals[(i * size) + j].Normalize());
            }
        }

        //Build resulting vertex geometry.
        VertexGeometryBuilder builder;
        if (!builder.SpliceGeometry(0, 0, vg)) {
            FEATSTD_LOG_WARN("SpliceGeometry failed.");
        }
        for (UInt32 i = 0; i < builder.GetVertexCount(); i++) {
            builder.SetVertexCursor(i);

            for (UInt32 j = 0; j < size; j++) {
                Int idx = static_cast<Int>(j);      //cast OK because size is originally Int but j and size needs to be UInt32 to have no loss of sign
                Vector3 tangent = tangents[(i * size) + j];
                Vector3 binormal = binormals[(i * size) + j];

                builder.SetVertexElement(VertexGeometry::Tangent, consideredUsageIndices[idx], tangent.GetX(), tangent.GetY(), tangent.GetZ());
                builder.SetVertexElement(VertexGeometry::BiNormal, consideredUsageIndices[idx], binormal.GetX(), binormal.GetY(), binormal.GetZ());
            }
        }

        VertexGeometry* newGeometry = builder.GetVertexGeometry();

        MemoryManagement::SharedPointer<VertexBuffer> newBuffer = VertexBuffer::Create();
        if (!newBuffer->SetVertexGeometry(newGeometry, VertexBuffer::VertexGeometryDisposer::Dispose)) {
            FEATSTD_LOG_WARN("SetVertexGeometry failed.");
        }
        newBuffer->SetPrimitiveType(vb->GetPrimitiveType());

        newBuffer->SetName(vg->GetName());

        FEATSTD_FREE(tangents);
        FEATSTD_FREE(binormals);
        return newBuffer;
    }

    bool Math3D::CalculatePickLine(const Camera& camera, Float x, Float y, Line& line/*[out]*/)
    {
        Vector3 unprojectedNearCoordinates;
        const bool convertNearOk = ConvertViewportCoordinates(camera, Vector3(x, y, 0.0F), unprojectedNearCoordinates);
        if (!convertNearOk) {
            FEATSTD_LOG_ERROR("CalculatePickLine, convert near failed.");
            return false;
        }
        line.SetStart(unprojectedNearCoordinates);

        Vector3 unprojectedFarCoordinates;
        const bool convertFarOk = ConvertViewportCoordinates(camera, Vector3(x, y, 1.0F), unprojectedFarCoordinates);
        if (!convertFarOk) {
            FEATSTD_LOG_ERROR("CalculatePickLine, convert far failed.");
            return false;
        }
        line.SetEnd(unprojectedFarCoordinates);
        return true;
    }

    bool Math3D::IsPickIntersectingGeometry(const Node& node, const Camera& camera, Int x, Int y, Float& distance /*out*/)
    {
        Line line;
        if (!Math3D::CalculatePickLine(camera, static_cast<Float>(x), static_cast<Float>(y), line)) {
            return false;
        }

        return node.IsLineIntersectingGeometry(line, distance);
    }

    bool Math3D::ConvertViewportCoordinates(const Camera& camera,
                                            const Vector3& screenspaceCoordinates,
                                            Vector3& unprojectedCoordinates /*[out]*/)
    {
        Rectangle viewport = camera.GetViewport();
        RenderTarget3D* const renderTarget = camera.GetRenderTarget();

        if (renderTarget == 0) {
            FEATSTD_LOG_ERROR("Convert viewport coordinates failed, renderTarget == 0.");
            return false;
        }

        // Default window to render target dimensions. Position is 0.
        Float windowX = 0.0F;
        Float windowY = 0.0F;
        Float windowHeight = static_cast<Float>(renderTarget->GetHeight());
        Float windowWidth = static_cast<Float>(renderTarget->GetWidth());

        // Replace the default window with the one from the Graphic Device Unit.
        GraphicDeviceUnit* const gdu = renderTarget->GetGraphicDeviceUnit();
        if (gdu != 0) {
            const Window* const window = gdu->ToWindow();
            if (window != 0) {
                Float width = static_cast<Float>(window->GetWidth());
                Float height = static_cast<Float>(window->GetHeight());
                if ((width > 0.0F) && (height > 0.0F)) {
                    windowX = static_cast<Float>(window->GetX());
                    windowY = static_cast<Float>(window->GetY());
                    windowWidth = width;
                    windowHeight = height;
                }
            }
        }

        // Bring coordinates from screen space to window.
        Vector4 vec4;
        vec4.SetX(screenspaceCoordinates.GetX() - windowX);
        // The following calculations are based on bottom-left as origin, so we have to switch the origin from top-left.
        vec4.SetY(windowHeight - (screenspaceCoordinates.GetY() - windowY));
        vec4.SetZ(screenspaceCoordinates.GetZ());
        vec4.SetW(1.0F);

        // Bring viewport to window space.
        Float viewportLeft = viewport.GetLeft() * windowWidth;
        Float viewportWidth = viewport.GetWidth() * windowWidth;
        Float viewportTop = viewport.GetTop() * windowHeight;
        Float viewportHeight = viewport.GetHeight() * windowHeight;

        // Normalize absolute viewport coordinates to 0 - 1.
        vec4.SetX((vec4.GetX() - viewportLeft) / viewportWidth);
        // Viewport has also origin
        vec4.SetY((vec4.GetY() - (windowHeight - (viewportTop + viewportHeight))) / viewportHeight);

        // Re-map to range -1 to 1.
        vec4.SetX((vec4.GetX() * 2.0F) - 1.0F);
        vec4.SetY((vec4.GetY() * 2.0F) - 1.0F);
        vec4.SetZ((vec4.GetZ() * 2.0F) - 1.0F);

        // un-project
        Matrix4 inverseViewProjection = camera.GetViewProjectionMatrix();
        inverseViewProjection.Inverse();
        vec4.TransformCoordinate(inverseViewProjection);

        if (vec4.GetW() == 0.0F) {
            FEATSTD_LOG_ERROR("Convert viewport coordinates failed, vec4.GetW() == 0.");
            return false;
        }

        unprojectedCoordinates.SetX(vec4.GetX() / vec4.GetW());
        unprojectedCoordinates.SetY(vec4.GetY() / vec4.GetW());
        unprojectedCoordinates.SetZ(vec4.GetZ() / vec4.GetW());

        return true;
    }

    bool Math3D::ComputeBoundingBox(const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer, Vector3& minBounds, Vector3& maxBounds, UInt8 index)
    {
        if ((vertexBuffer == 0) || (vertexBuffer->GetVertexGeometry() == 0)) {
            FEATSTD_LOG_ERROR("Compute bounding box failed, invalid vertexbuffer.");
            return false;
        }


        const VertexGeometry* vertexGeometry = vertexBuffer->GetVertexGeometry();
        VertexGeometry::VertexArrayResource vertexArrayResource(vertexGeometry->GetVertexArrayResourceHandle());
        if ((vertexArrayResource.GetData() == 0) || (vertexGeometry->GetVertexCount() == 0)) {
            FEATSTD_LOG_ERROR("Compute bounding box failed, invalid vertexbuffer.");
            return false;
        }

        const Float* geometry = 0;
        UInt16 offset = 0;
        UInt16 stride = vertexGeometry->GetVertexStride() / static_cast<UInt16>(sizeof(Float));
        VertexGeometry::FormatArrayResource formatArrayResource(vertexGeometry->GetFormatArrayResourceHandle());
        for (UInt16 i = 0; i < vertexGeometry->GetVertexElementCount(); ++i) {
            const VertexGeometry::VertexElementFormat& vertexElementFormat = formatArrayResource.GetData()[i];
            if ((vertexElementFormat.Usage == VertexGeometry::Position) &&
                    (vertexElementFormat.Type == VertexGeometry::Float32_3) &&
                    (vertexElementFormat.UsageIndex == index)) {
                offset = vertexElementFormat.Offset / static_cast<UInt16>(sizeof(Float));
                geometry = FeatStd::Internal::PointerToPointer<const Float*>(vertexArrayResource.GetData());
                break;
            }
        }
        if (geometry == 0) {
            return false;
        }

        //find bounding box based on vertices
        UInt32 vertex_count = vertexGeometry->GetVertexCount();
        Float min_x = Math::MaxFloat();
        Float min_y = Math::MaxFloat();
        Float min_z = Math::MaxFloat();
        Float max_x = -Math::MaxFloat();
        Float max_y = -Math::MaxFloat();
        Float max_z = -Math::MaxFloat();

        for (UInt32 idx = 0; idx < vertex_count; idx++) {
            // get one vertex
            UInt32 startPos = (idx * stride) + offset;
            Vector3 vertex(geometry[startPos],
                           geometry[startPos + 1],
                           geometry[startPos + 2]);
            if (vertex.GetX() < min_x) {
                min_x = vertex.GetX();
            }
            if (vertex.GetY() < min_y) {
                min_y = vertex.GetY();
            }
            if (vertex.GetZ() < min_z) {
                min_z = vertex.GetZ();
            }
            if (vertex.GetX() > max_x) {
                max_x = vertex.GetX();
            }
            if (vertex.GetY() > max_y) {
                max_y = vertex.GetY();
            }
            if (vertex.GetZ() > max_z) {
                max_z = vertex.GetZ();
            }
        }
        minBounds = Vector3(min_x, min_y, min_z);
        maxBounds = Vector3(max_x, max_y, max_z);
        return true;
    }

    void Math3D::TransformBoundingBox(const Vector3& srcMinBounds,
                                      const Vector3& srcMaxBounds,
                                      const Matrix4& transform,
                                      Vector3& minBounds,
                                      Vector3& maxBounds)
    {
        minBounds.SetX(transform(3, 0));
        minBounds.SetY(transform(3, 1));
        minBounds.SetZ(transform(3, 2));
        maxBounds = minBounds;

        for (UInt8 i = 0; i < 3; i ++) {
            for (UInt8 j = 0; j < 3; j ++) {
                Float minVal = srcMinBounds[j] * transform(j, i);
                Float maxVal = srcMaxBounds[j] * transform(j, i);

                if (minVal > maxVal) {
                    Float auxVal = minVal;
                    minVal = maxVal;
                    maxVal = auxVal;
                }

                minBounds[i] += minVal;
                maxBounds[i] += maxVal;
            }
        }
    }

    void Math3D::TransformBoundingBoxToViewport(const Vector3* bbBounds, const Camera* camera, Rectangle& boundingRectangle)
    {
        Matrix4 viewProjection = camera->GetViewProjectionMatrix();

        Vector2 topLeft;
        Vector2 bottomRight;

        Vector4 transformedVec4(bbBounds[0]);
        transformedVec4.TransformCoordinate(viewProjection);
        transformedVec4 /= transformedVec4.GetW();
        topLeft = (bottomRight = Vector2(transformedVec4.GetX(), -transformedVec4.GetY()));

        for (UInt8 boundIndex = 1; boundIndex < 8; ++boundIndex) {
            Vector4 boundVec4(bbBounds[boundIndex]);
            boundVec4.TransformCoordinate(viewProjection);
            boundVec4 /= boundVec4.GetW();

            Math::SetToMin(boundVec4.GetX(), bottomRight[0]);
            Math::SetToMin(-boundVec4.GetY(), bottomRight[1]);
            Math::SetToMax(boundVec4.GetX(), topLeft[0]);
            Math::SetToMax(-boundVec4.GetY(), topLeft[1]);
        }

        bottomRight -= topLeft;

        boundingRectangle.SetLeft(topLeft.GetX());
        boundingRectangle.SetTop(topLeft.GetY());
        boundingRectangle.SetWidth(bottomRight.GetX());
        boundingRectangle.SetHeight(bottomRight.GetY());
    }

    void Math3D::TransformPointToViewport(const Vector3& point, const Camera* camera, Vector3& transformedPoint)
    {
        Matrix4 viewProjection = camera->GetViewProjectionMatrix();

        Vector4 transformedVec4(point);
        transformedVec4.TransformCoordinate(viewProjection);
        transformedVec4 /= transformedVec4.GetW();

        transformedPoint.SetX(transformedVec4.GetX());
        transformedPoint.SetY(transformedVec4.GetY());
        transformedPoint.SetZ(transformedVec4.GetZ());
    }


    bool Math3D::TransformBoundingBoxToSurface(const Vector3* bbBounds, const Camera* camera, Rectangle& boundingRectangle)
    {
        RenderTarget3D* renderTarget = camera->GetRenderTarget();

        if (renderTarget == 0) {
            return false;
        }

        TransformBoundingBoxToViewport(bbBounds, camera, boundingRectangle);
        Rectangle viewport = camera->GetViewport();

        boundingRectangle.SetTop(((((boundingRectangle.GetTop() + 1.0F) / 2.0F) * viewport.GetHeight()) + viewport.GetTop()) * static_cast<Float>(renderTarget->GetHeight()));
        boundingRectangle.SetLeft(((((boundingRectangle.GetLeft() + 1.0F) / 2.0F) * viewport.GetWidth()) + viewport.GetLeft()) * static_cast<Float>(renderTarget->GetWidth()));
        boundingRectangle.SetWidth(((boundingRectangle.GetWidth() / 2.0F) * viewport.GetWidth()) * static_cast<Float>(renderTarget->GetWidth()));
        boundingRectangle.SetHeight(((boundingRectangle.GetHeight() / 2.0F) * viewport.GetHeight()) * static_cast<Float>(renderTarget->GetHeight()));

        return true;
    }

    bool Math3D::TransformPointToSurface(const Vector3 point, const Camera* camera, Vector2& transformedPoint)
    {
        RenderTarget3D* renderTarget = camera->GetRenderTarget();

        if (renderTarget == 0) {
            return false;
        }

        Vector3 transPoint;

        TransformPointToViewport(point, camera, transPoint);
        Rectangle viewport = camera->GetViewport();

        transformedPoint.SetX(((((transPoint.GetX() + 1.0F) / 2.0F) * viewport.GetWidth()) + viewport.GetLeft()) * static_cast<Float>(renderTarget->GetWidth()));
        transformedPoint.SetY((1.0F - ((((transPoint.GetY() + 1.0F) / 2.0F) * viewport.GetHeight()) + viewport.GetTop())) * static_cast<Float>(renderTarget->GetHeight()));

        return true;
    }

    bool Math3D::TransformPointToScreen(const Vector3 point, const Camera* camera, Vector2& transformedPoint)
    {
        bool result = false;

        RenderTarget3D* renderTarget = camera->GetRenderTarget();
        if (renderTarget != 0) {
            GraphicDeviceUnit* gdu = renderTarget->GetGraphicDeviceUnit();
            if (gdu != 0) {
                Window* window = gdu->ToWindow();
                if ((window != 0) && (window->GetHeight() > 0) && (window->GetWidth() > 0)) {
                    Vector2 surfaceSpacePoint;
                    if (TransformPointToSurface(point, camera, surfaceSpacePoint)) {
                        transformedPoint.SetX(surfaceSpacePoint.GetX() + static_cast<Float>(window->GetX()));
                        transformedPoint.SetY(surfaceSpacePoint.GetY() + static_cast<Float>(window->GetY()));
                        result = true;
                    }
                }
            }
        }
        return result;
    }

    bool Math3D::TransformBoundingBoxToScreen(const Vector3* bbBounds, const Camera* camera, Rectangle& boundingRectangle)
    {
        bool result = false;

        RenderTarget3D* renderTarget = camera->GetRenderTarget();
        if (renderTarget != 0) {
            GraphicDeviceUnit* gdu = renderTarget->GetGraphicDeviceUnit();
            if (gdu != 0) {
                Window* window = gdu->ToWindow();
                if ((window != 0) && (window->GetHeight() > 0) && (window->GetWidth() > 0)) {
                    Rectangle surfaceSpaceBoundingRectangle;
                    if (TransformBoundingBoxToSurface(bbBounds, camera,surfaceSpaceBoundingRectangle)) {
                        boundingRectangle.SetLeft(surfaceSpaceBoundingRectangle.GetLeft() + static_cast<Float>(window->GetX()));
                        boundingRectangle.SetTop(surfaceSpaceBoundingRectangle.GetTop() + static_cast<Float>(window->GetY()));
                        boundingRectangle.SetWidth(surfaceSpaceBoundingRectangle.GetWidth());
                        boundingRectangle.SetHeight(surfaceSpaceBoundingRectangle.GetHeight());
                        result = true;
                    }
                }
            }
        }
        return result;
    }

    void Math3D::ConvertRotationMatrixToEulerAngles(const Matrix4& rotationMatrix, Vector3& eulerAngles)
    {
        Float cosY = Math::SquareRoot((rotationMatrix(0, 0) * rotationMatrix(0, 0)) + (rotationMatrix(0, 1) * rotationMatrix(0, 1)));

        Float pitch;
        Float yaw;
        Float roll;

        if (cosY > (16.0F * Math::EpsilonFloat())) {
            pitch = Math::ATangent2(1.0F * rotationMatrix(1, 2), rotationMatrix(2, 2));
            yaw = Math::ATangent2(-1.0F * rotationMatrix(0, 2), cosY);
            roll = Math::ATangent2(1.0F * rotationMatrix(0, 1), rotationMatrix(0, 0));
        }
        else {
            pitch = Math::ATangent2(-1.0F * rotationMatrix(2, 1), rotationMatrix(1, 1));
            yaw = Math::ATangent2(-1.0F * rotationMatrix(0, 2), cosY);
            roll = 0.0F;
        }

        eulerAngles[0] = Math::RadianToDegree(pitch);
        eulerAngles[1] = Math::RadianToDegree(yaw);
        eulerAngles[2] = Math::RadianToDegree(roll);
    }

    static UInt32 OptimizeVertexCacheIteration(VertexCacheData* vertices, Int32 vertexIndex, VertexCache& vertexCache, Int32* triangles, UInt32& triangleIndex, const IndexAccessor& indexAccessor);

    UInt32 Math3D::OptimizeVertexCache(MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer, UInt8 vertexCacheSize)
    {
        VertexGeometry* mutableVertexGeometry = 0;
        if (vertexBuffer != 0) {
            VertexGeometry* vertexGeometry = vertexBuffer->GetVertexGeometry();
            mutableVertexGeometry = ((vertexGeometry != 0) &&
                (vertexGeometry->GetVertexArrayResourceHandle().m_isMutable == 1) &&
                (vertexGeometry->GetIndexArrayResourceHandle().m_isMutable == 1) &&
                (vertexGeometry->GetFormatArrayResourceHandle().m_isMutable == 1)) ? vertexGeometry : 0;
        }

        if ((vertexBuffer == 0) ||
                (vertexBuffer->GetPrimitiveType() != VertexBuffer::Triangles) ||
                (mutableVertexGeometry == 0) ||
                (vertexBuffer->GetVertexGeometry()->GetBufferType() != VertexGeometry::IndexedArrayBuffer)) {
            return 0;
        }
        UInt32 vertexCount = vertexBuffer->GetVertexGeometry()->GetVertexCount();
        UInt32 indexCount = vertexBuffer->GetVertexGeometry()->GetIndexCount();
        IndexAccessor indexAccessor(*mutableVertexGeometry);

        //The vertex cache misses is initialized with 0.
        UInt32 cacheMisses = 0;

        //Memory is allocated for vertex data used for caching, as well as a secondary vertex data array for simulating various scenarios.
        VertexCacheData* vertices = FEATSTD_NEW_ARRAY(VertexCacheData, vertexCount);
        if (vertices == 0) {
            return 0;
        }
        VertexCacheData* testVertices = FEATSTD_NEW_ARRAY(VertexCacheData, vertexCount);
        if (testVertices == 0) {
            FEATSTD_DELETE_ARRAY(vertices);
            return 0;
        }

        //The number of triangles that each vertex touches is initialized.
        for (UInt32 iIndex = 0; iIndex < indexCount; ++iIndex) {
            vertices[indexAccessor.GetIndex(iIndex)].triangleCount++;
        }

        //Memory is allocated for storing the unsorted triangles that each vertex touches.
        for (UInt32 vIndex = 0; vIndex < vertexCount; ++vIndex) {
            vertices[vIndex].triangles = FEATSTD_NEW_ARRAY(UInt32, vertices[vIndex].triangleCount);
            if (vertices[vIndex].triangles == 0) {
                for (UInt32 vIndex2 = 0; vIndex2 < vIndex; ++vIndex2) {
                    FEATSTD_DELETE_ARRAY(vertices[vIndex2].triangles);
                    FEATSTD_DELETE_ARRAY(testVertices[vIndex2].triangles);
                }
                FEATSTD_DELETE_ARRAY(testVertices);
                FEATSTD_DELETE_ARRAY(vertices);
                return 0;
            }
            testVertices[vIndex].triangles = FEATSTD_NEW_ARRAY(UInt32, vertices[vIndex].triangleCount);
            if (testVertices[vIndex].triangles == 0) {
                for (UInt32 vIndex2 = 0; vIndex2 < vIndex; ++vIndex2) {
                    FEATSTD_DELETE_ARRAY(vertices[vIndex2].triangles);
                    FEATSTD_DELETE_ARRAY(testVertices[vIndex2].triangles);
                }
                FEATSTD_DELETE_ARRAY(vertices[vIndex].triangles);
                FEATSTD_DELETE_ARRAY(testVertices);
                FEATSTD_DELETE_ARRAY(vertices);
                return 0;
            }
        }

        //Triangle list of each vertex is initialized.
        for (UInt32 iIndex = 0; iIndex < indexCount; ++iIndex) {
            vertices[indexAccessor.GetIndex(iIndex)].triangles[testVertices[indexAccessor.GetIndex(iIndex)].triangleCount++] = iIndex / 3;
        }

        //The main vertex cache is initialized.
        VertexCache vertexCache;
        if (!vertexCache.Init(vertexCacheSize)) {
            for (UInt32 vIndex = 0; vIndex < vertexCount; ++vIndex) {
                FEATSTD_DELETE_ARRAY(vertices[vIndex].triangles);
                FEATSTD_DELETE_ARRAY(testVertices[vIndex].triangles);
            }
            FEATSTD_DELETE_ARRAY(testVertices);
            FEATSTD_DELETE_ARRAY(vertices);
            return 0;
        }

        //A second vertex cache, used to simulate how the main cache would behave in various scenarios, is declared.
        VertexCache testVertexCache;

        //The number of triangles to be sorted is the number of indices divided by 3.
        UInt32 triangleCount = indexCount / 3;

        //The number of triangles already sorted is initialized with 0.
        UInt32 triangleIndex = 0;

        //Memory is allocated for the sorted triangle array
        Int32* triangles = FEATSTD_NEW_ARRAY(Int32, triangleCount);
        if (triangles == 0) {
            for (UInt32 vIndex = 0; vIndex < vertexCount; ++vIndex) {
                FEATSTD_DELETE_ARRAY(vertices[vIndex].triangles);
                FEATSTD_DELETE_ARRAY(testVertices[vIndex].triangles);
            }
            FEATSTD_DELETE_ARRAY(testVertices);
            FEATSTD_DELETE_ARRAY(vertices);
            return 0;
        }

        //and each triangle is initialized.
        for (UInt32 tIndex = 0; tIndex < triangleCount; ++tIndex) {
            triangles[tIndex] = tIndex;
        }

        //While the number of sorted triangles is less than the total number of triangles
        while (triangleIndex != triangleCount) {
            //one index is searched in the indexArray that will be cleared as soon as possible (all the triangles that
            // it touches will be added to the sorted triangle list, so the vertex will not be added to the cache in any
            // subsequent steps).
            UInt32 vertexIndex = vertexCount;

            //If the main vertex cache is empty
            if (vertexCache.GetSize() == 0) {
                //the vertex array is iterated
                for (UInt32 index = 0; index < vertexCount; ++index) {
                    //to find the vertex that still touches any triangles
                    if (vertices[index].triangleCount > 0) {
                        if (vertexIndex == vertexCount) {
                            vertexIndex = index;
                        }
                        else {
                            //and the number of triangles it touches is minimum.
                            if (vertices[index].triangleCount < vertices[vertexIndex].triangleCount) {
                                vertexIndex = index;
                            }
                        }
                    }
                }
            }
            //If the main vertex cache is not empty,
            else {
                Int32 minCost = FeatStd::Internal::NativeTypeLimit<Int32>::Max();
                //for each vertex in the vertex cache
                for (UInt8 index = 0; index < vertexCache.GetSize(); ++index) {
                    Int32 iIndex = vertexCache.GetVertex(index);
                    //that still touches any triangles,
                    if (vertices[iIndex].triangleCount != 0) {
                        //the test cache is initialized with the main one.
                        if (testVertexCache.Init(vertexCache)) {
                            for (UInt32 tIndex = 0; tIndex < vertices[iIndex].triangleCount; ++tIndex) {
                                for (UInt32 vIndex = 0; vIndex < 3; ++vIndex) {
                                    Int32 iIndex2 = indexAccessor.GetIndex((3 * vertices[iIndex].triangles[tIndex]) + vIndex);
                                    for (UInt32 tIndex2 = 0; tIndex2 < vertices[iIndex2].triangleCount; ++tIndex2) {
                                        for (Int32 vIndex2 = 0; vIndex2 < 3; ++vIndex2) {
                                            UInt32 iIndex3 = indexAccessor.GetIndex((3 * vertices[iIndex2].triangles[tIndex2]) + vIndex2);
                                            testVertices[iIndex3].isCached = vertices[iIndex3].isCached;
                                            testVertices[iIndex3].triangleCount = vertices[iIndex3].triangleCount;
                                            MemoryPlatform::Copy(testVertices[iIndex3].triangles, vertices[iIndex3].triangles, static_cast<SizeType>(vertices[iIndex3].triangleCount) * sizeof(UInt32));
                                        }
                                    }
                                }
                            }
                        }
                        else {
                            FEATSTD_LOG_ERROR("Init of testVertexCache failed.");
                        }

                        UInt32 newTriangleIndex = triangleIndex;

                        //The process of clearing the selected vertex is simulated, using the test cache, and a cost is computed based on:
                        // - the number of vertex cache misses that the simulation generates (less is better)
                        // - the number of triangles added the the sorted triangle list (more is better)
                        // - the initial position of the vertex in the cache (closer the the exit is better)
                        Int32 cost = 2 * static_cast<Int32>(OptimizeVertexCacheIteration(testVertices, iIndex, testVertexCache, triangles, newTriangleIndex, indexAccessor));
                        cost = (static_cast<Int32>(vertexCacheSize) * (cost - (static_cast<Int32>(newTriangleIndex)- static_cast<Int32>(triangleIndex)))) + (3 * static_cast<Int32>(index));

                        //The cost is compared with the minimum cost yet, and the vertex with the minimum cost is retained.
                        if (cost < minCost) {
                            vertexIndex = iIndex;
                            minCost = cost;
                        }
                    }
                }
            }

            //The process of clearing the retained vertex is ran, adding the triangles to the sorted triangle list.
            cacheMisses += OptimizeVertexCacheIteration(vertices, vertexIndex, vertexCache, triangles, triangleIndex, indexAccessor);

            //The vertex cache is checked to pop any vertex that is already cleared.
            while ((vertexCache.GetSize() != 0) && (vertices[vertexCache.GetVertex(0)].triangleCount == 0)) {
                vertices[vertexCache.Pop()].isCached = false;
            }
        }

        //The index array is sorted accordingly to the sorted triangle list.
        for (UInt32 tIndex = 0; tIndex < triangleCount; ++tIndex) {
            if (triangles[tIndex] == -1) {
                continue;
            }
            if (triangles[tIndex] == static_cast<Int32>(tIndex)) {
                continue;
            }

            UInt32 iIndex = 3 * tIndex;
            UInt32 i0 = indexAccessor.GetIndex(iIndex);
            UInt32 i1 = indexAccessor.GetIndex(iIndex + 1);
            UInt32 i2 = indexAccessor.GetIndex(iIndex + 2);

            triangleIndex = tIndex;
            bool isSuccess = true;
            while (triangles[triangles[triangleIndex]] != -1) {
                UInt32 iIndex2 = static_cast<UInt32>(3 * triangles[triangleIndex]);
                isSuccess = isSuccess && indexAccessor.SetIndex(iIndex, indexAccessor.GetIndex(iIndex2));
                isSuccess = isSuccess && indexAccessor.SetIndex(iIndex + 1, indexAccessor.GetIndex(iIndex2 + 1));
                isSuccess = isSuccess && indexAccessor.SetIndex(iIndex + 2, indexAccessor.GetIndex(iIndex2 + 2));
                UInt32 temp = triangleIndex;
                triangleIndex = triangles[triangleIndex];
                triangles[temp] = -1;
                iIndex = iIndex2;
            }

            isSuccess = isSuccess && indexAccessor.SetIndex(iIndex, i0);
            isSuccess = isSuccess && indexAccessor.SetIndex(iIndex + 1, i1);
            isSuccess = isSuccess && indexAccessor.SetIndex(iIndex + 2, i2);

            if (!isSuccess) {
                FEATSTD_LOG_ERROR("Failed to reorder vertices during optimization.");
            }
        }

        //Temporary needed memory is freed.
        for (UInt32 vIndex = 0; vIndex < vertexCount; ++vIndex) {
            if (vertices[vIndex].triangles != 0) {
                FEATSTD_DELETE_ARRAY(vertices[vIndex].triangles);
            }
            if (testVertices[vIndex].triangles != 0) {
                FEATSTD_DELETE_ARRAY(testVertices[vIndex].triangles);
            }
        }

        FEATSTD_DELETE_ARRAY(vertices);
        FEATSTD_DELETE_ARRAY(testVertices);
        FEATSTD_DELETE_ARRAY(triangles);

        return cacheMisses;
    }

#ifndef FEATSTD_FIXED_POINT_ARITHMETIC

    static const UInt32 FloatMaxExponent = static_cast<UInt32>(0xFFU) << 23U;
    static const UInt32 HalfFloatMaxExponent = static_cast<UInt32>(0x1FU) << 10U;
    static const UInt32 HalfFloatMaxExponentAsFloatExponent = 0x47800000U;
    static const UInt32 HalfFloatMinExponentAsFloatExponent = 0x38000000U;


    UInt16 Math3D::ConvertFloatToHalfFloat(Float floatValue)
    {

        UInt32 x = 0;
        MemoryPlatform::Copy(&x, &floatValue, 4);
        UInt32 sign = x >> 31;

        UInt16 result = 0;
        //lint --e{838} mantissa is used
        const UInt32 cMantissaMask = (static_cast<UInt32>(1U) << 23U) - 1U;
        UInt32 mantissa = x & cMantissaMask;
        //lint --e{838} exponent is used
        UInt32 exponent = x & FloatMaxExponent;


        if (exponent >= HalfFloatMaxExponentAsFloatExponent) {
            //Handle NaN and Infinity.
            if ((mantissa != 0) && (exponent == FloatMaxExponent)) {
                //NaN = Mantissa !=0 and max exponent.
mantissa = cMantissaMask;
            }
            else {
                //Store all values with exponent >= Float exponent as Infinity.
                mantissa = 0;
            }

            result = static_cast<UInt16>(sign << 15);
            result |= static_cast<UInt16>(HalfFloatMaxExponent);
            result |= static_cast<UInt16>(mantissa >> 13);
        }
        else if (exponent <= HalfFloatMinExponentAsFloatExponent) {
            //Handle exponent too small for half precision.
            exponent = (HalfFloatMinExponentAsFloatExponent - exponent) >> 23;
            mantissa = mantissa >> (14 + exponent);

            result = static_cast<UInt16>(sign << 15);
            result |= static_cast<UInt16>(mantissa);
        }
        else {
            //Valid case.
            result = static_cast<UInt16>(sign << 15);
            result |= static_cast<UInt16>((exponent - HalfFloatMinExponentAsFloatExponent) >> 13);
            result |= static_cast<UInt16>(mantissa >> 13);
        }

        return result;
    }

    Float Math3D::ConvertHalfFloatToFloat(UInt16 halfFloatValue)
    {
        UInt32 sign = static_cast<UInt32>(halfFloatValue) >> 15U;
        UInt32 mantissa = static_cast<UInt32>(halfFloatValue)& ((static_cast<UInt32>(1) << 10U) - 1U);
        UInt32 exponent = static_cast<UInt32>(halfFloatValue)& HalfFloatMaxExponent;
        UInt32 result = 0;

        if (exponent == HalfFloatMaxExponent) {
            //Handle NaN or Infinity.
            exponent = FloatMaxExponent;
            if (mantissa != 0) {
                mantissa = (static_cast<UInt32>(1) << 23U) - 1U;
            }
        }
        else if (exponent == 0) {
            if (mantissa != 0) {
                mantissa = mantissa << 1;
                exponent = HalfFloatMinExponentAsFloatExponent;

                while ((mantissa & (UInt32(1U) << 10U)) == 0) {
                    mantissa = mantissa << 1;
                    exponent = exponent - (UInt32(1U) << 23U);
                }

                mantissa = mantissa & ((UInt32(1U) << 10U) - 1U);

                mantissa = mantissa << 13U;
            }
        }
        else {
            mantissa = mantissa << 13;
            exponent = (exponent << 13) + HalfFloatMinExponentAsFloatExponent;
        }
        //lint --e{838} result is used correctly and the value is not lost
        result = sign << 31;
        result |= exponent;
        result |= mantissa;

        Float floatResult = 0.0F;
        MemoryPlatform::Copy(&floatResult, &result, 4);

        return floatResult;
    }

#endif

    struct IndexLine {
        UInt32 m_index0;
        UInt32 m_index1;
    };

    static bool LineExists(FeatStd::Internal::Vector<IndexLine>& lines, IndexLine& l)
    {
        bool result = false;
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, l, CANDERA_LINT_REASON_NONCONST)
            for (Int i = FeatStd::Internal::NumericConversion<Int>(lines.Size()) - 1; i >= 0; i--) {
            if ((l.m_index0 == lines[i].m_index0) && (l.m_index1 == lines[i].m_index1)) {
                result = true;
                break;
            }
            }

        return result;
    }

    static MemoryManagement::SharedPointer<VertexBuffer> CreateLinesFromTrianglesInternal(const MemoryManagement::SharedPointer<VertexBuffer>& vb)
    {
        MemoryManagement::SharedPointer<VertexBuffer> result = MemoryManagement::SharedPointer<VertexBuffer>(0);

        if (vb != 0) {
            if ((vb->GetPrimitiveType() != VertexBuffer::Triangles) &&
                (vb->GetPrimitiveType() != VertexBuffer::TriangleFan) &&
                (vb->GetPrimitiveType() != VertexBuffer::TriangleStrip)) {
                return result;
            }

            const VertexGeometry* vg = vb->GetVertexGeometry();
            if (vg != 0) {
                VertexGeometryBuilder builder;
                VertexGeometry* newGeometry = 0;

                UInt16 stride = vg->GetVertexStride();

                bool success = builder.SpliceGeometry(0, 0, vg);
                builder.SetIndexCursor(0);

                if (stride != 0) {
                    FeatStd::Internal::Vector<IndexLine> lines;
                    for (VertexBuffer::PrimitiveIterator triangleIterator = vb->GetPrimitiveIterator(); success && triangleIterator.IsValid(); ++triangleIterator) {
                        const VertexBuffer::Primitive& triangle = *triangleIterator;
                        UInt32 idx0 = triangle.m_index[0];
                        UInt32 idx1 = triangle.m_index[1];
                        UInt32 idx2 = triangle.m_index[2];

                        IndexLine l0;
                        IndexLine l1;
                        IndexLine l2;
                        l0.m_index0 = ((idx0 <= idx1) ? idx0 : idx1); //Always sort line indices to use lower index as first in order to speed up search.
                        l0.m_index1 = ((idx0 <= idx1) ? idx1 : idx0);
                        l1.m_index0 = ((idx1 <= idx2) ? idx1 : idx2);
                        l1.m_index1 = ((idx1 <= idx2) ? idx2 : idx1);
                        l2.m_index0 = ((idx2 <= idx0) ? idx2 : idx0);
                        l2.m_index1 = ((idx2 <= idx0) ? idx0 : idx2);
                        if (!LineExists(lines, l0)) {
                            builder.SetIndexElement(idx0);
                            builder.IncrementIndexCursor();
                            builder.SetIndexElement(idx1);
                            builder.IncrementIndexCursor();
                            success = lines.Add(l0);
                        }

                        if (!LineExists(lines, l1)) {
                            builder.SetIndexElement(idx1);
                            builder.IncrementIndexCursor();
                            builder.SetIndexElement(idx2);
                            builder.IncrementIndexCursor();
                            success = success && lines.Add(l1);
                        }
                        if (!LineExists(lines, l2)) {
                            builder.SetIndexElement(idx2);
                            builder.IncrementIndexCursor();
                            builder.SetIndexElement(idx0);
                            builder.IncrementIndexCursor();
                            success = success && lines.Add(l2);
                        }
                    }
                }

                if (builder.GetIndexCursor() > 0) {
                    //Only build vertex geometry, if valid case has been processed, don't add vertex geometry otherwise.
                    builder.SetMaxIndexCount(builder.GetIndexCursor());     //Discard all still remaining old indices.
                    builder.SetBufferType(VertexGeometry::IndexedArrayBuffer);
                    builder.SetBufferUsage(vg->GetBufferUsage());
                    builder.SetMemoryPool(vg->GetMemoryPool());
                    newGeometry = builder.GetVertexGeometry();
                }

                if (newGeometry != 0) {
                    result = VertexBuffer::Create();
                    result->SetName(vb->GetName());
                    result->SetPrimitiveType(VertexBuffer::Lines);
                    static_cast<void>(result->SetVertexGeometry(newGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));
                }
            }
        }

        return result;
    }

    static UInt32 OptimizeVertexCacheIteration(VertexCacheData* vertices, Int32 vertexIndex, VertexCache& vertexCache, Int32* triangles, UInt32& triangleIndex, const IndexAccessor& indexAccessor)
    {
        UInt32 cacheMisses = 0;

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, indexArray, CANDERA_LINT_REASON_NONCONST)

        //While the given vertex still touches any triangles
        while (vertices[vertexIndex].triangleCount != 0) {
            //the first triangle is selected to be added to the sorted triangle list.
            UInt32 tIndex = vertices[vertexIndex].triangles[0];

            //Each vertex of the selected triangle
            for (UInt32 vIndex = 0; vIndex < 9; ++vIndex) {
                UInt32 inVertex = indexAccessor.GetIndex((3 * tIndex) + (vIndex % 3));
                //is checked if already cached.
                if (!vertices[inVertex].isCached) {

                    //If the vertex is not cached and the cache is full, one vertex is popped out of the cache to make room.
                    if (vertexCache.IsFull()) {
                        vertices[vertexCache.Pop()].isCached = false;
                    }

                    //The vertex is pushed into the cache
                    vertexCache.Push(inVertex);
                    vertices[inVertex].isCached = true;
                    //updating the number of cache misses.
                    cacheMisses++;
                    //For each triangle that the last added vertex it touches,
                    for (UInt32 i = vertices[inVertex].triangleCount; i > 0; --i) {
                        UInt32 tIndex2 = vertices[inVertex].triangles[i - 1];
                        UInt32 iIndex2 = 3 * tIndex2;
                        //it is checked if all its vertices are in the cache.
                        if (vertices[indexAccessor.GetIndex(iIndex2)].isCached && vertices[indexAccessor.GetIndex(iIndex2 + 1)].isCached && vertices[indexAccessor.GetIndex(iIndex2 + 2)].isCached) {
                            //If so, we make sure that the triangle is not the one that was already selected
                            if (tIndex != tIndex2) {
                                //and add it to the sorted triangle list.
                                triangles[triangleIndex++] = tIndex2;
                                //For each vertex that connects the triangle
                                for (UInt16 vIndex2 = 0; vIndex2 < 3; ++vIndex2) {
                                    UInt32 iIndex3 = indexAccessor.GetIndex((3 * tIndex2) + vIndex2);
                                    bool copy = false;
                                    //the triangle is removed from the vertex triangle list.
                                    for (UInt32 tIndex3 = 0; tIndex3 < vertices[iIndex3].triangleCount; ++tIndex3) {
                                        if (vertices[iIndex3].triangles[tIndex3] == tIndex2) {
                                            vertices[iIndex3].triangleCount--;
                                            copy = true;
                                        }
                                        if (copy) {
                                            vertices[iIndex3].triangles[tIndex3] = vertices[iIndex3].triangles[tIndex3 + 1];
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }

            //Add the selected triangle to the triangle list.
            triangles[triangleIndex++] = tIndex;
            //For each vertex that connects the triangle
            for (UInt16 vIndex2 = 0; vIndex2 < 3; ++vIndex2) {
                UInt32 iIndex2 = indexAccessor.GetIndex((3 * tIndex) + vIndex2);
                bool copy = false;
                //the triangle is removed from the vertex triangle list.
                for (UInt32 tIndex3 = 0; tIndex3 < vertices[iIndex2].triangleCount; ++tIndex3) {
                    if (vertices[iIndex2].triangles[tIndex3] == tIndex) {
                        vertices[iIndex2].triangleCount--;
                        copy = true;
                    }
                    if (copy) {
                        vertices[iIndex2].triangles[tIndex3] = vertices[iIndex2].triangles[tIndex3 + 1];

                    }
                }
            }

        }

        //return total number of cache misses.
        return cacheMisses;
    }

    MemoryManagement::SharedPointer<VertexBuffer> Math3D::CreateVertexBufferWithIndexRemoved(MemoryManagement::SharedPointer<VertexBuffer> indexedVb)
    {
        MemoryManagement::SharedPointer<VertexBuffer> result(0);

        if (indexedVb != 0) {
            const VertexGeometry* vg = indexedVb->GetVertexGeometry();
            if (vg != 0) {
                if (vg->GetBufferType() == VertexGeometry::IndexedArrayBuffer) {

                    //Size of one vertex in bytes.
                    UInt16 stride = vg->GetVertexStride();

                    //Allocate place for index count vertices.
                    UInt8* destinationVertices = 0;
                    if (vg->GetIndexCount() > 0) {
                        destinationVertices = FEATSTD_NEW_ARRAY(UInt8, static_cast<SizeType>(vg->GetIndexCount()) * static_cast<SizeType>(stride));
                    }

                    VertexGeometry::FormatArrayResource formatArrayResource(vg->GetFormatArrayResourceHandle());
                    const VertexGeometry::VertexElementFormat* sourceVef = formatArrayResource.GetData();

                    //Allocate new vertex element format array.
                    VertexGeometry::VertexElementFormat* destinationVef = 0;
                    if (vg->GetVertexElementCount() > 0) {
                        destinationVef = FEATSTD_NEW_ARRAY(VertexGeometry::VertexElementFormat, vg->GetVertexElementCount());
                    }

                    //Index buffer to unindex.
                    VertexAccessor vertexAccessor(*vg);

                    if ((destinationVertices != 0) && (sourceVef != 0) && (destinationVef != 0)) {

                        IndexAccessor indexAccessor(*vg);
                        for (UInt32 i = 0; i < vg->GetIndexCount(); i++) {
                            MemoryPlatform::Copy(destinationVertices + (i * stride), vertexAccessor.GetVertex(indexAccessor.GetIndex(i)), stride);
                        }

                        //Copy vertex element format.
                        MemoryPlatform::Copy(destinationVef, sourceVef, static_cast<SizeType>(vg->GetVertexElementCount()) * sizeof(VertexGeometry::VertexElementFormat));

                        VertexGeometry* newGeometry = FEATSTD_NEW(VertexGeometry)(
                                                          destinationVertices,
                                                          MemoryManagement::AdaptedArrayDisposer<const void*, const UInt8*>::Dispose,
                                                          destinationVef,
                                                          VertexGeometry::VertexElementFormatDisposer::Dispose,
                                                          0,
                                                          0,
                                                          vg->GetIndexCount(),            //New VertexGeometry has as much vertices as previous one has indices.
                                                          stride,
                                                          vg->GetVertexElementCount(),
                                                          0,
                                                          vg->GetMemoryPool(),
                                                          VertexGeometry::ArrayBuffer,
                                                          vg->GetBufferUsage());

                        if (newGeometry != 0) {
                            result = VertexBuffer::Create();
                            result->SetPrimitiveType(indexedVb->GetPrimitiveType());
                            static_cast<void>(result->SetVertexGeometry(newGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));
                            result->SetDisposedAfterUpload(indexedVb->IsDisposedAfterUpload());
                        }
                        else {
                            FEATSTD_DELETE_ARRAY(destinationVertices);
                            FEATSTD_DELETE_ARRAY(destinationVef);
                        }
                    }
                    else {
                        //Delete eventually allocated arrays.
                        FEATSTD_DELETE_ARRAY(destinationVertices);
                        FEATSTD_DELETE_ARRAY(destinationVef);
                    }
                }
            }
        }

        return result;
    }

    struct IndexMapItem {
        UInt32 oldIndex;
        UInt32 newIndex;
    };

    class VertexComparator {
        public:

            VertexComparator(const VertexAccessor& accessor, UInt16 stride) :
                m_accessor(accessor),
                m_stride(stride)
            {
            }

            bool operator()(IndexMapItem lhs, IndexMapItem rhs) const {
                return (MemoryPlatform::Compare(m_accessor.GetVertex(lhs.oldIndex), m_accessor.GetVertex(rhs.oldIndex), m_stride) < 0);
            }

        private:
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_NONCOPYABLE)
            FEATSTD_MAKE_CLASS_UNCOPYABLE(VertexComparator);
            const VertexAccessor& m_accessor;
            UInt16 m_stride;

    };

    class IndexComparator {
        public:
            bool operator()(IndexMapItem lhs, IndexMapItem rhs) const {
                return lhs.oldIndex < rhs.oldIndex;
            }
    };

    MemoryManagement::SharedPointer<VertexBuffer> Math3D::CreateVertexBufferWithIndexGenerated(MemoryManagement::SharedPointer<VertexBuffer> nonIndexedVb, VertexGeometry::BufferType indexType)
    {
        MemoryManagement::SharedPointer<VertexBuffer> result(0);

        if (nonIndexedVb != 0) {
            const VertexGeometry* vg = nonIndexedVb->GetVertexGeometry();
            if (vg != 0) {
                VertexAccessor vertexAccessor(*vg);
                if (vertexAccessor.GetVertex(0) != 0) {
                    if (vg->GetBufferType() == VertexGeometry::ArrayBuffer) {
                        UInt16 stride = vg->GetVertexStride();

                        FeatStd::Internal::Vector<IndexMapItem> mappedIndices;

                        //I: Initialize mappedIndices with the current indices.
                        bool mappingSuccess = mappedIndices.Reserve(vg->GetVertexCount());
                        for (UInt32 i = 0; mappingSuccess && (i < vg->GetVertexCount()); i++) {
                            IndexMapItem mapItem = {i, 0U};
                            mappingSuccess = mappedIndices.Add(mapItem);
                        }
                        if (!mappingSuccess) {
                            mappedIndices.Clear();
                        }

                        //II: Sort by vertices
                        VertexComparator vertexComparator(vertexAccessor, stride);

                        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1502, vertexComparator, VertexComparator only defines the comparison operator for sorting the list)
                            mappedIndices.Sort(vertexComparator);

                        //III: Iterate over sorted vertices.
                        UInt32 currentIndex = 0;

                        FeatStd::Internal::Vector<const void*> mergedOldVertices; //Needed to collect vertices to copy.

                        bool mergeVerticesSuccess = true;
                        UInt32 mappedIndicesSize = static_cast<UInt32>(mappedIndices.Size());
                        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(850, "Necessary to modify i in the body of the for loop.")
                        for (UInt32 i = 0; mergeVerticesSuccess && (i < mappedIndicesSize); i++) {

                            //Add vertex to copy.
                            mergeVerticesSuccess = mergedOldVertices.Add(vertexAccessor.GetVertex(mappedIndices[i].oldIndex));

                            //Skip adjacent equal vertices, but update index mapping.
                            UInt32 j = i;
                            while ((j < mappedIndicesSize)
                                   && (MemoryPlatform::Compare(vertexAccessor.GetVertex(mappedIndices[i].oldIndex),
                                   vertexAccessor.GetVertex(mappedIndices[j].oldIndex), stride) == 0)) {
                                mappedIndices[j].newIndex = currentIndex;
                                j++;
                            }

                            //Skip adjacent equal vertices!
                            if (j > i) {
                                i = j - 1;  //-1  because of i++ at the end of for.
                            }

                            //Increment index.
                            currentIndex++;
                        }

                        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(850, "Necessary to modify i in the body of the for loop."
                            "Purpose of this algorithm is to merge equal vertices.")
                        if (!mergeVerticesSuccess) {
                            mergedOldVertices.Clear();
                        }

                        //IV resort index vector.
                        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1502, indexComparator, IndexComparator only defines the comparison operator for sorting the list)
                            IndexComparator indexComparator;
                        mappedIndices.Sort(indexComparator);

                        //Allocated needed arrays.
                        void* vertexArray = 0;
                        UInt32 mergedOldVerticesSize = static_cast<UInt32>(mergedOldVertices.Size());
                        if (mergedOldVerticesSize * stride > 0) {
                            vertexArray = FEATSTD_ALLOC(static_cast<SizeType>(mergedOldVerticesSize * stride));
                        }

                        //update indexType to fit all vertex indices
                        if ((indexType == VertexGeometry::UInt8IndexedArrayBuffer) && (mergedOldVerticesSize > FeatStd::Internal::Limits<UInt8>::Max())) {
                            indexType = VertexGeometry::UInt16IndexedArrayBuffer;
                        }

                        if ((indexType == VertexGeometry::UInt16IndexedArrayBuffer) && (mergedOldVerticesSize > FeatStd::Internal::Limits<UInt16>::Max())) {
                            indexType = VertexGeometry::UInt32IndexedArrayBuffer;
                        }

                        UInt8 indexStride = 0;
                        switch (indexType) {
                            case VertexGeometry::UInt8IndexedArrayBuffer: indexStride = 1; break;
                            case VertexGeometry::UInt16IndexedArrayBuffer: indexStride = 2; break;
                            case VertexGeometry::UInt32IndexedArrayBuffer: indexStride = 4; break;
                            default: break;
                        }
                        void* indexArray = 0;
                        if (mappedIndicesSize * indexStride > 0) {
                            indexArray = FEATSTD_ALLOC(static_cast<SizeType>(mappedIndicesSize * indexStride));
                        }

                        VertexGeometry::VertexElementFormat* vefArray = 0;
                        if ((vertexArray != 0) && (vg->GetVertexElementCount() > 0)) {
                            vefArray = FEATSTD_NEW_ARRAY(VertexGeometry::VertexElementFormat, vg->GetVertexElementCount());
                        }

                        VertexGeometry::FormatArrayResource formatArrayResource(vg->GetFormatArrayResourceHandle());
                        const VertexGeometry::VertexElementFormat* srcVefArray = formatArrayResource.GetData();

                        //Copy arrays.
                        if ((vertexArray != 0) && (indexArray != 0) && (vefArray != 0) && (srcVefArray != 0)) {

                            for (UInt32 i = 0; i < mergedOldVerticesSize; i++) {
                                MemoryPlatform::Copy(FeatStd::Internal::PointerAdd(vertexArray, static_cast<OffsetType>(i * stride)), mergedOldVertices[i], stride);
                            }

                            switch (indexType) {
                                case VertexGeometry::UInt8IndexedArrayBuffer:
                                    for (UInt32 i = 0; i < mappedIndicesSize; i++) {
                                        *(FeatStd::Internal::PointerAdd<UInt8*>(indexArray, static_cast<OffsetType>(i * indexStride))) = static_cast<UInt8>(mappedIndices[i].newIndex);
                                    }
                                    break;
                                case VertexGeometry::UInt16IndexedArrayBuffer:
                                    for (UInt32 i = 0; i < mappedIndicesSize; i++) {
                                        *(FeatStd::Internal::PointerAdd<UInt16*>(indexArray, static_cast<OffsetType>(i * indexStride))) = static_cast<UInt16>(mappedIndices[i].newIndex);
                                    }
                                    break;
                                case VertexGeometry::UInt32IndexedArrayBuffer:
                                    for (UInt32 i = 0; i < mappedIndicesSize; i++) {
                                        *(FeatStd::Internal::PointerAdd<UInt32*>(indexArray, static_cast<OffsetType>(i * indexStride))) = mappedIndices[i].newIndex;
                                    }
                                    break;
                                default: break;
                            }

                            //Copy vertex element format.
                            UInt32 vefArraySize = static_cast<UInt32>(vg->GetVertexElementCount()) * sizeof(VertexGeometry::VertexElementFormat);
                            MemoryPlatform::Copy(vefArray, srcVefArray, vefArraySize);

                            VertexGeometry* newGeometry = FEATSTD_NEW(VertexGeometry)(
                                VertexGeometry::VertexArrayResource::CreateHandle(vertexArray, MemoryManagement::FreeMemoryDisposer<const void*>::Dispose, mergedOldVerticesSize * stride),
                                VertexGeometry::IndexArrayResource::CreateHandle(indexArray, MemoryManagement::FreeMemoryDisposer<const void*>::Dispose, mappedIndicesSize * indexStride),
                                VertexGeometry::FormatArrayResource::CreateHandle(vefArray, VertexGeometry::VertexElementFormatDisposer::Dispose, vefArraySize),
                                stride,
                                vg->GetMemoryPool(),
                                indexType,
                                vg->GetBufferUsage());

                            if (newGeometry != 0) {
                                result = VertexBuffer::Create();
                                result->SetPrimitiveType(nonIndexedVb->GetPrimitiveType());
                                static_cast<void>(result->SetVertexGeometry(newGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));
                                result->SetDisposedAfterUpload(nonIndexedVb->IsDisposedAfterUpload());
                            }
                            else {
                                FEATSTD_FREE(vertexArray);
                                FEATSTD_FREE(indexArray);
                                FEATSTD_DELETE_ARRAY(vefArray);
                            }
                        }
                        else {
                            FEATSTD_FREE(vertexArray);
                            FEATSTD_FREE(indexArray);
                            FEATSTD_DELETE_ARRAY(vefArray);
                        }
                    }
                }
            }
        }
        return result;
    }
    
    MemoryManagement::SharedPointer<VertexBuffer> Math3D::CreateVertexBufferByPrimitiveTypeConversion(const MemoryManagement::SharedPointer<VertexBuffer>& vb, PrimitiveTypeConversion target)
    {
        MemoryManagement::SharedPointer<VertexBuffer> result(0);

        switch (target) {
            case LineConversion:
                result = CreateLinesFromTrianglesInternal(vb);
                break;
            case PrimitiveConversionCount:
                break;
            default:
                break;
        }

        return result;
    }

    MemoryManagement::SharedPointer<VertexBuffer> Math3D::CreateVertexBufferWithAttributeGenerated(const MemoryManagement::SharedPointer<VertexBuffer>& vb, AttributeGeneration attributes)
    {
        MemoryManagement::SharedPointer<VertexBuffer> result(0);

        switch (attributes) {
            case TangentAndBinormalGeneration:
                result = CreateTangentsAndBinormalsFromVertexBufferInternal(vb);
                break;
            case AttributeGenerationCount:
                break;
            default:
                break;
        }

        return result;
    }

    MemoryManagement::SharedPointer<VertexBuffer> Math3D::CreateVertexBufferWithAttributeRemoved(const MemoryManagement::SharedPointer<VertexBuffer>& vb, VertexGeometry::VertexUsage usage, UInt8 usageIndex)
    {
        MemoryManagement::SharedPointer<VertexBuffer> result(0);

        if (vb != 0) {
            const VertexGeometry* vg = vb->GetVertexGeometry();
            VertexGeometry* newVg  = 0;

            VertexGeometryBuilder builder;
            if (builder.SpliceGeometry(0, 0, vg)) {
                builder.RemoveVertexElement(usage, usageIndex);
                newVg = builder.GetVertexGeometry();
            }

            if (newVg != 0) {
                result = VertexBuffer::Create();
                result->SetPrimitiveType(vb->GetPrimitiveType());
                static_cast<void>(result->SetVertexGeometry(newVg, VertexBuffer::VertexGeometryDisposer::Dispose));
                result->SetDisposedAfterUpload(vb->IsDisposedAfterUpload());
            }
        }

        return result;
    }

    MemoryManagement::SharedPointer<VertexBuffer> Math3D::CreateVertexBufferByPrecisionConversion(const MemoryManagement::SharedPointer<VertexBuffer>& vb,
                                                                                                  PrecisionConversionTarget precision, VertexGeometry::VertexUsage usage, UInt8 usageIndex)
    {

        MemoryManagement::SharedPointer<VertexBuffer> result(0);



        if ((vb != 0) && (precision != PrecisionConversionCount)) {
            const VertexGeometry* vg = vb->GetVertexGeometry();

            if (vg != 0) {

                bool hasAttribute = false;

                VertexGeometry::FormatArrayResource formatArrayResource(vg->GetFormatArrayResourceHandle());
                const VertexGeometry::VertexElementFormat* attributes = formatArrayResource.GetData();

                for (UInt16 i = 0; i < vg->GetVertexElementCount(); i++) {
                    if (attributes != 0) {
                        if ((attributes[i].Usage == usage) && (attributes[i].UsageIndex == usageIndex)) {
                            hasAttribute = true;
                        }
                    }
                }

                if (hasAttribute) {
                    VertexGeometryBuilder builder;
                    bool spliceSuccess = builder.SpliceGeometry(0, 0, vg);

                    if (spliceSuccess) {
                        VertexGeometry::VertexDataType dt = builder.GetVertexElementFormat(usage, usageIndex);

                        bool changed = false;
                        if (((dt >= VertexGeometry::Float32_1) && (dt <= VertexGeometry::Float32_4)) && (precision == HalfFloatConversion)) {
                            changed = true;
                            builder.SetVertexElementFormat(usage, usageIndex,
                                                           static_cast<VertexGeometry::VertexDataType>(VertexGeometry::Float16_1 + (dt - VertexGeometry::Float32_1)));
                        }
                        else {
                            if (((dt >= VertexGeometry::Float16_1) && (dt <= VertexGeometry::Float16_4)) && (precision == FloatConversion)) {
                                changed = true;
                                builder.SetVertexElementFormat(usage, usageIndex,
                                                               static_cast<VertexGeometry::VertexDataType>(VertexGeometry::Float32_1 + (dt - VertexGeometry::Float16_1)));
                            }
                        }

                        if (changed) {
                            VertexGeometry* newVg = builder.GetVertexGeometry();

                            if (newVg != 0) {
                                result = VertexBuffer::Create();
                                result->SetPrimitiveType(vb->GetPrimitiveType());
                                static_cast<void>(result->SetVertexGeometry(newVg, VertexBuffer::VertexGeometryDisposer::Dispose));
                                result->SetDisposedAfterUpload(vb->IsDisposedAfterUpload());
                            }
                        }
                    }
                }
            }
        }

        return result;
    }

    void Math3D::NormalizeRectangle(const RenderTarget& renderTarget, const Rectangle& renderTargetSpaceRectangle, Rectangle& normalizedRectangle)
    {
        Int width = renderTarget.GetWidth();
        Int height = renderTarget.GetHeight();
        Float widthNormalizationFactor = (0 != width) ? (1.0F / Float(width)) : 0.0F;
        Float heightNormalizationFactor = (0 != height) ? (1.0F / Float(height)) : 0.0F;
        normalizedRectangle.Set(renderTargetSpaceRectangle.GetLeft() * widthNormalizationFactor, renderTargetSpaceRectangle.GetTop() * heightNormalizationFactor,
            renderTargetSpaceRectangle.GetWidth() * widthNormalizationFactor, renderTargetSpaceRectangle.GetHeight() * heightNormalizationFactor);
    }
} // namespace Candera

