//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexGeometryTools.h"
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Mathematics/Math.h>

CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(753, CANDERA_LINT_REASON_TEMPLATEINSTANCESUSED)

namespace Candera {
    namespace Internal {
        namespace Warning {
            template<class T> void ignore( const T& ) { }
        }
        /**
        *  Base converter class.
        */
        class VertexGeometryTools::BaseBufferConverter
        {
        public:
            virtual void Convert(
                void* dstBuffer, UInt8 dstChannelCount, UInt16 dstStride,
                const void* srcBuffer, UInt8 srcChannelCount, UInt16 srcStride,
                UInt32 itemCount) const = 0;
        };

        namespace {

            const VertexGeometryTools::DataTypeTraits c_dataTypeTraits[] = {
                { VertexGeometryTools::Float32Type, 1, },  ///< Float32_1
                { VertexGeometryTools::Float32Type, 2, },  ///< Float32_2
                { VertexGeometryTools::Float32Type, 3, },  ///< Float32_3
                { VertexGeometryTools::Float32Type, 4, },  ///< Float32_4
                { VertexGeometryTools::UInt8Type, 1, },    ///< UInt8_1
                { VertexGeometryTools::UInt8Type, 1, },    ///< UInt8_1_N
                { VertexGeometryTools::UInt8Type, 2, },    ///< UInt8_2
                { VertexGeometryTools::UInt8Type, 2, },    ///< UInt8_2_N
                { VertexGeometryTools::UInt8Type, 3, },    ///< UInt8_3
                { VertexGeometryTools::UInt8Type, 3, },    ///< UInt8_3_N
                { VertexGeometryTools::UInt8Type, 4, },    ///< UInt8_4
                { VertexGeometryTools::UInt8Type, 4, },    ///< UInt8_4_N
                { VertexGeometryTools::Int8Type, 1, },     ///< Int8_1
                { VertexGeometryTools::Int8Type, 1, },     ///< Int8_1_N
                { VertexGeometryTools::Int8Type, 2, },     ///< Int8_2
                { VertexGeometryTools::Int8Type, 2, },     ///< Int8_2_N
                { VertexGeometryTools::Int8Type, 3, },     ///< Int8_3
                { VertexGeometryTools::Int8Type, 3, },     ///< Int8_3_N
                { VertexGeometryTools::Int8Type, 4, },     ///< Int8_4
                { VertexGeometryTools::Int8Type, 4, },     ///< Int8_4_N
                { VertexGeometryTools::UInt16Type, 1, },   ///< UInt16_1
                { VertexGeometryTools::UInt16Type, 1, },   ///< UInt16_1_N
                { VertexGeometryTools::UInt16Type, 2, },   ///< UInt16_2
                { VertexGeometryTools::UInt16Type, 2, },   ///< UInt16_2_N
                { VertexGeometryTools::UInt16Type, 3, },   ///< UInt16_3
                { VertexGeometryTools::UInt16Type, 3, },   ///< UInt16_3_N
                { VertexGeometryTools::UInt16Type, 4, },   ///< UInt16_4
                { VertexGeometryTools::UInt16Type, 4, },   ///< UInt16_4_N
                { VertexGeometryTools::Int16Type, 1, },    ///< Int16_1
                { VertexGeometryTools::Int16Type, 1, },    ///< Int16_1_N
                { VertexGeometryTools::Int16Type, 2, },    ///< Int16_2
                { VertexGeometryTools::Int16Type, 2, },    ///< Int16_2_N
                { VertexGeometryTools::Int16Type, 3, },    ///< Int16_3
                { VertexGeometryTools::Int16Type, 3, },    ///< Int16_3_N
                { VertexGeometryTools::Int16Type, 4, },    ///< Int16_4
                { VertexGeometryTools::Int16Type, 4, },    ///< Int16_4_N
                { VertexGeometryTools::Float16Type, 1, },  ///< Float16_1
                { VertexGeometryTools::Float16Type, 2, },  ///< Float16_2
                { VertexGeometryTools::Float16Type, 3, },  ///< Float16_3
                { VertexGeometryTools::Float16Type, 4, },  ///< Float16_4
                { VertexGeometryTools::UInt32Type, 1, },   ///< UInt32_1
                { VertexGeometryTools::UInt32Type, 1, },   ///< UInt32_1_N
                { VertexGeometryTools::UInt32Type, 2, },   ///< UInt32_2
                { VertexGeometryTools::UInt32Type, 2, },   ///< UInt32_2_N
                { VertexGeometryTools::UInt32Type, 3, },   ///< UInt32_3
                { VertexGeometryTools::UInt32Type, 3, },   ///< UInt32_3_N
                { VertexGeometryTools::UInt32Type, 4, },   ///< UInt32_4
                { VertexGeometryTools::UInt32Type, 4, },   ///< UInt32_4_N
                { VertexGeometryTools::Int32Type, 1, },    ///< Int32_1
                { VertexGeometryTools::Int32Type, 1, },    ///< Int32_1_N
                { VertexGeometryTools::Int32Type, 2, },    ///< Int32_2
                { VertexGeometryTools::Int32Type, 2, },    ///< Int32_2_N
                { VertexGeometryTools::Int32Type, 3, },    ///< Int32_3
                { VertexGeometryTools::Int32Type, 3, },    ///< Int32_3_N
                { VertexGeometryTools::Int32Type, 4, },    ///< Int32_4
                { VertexGeometryTools::Int32Type, 4, },    ///< Int32_4_N
                { VertexGeometryTools::Float32Type, 9, },  ///< Float32_3x3
                { VertexGeometryTools::Float32Type, 16, }, ///< Float32_4x4
            };

            /**
             *  Map BaseTypes to actual C++ types.
             */
            template <VertexGeometryTools::BaseType T> struct BaseTypeHost
            {
            };
            template <> struct BaseTypeHost < VertexGeometryTools::Float32Type >
            {
                typedef Float Type;
            };
            template <> struct BaseTypeHost < VertexGeometryTools::UInt8Type >
            {
                typedef UInt8 Type;
            };
            template <> struct BaseTypeHost < VertexGeometryTools::Int8Type >
            {
                typedef Int8 Type;
            };
            template <> struct BaseTypeHost < VertexGeometryTools::UInt16Type >
            {
                typedef UInt16 Type;
            };
            template <> struct BaseTypeHost < VertexGeometryTools::Int16Type >
            {
                typedef Int16 Type;
            };
            template <> struct BaseTypeHost < VertexGeometryTools::Float16Type >
            {
                typedef UInt16 Type;
            };
            template <> struct BaseTypeHost < VertexGeometryTools::UInt32Type >
            {
                typedef UInt32 Type;
            };
            template <> struct BaseTypeHost < VertexGeometryTools::Int32Type >
            {
                typedef Int32 Type;
            };

            /**
             *  Map BaseTypes to a few useful properties.
             */
            template <VertexGeometryTools::BaseType T>
            struct BaseTypeTraits
            {
                // C++ type of the VertexGeometryTools::BaseType.
                typedef typename BaseTypeHost<T>::Type HostType;
                static const UInt8 c_byteSize = sizeof(HostType);

                static const HostType& Zero()
                {
                    static const HostType object(static_cast<HostType>(0));
                    return object;
                }
                static const HostType& One()
                {
                    static const HostType object(static_cast<HostType>(1));
                    return object;
                }
            };

            /**
             *  Convert a value from one VertexGeometryTools::BaseType to another.
             */
            template <VertexGeometryTools::BaseType DstType, VertexGeometryTools::BaseType SrcType>
            class BaseTypeConverter
            {
            public:
                static void Convert(void *typedDstBuffer, const void* typedSrcBuffer)
                {
                    *FeatStd::Internal::PointerToPointer<typename BaseTypeTraits<DstType>::HostType*>(typedDstBuffer) =
                        static_cast<typename BaseTypeTraits<DstType>::HostType> (
                        *FeatStd::Internal::PointerToPointer<const typename BaseTypeTraits<SrcType>::HostType*>(typedSrcBuffer));
                }
            };

#ifndef FEATSTD_FIXED_POINT_ARITHMETIC
            template <>
            class BaseTypeConverter < VertexGeometryTools::Float16Type, VertexGeometryTools::Float16Type >
            {
            public:
                static void Convert(void *typedDstBuffer, const void* typedSrcBuffer)
                {
                    Warning::ignore(Convert); // suppresses warning: function was declared but never referenced
                    using FeatStd::Internal::PointerToPointer;
                    *PointerToPointer<BaseTypeTraits<VertexGeometryTools::Float16Type>::HostType*>(typedDstBuffer) =
                        *PointerToPointer<const BaseTypeTraits<VertexGeometryTools::Float16Type>::HostType*>(typedSrcBuffer);
                }
            };

            template <VertexGeometryTools::BaseType SrcType>
            class BaseTypeConverter < VertexGeometryTools::Float16Type, SrcType >
            {
            public:
                static void Convert(void *typedDstBuffer, const void* typedSrcBuffer)
                {
                    const BaseTypeTraits<VertexGeometryTools::Float32Type>::HostType tmp =
                        static_cast<BaseTypeTraits<VertexGeometryTools::Float32Type>::HostType> (
                        *FeatStd::Internal::PointerToPointer<const typename BaseTypeTraits<SrcType>::HostType*>(typedSrcBuffer));
                    *FeatStd::Internal::PointerToPointer<BaseTypeTraits<VertexGeometryTools::Float16Type>::HostType*>(typedDstBuffer) = 
                        Math3D::ConvertFloatToHalfFloat(tmp);
                }
            };

            template <VertexGeometryTools::BaseType DstType>
            class BaseTypeConverter < DstType, VertexGeometryTools::Float16Type >
            {
            public:
                static void Convert(void *typedDstBuffer, const void* typedSrcBuffer)
                {
                    const BaseTypeTraits<VertexGeometryTools::Float16Type>::HostType tmp =
                        static_cast<BaseTypeTraits<VertexGeometryTools::Float16Type>::HostType> (
                        *FeatStd::Internal::PointerToPointer<const BaseTypeTraits<VertexGeometryTools::Float16Type>::HostType*>(typedSrcBuffer));

                    BaseTypeTraits<VertexGeometryTools::Float32Type>::HostType floatValue = Math3D::ConvertHalfFloatToFloat(tmp);

                    *FeatStd::Internal::PointerToPointer<typename BaseTypeTraits<DstType>::HostType*>(typedDstBuffer) = 
                        static_cast<typename BaseTypeTraits<DstType>::HostType>(floatValue);
                }
            };
#endif
            /**
             *  Convert a value from one VertexGeometry::VertexDataType to another.
             * when the destination is smaller than the source.
             */
            template <VertexGeometryTools::BaseType DstType, VertexGeometryTools::BaseType SrcType>
            class ItemConverter
            {
            public:
                static void Convert(UInt8* typedDstBuffer, const UInt8* typedSrcBuffer, UInt8 channelCount)
                {
                    for (UInt32 j = 0; j < channelCount; j++) {
                        BaseTypeConverter<DstType, SrcType>::Convert(typedDstBuffer, typedSrcBuffer);
                        typedDstBuffer += BaseTypeTraits<DstType>::c_byteSize;
                        typedSrcBuffer += BaseTypeTraits<SrcType>::c_byteSize;
                    }
                }
            };

            /**
             *  Specialization for when no conversion is needed.
             */
            template <VertexGeometryTools::BaseType Type>
            class ItemConverter < Type, Type >
            {
            public:
                static void Convert(UInt8* typedDstBuffer, const UInt8* typedSrcBuffer, UInt8 channelCount)
                {
                    MemoryPlatform::Copy(typedDstBuffer, typedSrcBuffer, static_cast<SizeType>(channelCount * BaseTypeTraits<Type>::c_byteSize));
                }
            };

            /**
             *  Convert a value from one VertexGeometry::VertexDataType to another.
             * when the source is smaller than the destination and pad some
             * constant values.
             */
            template <VertexGeometryTools::BaseType DstType, VertexGeometryTools::BaseType SrcType>
            class ItemPadConverter
            {
            public:
                static void Convert(UInt8* typedDstBuffer, UInt8 dstChannelCount,
                    const UInt8* typedSrcBuffer, UInt8 srcChannelCount)
                {
                    ItemConverter<DstType, SrcType>::Convert(typedDstBuffer, typedSrcBuffer, srcChannelCount);
                    typedDstBuffer += srcChannelCount * BaseTypeTraits<DstType>::c_byteSize;
                    for (UInt32 j = srcChannelCount; j < dstChannelCount; j++) {
                        MemoryPlatform::Copy(
                            typedDstBuffer,
                            (j < 3) ? &BaseTypeTraits<DstType>::Zero() : &BaseTypeTraits<DstType>::One(),
                            BaseTypeTraits<DstType>::c_byteSize);
                        typedDstBuffer += BaseTypeTraits<DstType>::c_byteSize;
                    }
                }
            };

            /**
             *  Convert full buffers.
             */
            template <VertexGeometryTools::BaseType DstType, VertexGeometryTools::BaseType SrcType>
            class BufferConverter : public VertexGeometryTools::BaseBufferConverter
            {
            public:
                virtual void Convert(
                    void* dstBuffer, UInt8 dstChannelCount, UInt16 dstStride,
                    const void* srcBuffer, UInt8 srcChannelCount, UInt16 srcStride,
                    UInt32 itemCount) const
                {
                    UInt8* typedDstBuffer = FeatStd::Internal::PointerToPointer<UInt8*>(dstBuffer);
                    const UInt8* typedSrcBuffer = FeatStd::Internal::PointerToPointer<const UInt8*>(srcBuffer);
                    if (dstChannelCount <= srcChannelCount) {
                        for (UInt32 i = 0; i < itemCount; i++) {
                            ItemConverter<DstType, SrcType>::Convert(
                                typedDstBuffer,
                                typedSrcBuffer, dstChannelCount);
                            typedDstBuffer += dstStride;
                            typedSrcBuffer += srcStride;
                        }
                    }
                    else {
                        for (UInt32 i = 0; i < itemCount; i++) {
                            ItemPadConverter<DstType, SrcType>::Convert(
                                typedDstBuffer, dstChannelCount,
                                typedSrcBuffer, srcChannelCount);

                            typedDstBuffer += dstStride;
                            typedSrcBuffer += srcStride;
                        }
                    }
                }

                static const VertexGeometryTools::BaseBufferConverter* Get()
                {
                    static BufferConverter<DstType, SrcType> object = BufferConverter<DstType, SrcType>();
                    return &object;
                }
            };

            /**
             *  Helper class to make dynamic choosing of Converter easier.
             */
            class BaseBufferConverterChooser
            {
            public:
                virtual const VertexGeometryTools::BaseBufferConverter * GetBufferConverter(VertexGeometryTools::BaseType secondType) const = 0;
            };

            /**
             *  Helper class to make dynamic choosing of Converter easier.
             * Creates a two dimensional table to map every base type
             * to other base types.
             */
            template <VertexGeometryTools::BaseType FirstType>
            class BufferConverterChooser : public BaseBufferConverterChooser
            {
            public:
                virtual const VertexGeometryTools::BaseBufferConverter * GetBufferConverter(VertexGeometryTools::BaseType secondType) const override
                {
                    FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: Only getter, no side effect")
                        static const VertexGeometryTools::BaseBufferConverter* const table[VertexGeometryTools::BaseTypeCount] = {
                        BufferConverter<FirstType, VertexGeometryTools::Float32Type>::Get(),
                        BufferConverter<FirstType, VertexGeometryTools::UInt8Type  >::Get(),
                        BufferConverter<FirstType, VertexGeometryTools::Int8Type  >::Get(),
                        BufferConverter<FirstType, VertexGeometryTools::UInt16Type >::Get(),
                        BufferConverter<FirstType, VertexGeometryTools::Int16Type  >::Get(),
                        BufferConverter<FirstType, VertexGeometryTools::Float16Type>::Get(),
                        BufferConverter<FirstType, VertexGeometryTools::UInt32Type >::Get(),
                        BufferConverter<FirstType, VertexGeometryTools::Int32Type  >::Get() };
                    return table[secondType];
                }
                static const BaseBufferConverterChooser* Get()
                {
                    static BufferConverterChooser<FirstType> object = BufferConverterChooser<FirstType>();
                    return &object;
                }
            };

            /**
             *  Dynamically choose the statical Converter.
             */
            const VertexGeometryTools::BaseBufferConverter* GetBufferConverter(VertexGeometryTools::BaseType firstType, VertexGeometryTools::BaseType secondType)
            {
                FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: only getter, no side effect")
                    static const BaseBufferConverterChooser* const table[VertexGeometryTools::BaseTypeCount] = {
                    BufferConverterChooser<VertexGeometryTools::Float32Type>::Get(),
                    BufferConverterChooser<VertexGeometryTools::UInt8Type  >::Get(),
                    BufferConverterChooser<VertexGeometryTools::Int8Type  >::Get(),
                    BufferConverterChooser<VertexGeometryTools::UInt16Type >::Get(),
                    BufferConverterChooser<VertexGeometryTools::Int16Type  >::Get(),
                    BufferConverterChooser<VertexGeometryTools::Float16Type>::Get(),
                    BufferConverterChooser<VertexGeometryTools::UInt32Type >::Get(),
                    BufferConverterChooser<VertexGeometryTools::Int32Type  >::Get()
                };

                return table[firstType]->GetBufferConverter(secondType);
            }
        }

        VertexGeometryTools::Converter::Converter() :
            m_baseConverter(0)
        {
        }

        VertexGeometryTools::Converter::~Converter()
        {
            m_baseConverter = 0;
        }

        void VertexGeometryTools::Converter::ConvertBuffer(
            void* dstBuffer, UInt8 dstCount, UInt16 dstStride,
            const void* srcBuffer, UInt8 srcCount, UInt16 srcStride,
            UInt32 itemCount) const
        {
            if (m_baseConverter != 0) {
                m_baseConverter->Convert(
                    dstBuffer, dstCount, dstStride,
                    srcBuffer, srcCount, srcStride,
                    itemCount);
            }
        }

        void VertexGeometryTools::Converter::ConvertItem(
            void* dstBuffer, UInt8 dstCount,
            const void* srcBuffer, UInt8 srcCount) const
        {
            if (m_baseConverter != 0) {
                m_baseConverter->Convert(
                    dstBuffer, dstCount, 0,
                    srcBuffer, srcCount, 0,
                    1);
            }
        }

        VertexGeometryTools::Converter::Converter(BaseType dstType, BaseType srcType) :
            m_baseConverter(GetBufferConverter(dstType, srcType))
        {
        }

        VertexGeometryTools::DataTypeTraits VertexGeometryTools::GetTypeTraits(VertexGeometry::VertexDataType type)
        {
            FEATSTD_DEBUG_ASSERT((static_cast<Int32>(type) >= 0) && 
                (static_cast<UInt32>(type) < (sizeof(c_dataTypeTraits) / sizeof(c_dataTypeTraits[0]))));
            return c_dataTypeTraits[type];
        }

        VertexGeometryTools::Converter VertexGeometryTools::GetConverter(BaseType dstType, BaseType srcType)
        {
            return Converter(dstType, srcType);
        }
    }
} // namespace
