//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VERTEXGEOMETRYBUILDER_H)
#define CANDERA_VERTEXGEOMETRYBUILDER_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>

namespace Candera {

/**  @addtogroup Core3D
 *   @{
 */

/**
 *  @brief   The VertexGeometryBuilder helps effectively build VertexGeometry objects.
 *
 *  VertexGeometryBuilder is usefully when creating procedural geometry.
 *  E.g. : a simple triangle:
 *   builder.SetVertexElement(VertexGeometry::Position, 0, 0.f, 0.f, 0.f);
 *   builder.SetVertexElement(VertexGeometry::Position, 0, 0.f, 1.f, 0.f);
 *   builder.SetVertexElement(VertexGeometry::Position, 0, 0.f, 0.f, 1.f);
 *   vertexGeometry = builder.GetVertexGeometry();
 *  Usefully for removing or adding vertex elements.
 *  E.g. : remove normals from a geometry and replace them by vertex color.
 *   builder.SpliceGeometry(0, 0, geometryWithNormal);
 *   builder.RemoveVertexElement(VertexGeometry::Normal, 0);
 *   builder.SetVertexElementRange(VertexGeometry::Color, 0, 0, VertexGeometry::Float32, itemNum, itemStride, colorBuffer);
 *   geometryWithColor = builder.GetGeometry();
 *  Usefully for concatenating geometries.
 *  E.g. : concatenate two vertex geometries.
 *   builder.SpliceGeometry(0, 0, vertexGeometry1);
 *   builder.SpliceGeometry(vertexGeometry1->GetVertexCount(), vertexGeometry1->GetIndexCount(), vertexGeometry2);
 *   vertexGeometry1cat2 = builder.GetGeometry();
 *  Usefully for random manipulation of geometry.
 *  E.g. : Change the position of the 3rd vertex;
 *   builder.SetVertexCursor(2);
 *   builder.SetVertexElement(VertexGeometry::Position, 0, 0.f, 0.f, 0.f);
 */
class VertexGeometryBuilder
{
    public:
        typedef MemoryManagement::AdaptedArrayDisposer<const void*, const UInt8*> AdaptedVertexGeometryDisposer;
        typedef MemoryManagement::ArrayDisposer<const void*> Disposer;
        typedef Disposer::DisposerFunction DisposerFn;

        /**
         *  Constructor
         */
        VertexGeometryBuilder();

        /**
         *  Destructor
         */
        ~VertexGeometryBuilder();

        // Vertex manipulation.

        /**
         *  Set the position of the vertex cursor. Vertices set by SetVertexElement
         *  are added from the vertex cursor position.
         *  @param offset position in the vertex array where to splice the provided vertex definitions.
         */
        void SetVertexCursor(UInt32 offset = 0);

        /*
         *  Get the position of the vertex cursor
         *  @return position of the vertex cursor
         */
        UInt32 GetVertexCursor() const;

        /**
         *  Increment the vertex cursor.
         *  @param steps number of steps by which to increment the vertex cursor.
         */
        void IncrementVertexCursor(UInt32 steps = 1);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x, Float y);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         *  @param z Z component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x, Float y, Float z);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         *  @param z Z component of this value.
         *  @param w W component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x, Float y, Float z, Float w);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x, Int y);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         *  @param z Z component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x, Int y, Int z);

        /**
         *  Define the value of a vertex element.
         *  All element definitions are attached to the current vertex.
         *  Missing vertex elements take the value of the previous vertex.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         *  @param z Z component of this value.
         *  @param w W component of this value.
         */
        void SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x, Int y, Int z, Int w);

        /**
         *  Splice values to a vertex element, starting with the given vertex.
         *  The buffer is copied internally by the builder, so it can be destroyed as soon as
         *  the method returns.
         *  @param usage Usage type of the vertex element.
         *  @param usageIndex Usage index of the vertex element.
         *  @param offset Vertex from which to begin splicing.
         *  @param type Value type of the vertex element.
         *  @param count Number of vertices to splice.
         *  @param stride Distance between consecutive vertex element values in the source buffer.
         *  @param buffer Buffer containing the spliced values.
         *  @return true if the splice is successful.
         */
        bool SetVertexElementRange(
            VertexGeometry::VertexUsage usage,
            UInt8 usageIndex,
            UInt32 offset,
            VertexGeometry::VertexDataType type,
            UInt32 count,
            UInt16 stride,
            const void *buffer);

        /**
         *  Splice values to a vertex element, starting with the given vertex.
         *  The buffer disposed using the provided disposer, as soon as the builder
         *  decides it is no longer needed. This can happen past the boundaries of this
         *  method. If the buffer does not need disposing, or predictions can be made
         *  to when the builder no longer uses the buffer (i.e. the builder was destroyed
         *  or cleared), than disposer may be 0.
         *  @param usage        Usage type of the vertex element.
         *  @param usageIndex   Usage index of the vertex element.
         *  @param offset       Vertex from which to begin splicing.
         *  @param type         Value type of the vertex element.
         *  @param count        Number of vertices to splice.
         *  @param stride       Distance between consecutive vertex element values in the buffer.
         *  @param buffer       Buffer containing the spliced values.
         *  @param disposerFn   Disposer function used to dispose the buffer.
         *  @return             true if the splice is successful.
         */
        bool SetVertexElementRange(VertexGeometry::VertexUsage usage,
                                   UInt8 usageIndex,
                                   UInt32 offset,
                                   VertexGeometry::VertexDataType type,
                                   UInt32 count,
                                   UInt16 stride,
                                   const void* buffer,
                                   DisposerFn disposerFn);

        //Index manipulation

        /**
         *  Set the position of the index cursor. Indices set by SetIndexElement
         *  are added from the index cursor position.
         *  @param offset position in the index array where to splice the provided index definitions.
         */
        void SetIndexCursor(UInt32 offset = 0);
        /*
         *  Get the position of the index cursor
         *  @return position of the index cursor
         */
        UInt32 GetIndexCursor() const;
        /**
         *  Increment the index cursor.
         *  @param steps number of steps by which to increment the index cursor.
         */
        void IncrementIndexCursor(UInt32 steps = 1);

        /**
         *  Define the value of an index.
         *  @param i Index value.
         */
        void SetIndexElement(UInt32 i);

        /**
         *  Splice values to the index buffer, starting with the given index.
         *  The buffer is copied internally by the builder, so it can be destroyed as soon as
         *  the method returns.
         *  @param offset Index from which to begin splicing.
         *  @param count Number of indices to splice.
         *  @param buffer Buffer containing the spliced values.
         *  @return true if the splice is successful.
         */
        bool SetIndexElementRange(
            UInt32 offset,
            UInt32 count,
            const UInt16 *buffer);

        /**
         *  Splice values to the index buffer, starting with the given index.
         *  The buffer disposed using the provided disposer, as soon as the builder
         *  decides it is no longer needed. This can happen past the boundaries of this
         *  method. If the buffer does not need disposing, or predictions can be made
         *  to when the builder no longer uses the buffer (i.e. the builder was destroyed
         *  or cleared), than disposer may be 0.
         *  @param offset       Index from which to begin splicing.
         *  @param count        Number of indices to splice.
         *  @param buffer       Buffer containing the spliced values.
         *  @param disposerFn   Disposer function used to dispose the buffer.
         *  @return             true if the splice is successful.
         */
        bool SetIndexElementRange(UInt32 offset,
                                  UInt32 count,
                                  const void* buffer,
                                  DisposerFn disposerFn);

        //Buffer manipulation

        /**
         *  Splice the values associated with each vertex element,
         *  and with the index data, respectively.
         *  Values are set starting with the given vertex and index cursors.
         *  Indices are adjusted such that they reference the vertices being set by the current splice.
         *  The geometry is copied internally by the builder, so it can be destroyed as soon as
         *  the method returns.
         *  @param vertexOffset Vertex from which to begin splicing.
         *  @param indexOffset Index from which to begin splicing.
         *  @param geometry Vertex geometry object containing the spliced values.
         *  @return true if the splice is successful.
         */
        bool SpliceGeometry(
            UInt32 vertexOffset,
            UInt32 indexOffset,
            const VertexGeometry* geometry);

        /**
         *  Remove all data associated with a vertex element.
         *  @param usage Usage type of the vertex element.
         *  @param usageIndex Usage index of the vertex element.
         */
        void RemoveVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex);

        /**
         *  Remove all data associated with a vertex usage.
         *  @param usage Usage type of the vertex element.
         */
        void RemoveVertexElements(VertexGeometry::VertexUsage usage);

        /**
         *  Clear all data stored by the builder. This effectively reverts the object
         *  to the same state as when it was constructed.
         */
        void Clear();

        // State manipulation.

        /**
         *  Force the data type used when outputting the Vertex Geometry.
         *  @param usage Usage type of the vertex element.
         *  @param usageIndex Usage index of the vertex element.
         *  @param type Value type of the vertex element.
         */
        void SetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex, VertexGeometry::VertexDataType type);

        /**
         *  Force the maximum size allowed for the vertex data.
         *  If the client sets more vertices to the builder they would be ignored upon creation
         *  of the Vertex Geometry.
         *  @param size Maximum allowed size.
         */
        void SetMaxVertexCount(UInt32 size);

        /**
         *  Force the maximum size allowed for the index data.
         *  If the client sets more indices to the builder they would be ignored upon creation
         *  of the Vertex Geometry.
         *  @param size Maximum allowed size.
         */
        void SetMaxIndexCount(UInt32 size);

        /**
         *  Set the memory pool of the exported VertexGeometry.
         *  @param memoryPool Memory pool set to the exported VertexGeometry.
         */
        void SetMemoryPool(VertexGeometry::MemoryPool memoryPool);
        /**
         *  Set the buffer type of the exported VertexGeometry.
         *  @param bufferType Buffer type set to the exported VertexGeometry.
         */
        void SetBufferType(VertexGeometry::BufferType bufferType);
        /**
         *  Set the buffer usage of the exported VertexGeometry.
         *  This also enables and disables the export of index data.
         *  @param bufferUsage Buffer usage set to the exported VertexGeometry.
         */
        void SetBufferUsage(VertexGeometry::BufferUsage bufferUsage);

        // State query.

        /**
         *  Retrieve the data type used when outputting the Vertex Geometry.
         *  @param usage Usage type of the vertex element.
         *  @param usageIndex Usage index of the vertex element.
         *  @return Value type of the vertex element.
         */
        VertexGeometry::VertexDataType GetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const;

        /**
         *  Retrieve the data type used when index buffer is put out.
         *  @return Type of indices. Can be one of UInt8_1, UInt16_1 or UInt32_1.
         */
        VertexGeometry::VertexDataType GetIndexDataType() const;

        /**
         *  Retrieve the maximum size allowed for the vertex data.
         *  @return Maximum allowed size.
         */
        UInt32 GetMaxVertexCount() const;

        /**
         *  Retrieve the maximum size allowed for the index data.
         *  @return Maximum allowed size.
         */
        UInt32 GetMaxIndexCount() const;

        /**
         *  Retrieve the memory pool of the exported VertexGeometry.
         *  @return Memory pool set to the exported VertexGeometry.
         */
        VertexGeometry::MemoryPool GetMemoryPool() const;
        /**
         *  Retrieve the buffer type of the exported VertexGeometry.
         *  @return Buffer type set to the exported VertexGeometry.
         */
        VertexGeometry::BufferType GetBufferType() const;
        /**
         *  Retrieve the buffer usage of the exported VertexGeometry.
         *  @return Buffer usage set to the exported VertexGeometry.
         */
        VertexGeometry::BufferUsage GetBufferUsage() const;

        /**
         *  Retrieve the current vertex count.
         *  @return Current count.
         */
        UInt32 GetVertexCount() const;

        /**
         *  Retrieve the current index count.
         *  @return Current count.
         */
        UInt32 GetIndexCount() const;

        /**
         *  Retrieve the current size of the vertex data in bytes.
         *  @return Current size.
         */
        UInt32 GetVertexDataSize() const;

        /**
         *  Retrieve the current stride of the exported VertexGeometry.
         *  @return Current size.
         */
        UInt16 GetVertexStride() const;

        /**
         *  Retrieve the count of vertex elements the exported VertexGeometry.
         *  @return Current size.
         */
        UInt16 GetVertexElementCount() const;

        /**
         *  Checks whether all the created arrays have had all their items
         *  set to a user provided value, be they vertex elements, or indices.
         *  @return false if there are some default items, true otherwise.
         */
        bool IsComplete() const;

        // Vertex Geometry building.

        /**
         *  Retrieve the Vertex Geometry. The buffers provided must be large enough
         *  to store the data, as described by the state getter methods.
         *  (i.e. GetVertexSize() bytes for the vertex buffer,
         *  GetVertexElementCount() VertexElementFormat objects for format buffer and
         *  GetIndexCount() UInt16 objects for the index buffer.).
         *  If one of the pointers is 0, that buffer is not computed.
         *  If the builder doesn't have index data, indexBuffer should be 0.
         *  @param vertexData address of a buffer to store vertex data.
         *  @param vertexElementFormat address of a buffer to store vertex element format data.
         *  @param indexData address of a buffer to store index data.
         *  @return true if the vertex geometry was generated successfully.
         */
        bool GetVertexGeometry(
            void* vertexData,
            VertexGeometry::VertexElementFormat* vertexElementFormat,
            UInt16* indexData) const;

        /**
         *  Retrieve the Vertex Geometry. The buffers provided must be large enough
         *  to store the data, as described by the state getter methods.
         *  (i.e. GetVertexSize() bytes for the vertex buffer,
         *  GetVertexElementCount() VertexElementFormat objects for format buffer and
         *  GetIndexCount() * size of GetIndexDataType() for the index buffer.).
         *  If one of the pointers is 0, that buffer is not computed.
         *  If the builder doesn't have index data, indexBuffer should be 0.
         *  @param vertexData address of a buffer to store vertex data.
         *  @param vertexElementFormat address of a buffer to store vertex element format data.
         *  @param indexData address of a buffer to store index data.
         *  @return true if the vertex geometry was generated successfully.
         */
        bool GetVertexGeometry(
            void* vertexData,
            VertexGeometry::VertexElementFormat* vertexElementFormat,
            void* indexData) const;

        /**
         *  Retrieve the vertex data for the built Vertex Geometry.
         *  The buffer returned by this method should be released by the client using
         *  MemoryManagement::AdaptedArrayDisposer<void*, UInt8*>.
         *  @return pointer to a buffer containing vertex data.
         */
        void* GetVertexData() const;

        /**
         *  Retrieve the vertex data associated to a vertex element.
         *  The data is tightly packaged as objects of the format returned by
         *  GetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex).
         *  The item count is the same as returned by GetVertexCount().
         *  The buffer returned by this method should be released by the client using
         *  MemoryManagement::AdaptedArrayDisposer<void*, UInt8*>.
         *  @param usage      VertexUsage for the element.
         *  @param usageIndex UsageIndex for the element.
         *  @return pointer to a buffer containing vertex data.
         *  NOTE: if there is a vertex geometry containing a vertex element,
         *  it is more optimal to set that full VertexGeometry buffer as a vertex
         *  element using SetVertexElmentRange, and a proper stride,
         *  then by calling SpliceGeometry() and GetVertexElement(), and then
         *  setting this buffer with SetVertexElementRange.
         */
        void* GetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const;

        /**
         *  Retrieve the vertex element format data for the built Vertex Geometry.
         *  The buffer returned by this method should be released by the client using
         *  MemoryManagement::CanderaArrayDisposer<VertexGeometry::VertexElementFormat*>.
         *  @return pointer to a buffer containing vertex element format data.
         */
        VertexGeometry::VertexElementFormat* GetVertexElementFormat() const;

        /**
         *  Retrieve the index data for the built Vertex Geometry.
         *  The buffer returned by this method should be released by the client using
         *  MemoryManagement::CanderaArrayDisposer<UInt16*>.
         *  @return pointer to a buffer containing index data.
         */
        UInt16* GetIndexData() const;

        /**
        *  Retrieve the index data for the built Vertex Geometry.
        *  The buffer returned by this method should be released by the client using
        *  MemoryManagement::AdaptedArrayDisposer<void*, UInt(8/16/32)*>, depending
        *  on the data type requested for the indices.
        *  @return pointer to a buffer containing index data.
        */
        void* GetIndexData(VertexGeometry::VertexDataType indexDataType) const;

        /**
         *  Build a full Vertex Geometry object.
         *  The object returned by this method should be released by the client using
         *  MemoryManagement::CanderaDisposer<VertexGeometry*>.
         *  @return pointer Vertex Geometry object.
         */
        VertexGeometry* GetVertexGeometry() const;


        class StoreBuffer;
        class ElementBuffer;
        class Element;
        class VertexData;
        class IndexData;

    private:

        bool m_isValid;
        bool m_isUserMemoryPool;
        bool m_isUserBufferType;
        bool m_isUserBufferUsage;

        VertexGeometry::MemoryPool m_memoryPool;
        VertexGeometry::BufferType m_bufferType;
        VertexGeometry::BufferUsage m_bufferUsage;

        UInt32 m_maxVertexCount;
        UInt32 m_maxIndexCount;

        UInt32 m_vertexCursor;
        UInt32 m_indexCursor;

        StoreBuffer* m_store;

        VertexData* m_vertexData;
        IndexData* m_indexData;
};

/**  @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_VERTEXGEOMETRYBUILDER_H

