//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexGeometryBuilder.h"
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/Core/VertexGeometryAccessors.h>
#include <Candera/Engine3D/Core/VertexGeometryTools.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/Macros.h>

CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(753, CANDERA_LINT_REASON_TEMPLATEINSTANCESUSED)

namespace Candera {
    using namespace MemoryManagement;

    /**
     *  StoreBuffer is used to manage buffers within the vertex geometry builder.
     *  A buffer may be used by multiple elements, or it can be split over multiple
     *  element buffers by overlapping data of different type.
     */
    class VertexGeometryBuilder::StoreBuffer
    {
    public:

        // Methods used by VertexGeometryBuilder to create a base StoreBuffer.
        static StoreBuffer* CreateRootBuffer();
        static void DestroyBuffer(StoreBuffer* store);

        // Methods used for reference counting and safe destruction of StoreBuffers
        //left unused.
        static void Acquire(StoreBuffer *store);
        static void Release(StoreBuffer *store);

        // Methods used for creating StoreBuffers.
        // Creates a mutable store buffer.
        StoreBuffer* CreateBuffer(
            VertexGeometry::VertexDataType type,
            UInt32 count,
            UInt16 stride,
            void* buffer,
            DisposerFn disposerFn);

        // Creates an immutable store buffer.
        StoreBuffer* CreateConstBuffer(
            VertexGeometry::VertexDataType type,
            UInt32 count,
            UInt16 stride,
            const void* buffer,
            DisposerFn disposerFn);

        // Methods for accessing useful properties of the StoreBuffer.
        inline VertexGeometry::VertexDataType GetType() const { return m_type; }
        inline UInt32 GetCount() const { return m_count; }
        inline UInt16 GetStride() const { return m_stride; }

        inline void* GetStartLocation() { return m_buffer; }
        inline const void* GetConstStartLocation() const { return m_constBuffer; }

    //private:
        // Destructor can't be hidden because it might be called by some Platform
        //allocator.
        ~StoreBuffer();

    protected:
        StoreBuffer(
            VertexGeometry::VertexDataType type,
            UInt32 count,
            UInt16 stride,
            void* buffer,
            DisposerFn disposerFn,
            const void* constBuffer);

    private:
        //fields
        VertexGeometry::VertexDataType m_type;

        UInt16 m_refCount;
        UInt16 m_stride;

        UInt32 m_count;

        void* m_buffer;
        DisposerFn m_disposerFn;
        const void* m_constBuffer;

        // Store buffers are linked in a doubly linked list to allow
        //easy manipulation.
        StoreBuffer* m_next;
        StoreBuffer* m_prev;
    }; //class VertexGeometryBuilder::StoreBuffer

    /**
     *  ElementBuffer is used to patch StoreBuffers together into elements.
     */
    class VertexGeometryBuilder::ElementBuffer
    {
    public:
        // Cleans any old data and sets the ElementBuffer to the given properties.
        void Initialize(
            UInt32 offset,
            UInt32 startPosition,
            UInt32 count,
            StoreBuffer* store);

        // Check whether this element buffer overlaps with any subsequent element buffers.
        void AdjustElementCoverage();

        // Force the resize of the element buffer to accommodate a number of items
        //starting from the given cursor position and return the start address.
        void* GetLocation(
            UInt32 cursor,
            UInt32 count);

        // Link a new ElementBuffer right after the current one.
        void Link(ElementBuffer* element);

        // Split the element buffer at the cursor. If a new element buffer is created
        //it will be the second chunk. The last chunk is returned.
        ElementBuffer* Split(UInt32 cursor);

        // Create a fresh element buffer and link it after the current one.
        ElementBuffer* Create(
            UInt32 cursor,
            VertexGeometry::VertexDataType type,
            UInt32 startPosition,
            UInt32 storeCount,
            UInt32 bufferCount,
            StoreBuffer* store);

        // Useful getter functions.
        // Element buffers are linked in a doubly linked list.
        inline ElementBuffer* GetNext() { return m_next; }
        inline ElementBuffer* GetPrev() { return m_prev; }

        inline const ElementBuffer* GetNext() const { return m_next; }
        inline const ElementBuffer* GetPrev() const { return m_prev; }

        inline UInt32 GetStart() const;
        inline UInt32 GetEnd() const;
        inline UInt32 GetStoreEnd() const;

        inline const void* GetConstStartLocation() const;
        inline UInt16 GetStride() const;
        inline VertexGeometry::VertexDataType GetType() const;
        inline UInt32 GetCount() const;

        inline const StoreBuffer* GetConstStore() const;

        // Checks whether the element buffer is ready to store items
        // of the given type.
        inline bool MayWriteType(VertexGeometry::VertexDataType type);

        ~ElementBuffer();
        ElementBuffer();

    private:
        bool m_isInitialized;

        UInt32 m_offset;
        UInt32 m_startLocation;
        UInt32 m_count;

        StoreBuffer* m_store;

        ElementBuffer* m_next;
        ElementBuffer* m_prev;
    }; //class VertexGeometryBuilder::ElementBuffer

    /**
     *  Element abstracts the elements of a vertex or index array.
     *  An element is consistent if it covers the full index space
     *  with non-overlapping element buffers. The index space is
     *  the interval [0 .. 0xffffffff].
     *  Each operation on the element should maintain consistency.
     */
    class VertexGeometryBuilder::Element
    {
    public:
        void Initialize(StoreBuffer* store);

        // Finds the correct ElementBuffer, or creates a new one
        //and provides an address to store the requested items.
        void* GetElementLocation(
            UInt32 cursor,
            VertexGeometry::VertexDataType type,
            UInt32 count);

        // Create a new element buffer over the given store.
        bool CreateElementBuffer(
            UInt32 offset,
            VertexGeometry::VertexDataType type,
            UInt32 startPosition,
            UInt32 count,
            StoreBuffer* store);

        // Get the highest value (+1) of an item index that was
        //dirtied by GetElementLocation or CreateElementBuffer.
        UInt32 GetCount() const;

        // Check whether the first "count" items are all set.
        bool IsComplete(UInt32 count) const;

        // Pack the data in the element to the provided destination
        //buffer.
        bool GetData(
            void* destination,
            VertexGeometry::VertexDataType type,
            UInt32 count,
            UInt16 stride) const;

        Element();
        ~Element();

    private:
        // As new store need to be created this method estimates
        //the best solution for increasing the capacity of the element.
        inline UInt32 GetNewStoreCount() const;

        enum {
            c_storeMinCount = 16, ///< Minimum increase of the Element.
            c_storeCountIncreaseRate = 4 ///< Maximum increase is m_userCount / c_storeCountIncreaseRate.
        };

        // Last cursor position.
        UInt32 m_userOffset;
        UInt32 m_userCount;

        // Last used element buffer. This helps with fast access to
        //consecutive locations.
        ElementBuffer* m_current;

        // Base element buffer.
        ElementBuffer m_element;
    }; //class VertexGeometryBuilder::Element

    /**
     *  VertexData is a hash table of Elements mapped to
     *  Vertex semantics.
     */
    class VertexGeometryBuilder::VertexData
    {
    public:
        // Methods used by VertexGemetryBuilder to create the initial VertexData.
        static VertexData* Create(StoreBuffer* store);
        static void Destroy(VertexData* data);

        // These methods map to Element method.
        void* GetElementLocation(
            VertexGeometry::VertexUsage usage,
            UInt8 usageIndex,
            UInt32 cursor,
            VertexGeometry::VertexDataType type,
            UInt32 count);

        bool CreateElementBuffer(
            VertexGeometry::VertexUsage usage,
            UInt8 usageIndex,
            UInt32 offset,
            VertexGeometry::VertexDataType type,
            UInt32 startPosition,
            UInt32 count,
            StoreBuffer* store);

        // Used to keep user information.
        // This is used in packing the tightly packed buffer.
        void SetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex, VertexGeometry::VertexDataType type);
        VertexGeometry::VertexDataType GetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const;

        // Destroy a vertex element.
        void RemoveVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex);
        void RemoveVertexElements(VertexGeometry::VertexUsage usage);

        // Helpful getter functions.
        UInt32 GetVertexCount() const;
        UInt16 GetVertexStride() const;
        UInt16 GetVertexElementCount() const;

        // Check whether the first "count" items are all set.
        bool IsComplete(UInt32 count) const;

        // Pack vertex element tightly packed buffer.
        bool GetVertexElement(void *elementData, VertexGeometry::VertexUsage usage, UInt8 usageIndex, UInt32 count) const;

        // Pack a full vertex geometry tightly packed buffer.
        bool GetVertexData(void *vertexData, UInt32 count) const;
        // Pack the full vertex geometry description.
        bool GetVertexElementFormat(VertexGeometry::VertexElementFormat* vertexElementFormat, UInt16 count) const;

    //private:
        ~VertexData();
    protected:
        VertexData(StoreBuffer* store);

    private:
         /**
          *  VertexElement holds information needed for hashing
          *  and type information provided by the user.
          */
        struct VertexElement
        {
            bool m_isUserType;

            VertexGeometry::VertexDataType m_type;
            VertexGeometry::VertexUsage m_usage;

            UInt8 m_usageIndex;

            VertexElement* m_next;
            Element m_element;

            VertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex);
            ~VertexElement();
        };

        // Retrieve elements from the hash, or force the creation of new ones.
        VertexElement* ForceGetElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex);
        VertexElement* GetElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const;

        Int GetHashIndex(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const;

        enum {
            c_firstPrimeNumber = 3,
            c_secondPrimeNumber = 11,
            c_vertexHashMask = 0xfU,
            c_sizeOfVertexHash = 16
        };

        VertexElement *m_hash[c_sizeOfVertexHash];

        StoreBuffer* m_store;
    }; //class VertexGeometryBuilder::VertexData

    /**
     *  IndexData abstracts the only Element used for index storing.
     */
    class VertexGeometryBuilder::IndexData
    {
    public:
        static IndexData* Create(VertexGeometryBuilder::StoreBuffer* store);
        static void Destroy(IndexData* data);

        UInt32* GetElementLocation(
            UInt32 cursor,
            UInt32 count);

        bool CreateElementBuffer(
            UInt32 offset,
            UInt32 startPosition,
            UInt32 count,
            StoreBuffer* store);

        UInt32 GetIndexCount() const;

        // Check whether the first "count" items are all set.
        bool IsComplete(UInt32 count) const;

        bool GetIndexData(void* indexData, VertexGeometry::VertexDataType indexDataType, UInt32 count) const;
    private:
        Element m_element;
    }; //class VertexGeometryBuilder::IndexData

    /**************************************************************************
     *  StoreBuffer
     **************************************************************************/
    VertexGeometryBuilder::StoreBuffer* VertexGeometryBuilder::StoreBuffer::CreateRootBuffer()
    {
        static Float val[4] = {0.0F, 0.0F, 0.0F, 1.0F};
        StoreBuffer* store = FEATSTD_NEW(StoreBuffer)(VertexGeometry::Float32_4, ~UInt32(0), 0, 0, 0, val);
        if (store != 0) {
            Acquire(store);
        }

        return store;
    }
    void VertexGeometryBuilder::StoreBuffer::DestroyBuffer(StoreBuffer* store)
    {
        if (store == 0) {
            return;
        }
        StoreBuffer* current = store;
        do {
            StoreBuffer* next = current->m_next;
            FEATSTD_DELETE(current);
            current = next;
        } while (current != store);
    }

    void VertexGeometryBuilder::StoreBuffer::Acquire(StoreBuffer *store)
    {
        store->m_refCount ++;
    }
    
    void VertexGeometryBuilder::StoreBuffer::Release(StoreBuffer *store)
    {
        store->m_refCount --;
        if (store->m_refCount == 0) {
            //remove store from list
            store->m_next->m_prev = store->m_prev;
            store->m_prev->m_next = store->m_next;

            FEATSTD_DELETE(store);
        }
    }

    VertexGeometryBuilder::StoreBuffer* VertexGeometryBuilder::StoreBuffer::CreateBuffer(
        VertexGeometry::VertexDataType type,
        UInt32 count,
        UInt16 stride,
        void* buffer,
        DisposerFn disposerFn)
    {
        StoreBuffer* current = FEATSTD_NEW(StoreBuffer)(type, count, stride, buffer, disposerFn, buffer);
        if (current != 0) {
            this->m_prev->m_next = current;
            current->m_prev = this->m_prev;
            this->m_prev = current;
            current->m_next = this;
        }

        return current;
    }

    VertexGeometryBuilder::StoreBuffer* VertexGeometryBuilder::StoreBuffer::CreateConstBuffer(
        VertexGeometry::VertexDataType type,
        UInt32 count,
        UInt16 stride,
        const void* buffer,
        DisposerFn disposerFn)
    {
        StoreBuffer* current = FEATSTD_NEW(StoreBuffer)(type, count, stride, 0, disposerFn, buffer);
        if (current != 0) {
            this->m_prev->m_next = current;
            current->m_prev = this->m_prev;
            this->m_prev = current;
            current->m_next = this;
        }

        return current;
    }

    VertexGeometryBuilder::StoreBuffer::StoreBuffer(VertexGeometry::VertexDataType type,
                                                    UInt32 count,
                                                    UInt16 stride,
                                                    void* buffer,
                                                    DisposerFn disposerFn,
                                                    const void* constBuffer) :
        m_type(type),
        m_refCount(0),
        m_stride(stride),
        m_count(count),
        m_buffer(buffer),
        m_disposerFn(disposerFn),
        m_constBuffer(constBuffer),
        m_next(0),
        m_prev(0)
    {
        m_next = this;
        m_prev = this;
    }

    VertexGeometryBuilder::StoreBuffer::~StoreBuffer()
    {
        if (m_disposerFn != 0) {
            m_disposerFn(m_buffer);
            m_disposerFn = 0;
        }

        m_buffer = 0;
        m_constBuffer = 0;

        m_next = 0;
        m_prev = 0;
    }

    /**************************************************************************
     *  ElementBuffer
     **************************************************************************/

    void VertexGeometryBuilder::ElementBuffer::Initialize(
        UInt32 offset,
        UInt32 startPosition,
        UInt32 count,
        StoreBuffer* store)
    {
        if (m_isInitialized) {
            StoreBuffer::Release(m_store);

            m_isInitialized = false;
        }

        if (store != 0) {
            m_offset = offset;

            m_startLocation = startPosition;
            m_count = count;

            m_store = store;
            StoreBuffer::Acquire(m_store);

            m_isInitialized = true;
        }
    }

    void VertexGeometryBuilder::ElementBuffer::AdjustElementCoverage()
    {
        UInt32 end = m_offset + m_count;
        ElementBuffer* current = m_next;
        while ((0 != current) && (current != this)) {
            //current block starts after the end of the new block
            if (current->m_offset >= end) {
                break;
            }
            //current block ends after the end of the new block
            UInt32 currentEnd = current->m_offset + current->m_count;
            if (end < currentEnd) {
                UInt32 newCount = currentEnd - end;
                UInt32 step = current->m_count - newCount;

                current->m_count = newCount;
                current->m_startLocation += step;
                current->m_offset += step;

                break;
            }
            //current block is completly covered by the old block.
            ElementBuffer* next = current->m_next;

            current->m_next = current;
            next->m_prev = current->m_prev;
            current->m_prev = current;
            next->m_prev->m_next = next;

            FEATSTD_DELETE(current);

            current = next;
        }
    }

    void* VertexGeometryBuilder::ElementBuffer::GetLocation(
        UInt32 cursor,
        UInt32 count)
    {
        void *start = m_store->GetStartLocation();

        if (0 == start) {
            return 0;
        }

        UInt32 newCount = (cursor + count) - m_offset;
        if (newCount > m_count) {
            m_count = newCount;
        }

        UInt32 location = (m_startLocation + cursor) - m_offset;
        return FeatStd::Internal::PointerToPointer<UInt8*>(start) + (location * m_store->GetStride());
    }

    void VertexGeometryBuilder::ElementBuffer::Link(ElementBuffer* element)
    {
        element->m_prev = this;
        this->m_next->m_prev = element;
        element->m_next = this->m_next;
        this->m_next = element;
    }

    VertexGeometryBuilder::ElementBuffer* VertexGeometryBuilder::ElementBuffer::Split(UInt32 cursor)
    {
        if (cursor <= GetStart()){
            return GetPrev();
        }
        if (cursor >= GetEnd()){
            return this;
        }

        ElementBuffer* element = FEATSTD_NEW(ElementBuffer);
        if (element == 0) {
            return 0;
        }

        UInt32 newCount = cursor - m_offset;

        element->Initialize(
            m_offset,
            m_startLocation,
            newCount,
            m_store);
        GetPrev()->Link(element);

        element->AdjustElementCoverage();

        return element;
    }
    VertexGeometryBuilder::ElementBuffer* VertexGeometryBuilder::ElementBuffer::Create(
        UInt32 cursor,
        VertexGeometry::VertexDataType type,
        UInt32 startPosition,
        UInt32 storeCount,
        UInt32 bufferCount,
        StoreBuffer* store)
    {
        bool releaseStore = false;
        if (store == 0) {
            UInt16 stride = VertexGeometry::GetVertexElementSize(type);

            void* buffer = FEATSTD_NEW_ARRAY(UInt8, static_cast<SizeType>(storeCount) * static_cast<SizeType>(stride));

            if (buffer != 0) {
                store = m_store->CreateBuffer(
                    type,
                    storeCount,
                    stride,
                    buffer,
                    AdaptedVertexGeometryDisposer::Dispose);
            }

            if (store == 0) {
                return 0;
            }
            StoreBuffer::Acquire(store);
            releaseStore = true;
        }

        ElementBuffer* element = FEATSTD_NEW(ElementBuffer);
        if (element != 0) {
            element->Initialize(
                cursor,
                startPosition,
                bufferCount,
                store);
            Link(element);
        }
        if (releaseStore) {
            StoreBuffer::Release(store);
        }
        return element;
    }

    inline UInt32 VertexGeometryBuilder::ElementBuffer::GetStart() const
    {
        return m_offset;
    }
    inline UInt32 VertexGeometryBuilder::ElementBuffer::GetEnd() const
    {
        return m_count + m_offset;
    }
    inline UInt32 VertexGeometryBuilder::ElementBuffer::GetStoreEnd() const
    {
        return (m_store->GetCount() + m_offset) - m_startLocation;
    }

    inline const void* VertexGeometryBuilder::ElementBuffer::GetConstStartLocation() const
    {
        return FeatStd::Internal::PointerToPointer<const UInt8*>(m_store->GetConstStartLocation()) +
            (m_store->GetStride() * m_startLocation);
    }

    inline UInt16 VertexGeometryBuilder::ElementBuffer::GetStride() const
    {
        return m_store->GetStride();
    }

    inline VertexGeometry::VertexDataType VertexGeometryBuilder::ElementBuffer::GetType() const
    {
        return m_store->GetType();
    }

    inline UInt32 VertexGeometryBuilder::ElementBuffer::GetCount() const
    {
        return m_count;
    }

    inline const VertexGeometryBuilder::StoreBuffer* VertexGeometryBuilder::ElementBuffer::GetConstStore() const
    {
        return m_store;
    }

    inline bool VertexGeometryBuilder::ElementBuffer::MayWriteType(VertexGeometry::VertexDataType type)
    {
        return (m_store->GetStartLocation() != 0) && (m_store->GetType() == type);
    }

    VertexGeometryBuilder::ElementBuffer::ElementBuffer() :
        m_isInitialized(false),
        m_offset(0),
        m_startLocation(0),
        m_count(0),
        m_store(),
        m_next(0),
        m_prev(0)
    {
        m_next = this;
        m_prev = this;
    }

    VertexGeometryBuilder::ElementBuffer::~ElementBuffer()
    {
        ElementBuffer* current = this->m_next;

        current->m_prev = current;
        while (current != this) {
            ElementBuffer* next = current->m_next;

            next->m_prev = current->m_next;
            current->m_next = current;

            FEATSTD_DELETE(current);

            current = next;
        }
        current->m_next = current;

        if (m_isInitialized) {
            StoreBuffer::Release(m_store);
        }

        m_store = 0;
        m_next = 0;
        m_prev = 0;
    }

    /**************************************************************************
     *  Element
     **************************************************************************/
    inline void VertexGeometryBuilder::Element::Initialize(StoreBuffer* store)
    {
        if (0 != store){
            m_element.Initialize(0, 0, store->GetCount(), store);
        }
    }

    void* VertexGeometryBuilder::Element::GetElementLocation(
        UInt32 cursor,
        VertexGeometry::VertexDataType type,
        UInt32 count)
    {
        if (cursor < m_userOffset) {
            m_current = m_element.GetNext();
        }

        for (;;) {
            if (0 != m_current) {
                if (cursor >= m_current->GetStart()) {
                    bool mayReplace =
                        (cursor <= m_current->GetEnd()) &&
                        ((cursor + count) <= m_current->GetStoreEnd()) &&
                        m_current->MayWriteType(type);
                    if (mayReplace) {
                        break;
                    }

                    bool mayInsert =
                        cursor < m_current->GetEnd();
                    if (mayInsert) {
                        UInt32 newCount;

                        newCount = GetNewStoreCount();
                        newCount = Math::Maximum(newCount, count);

                        m_current = m_current->Split(cursor);
                        if (m_current == 0) {
                            m_current = &m_element;
                            return 0;
                        }
                        m_current = m_current->Create(
                            cursor,
                            type,
                            0,
                            newCount,
                            count,
                            0);
                        if (m_current == 0) {
                            m_current = &m_element;
                            return 0;
                        }

                        break;
                    }
                }
                m_current = m_current->GetNext();
            } else {
                break;
            }
        }

        void *location = 0;
        if (0 != m_current) {
            location = m_current->GetLocation(cursor, count);
            m_current->AdjustElementCoverage();

            if (m_current->GetEnd() > m_userCount) {
                m_userCount = m_current->GetEnd();
            }
        }
        return location;
    }

    bool VertexGeometryBuilder::Element::CreateElementBuffer(UInt32 offset, VertexGeometry::VertexDataType type, UInt32 startPosition, UInt32 count, StoreBuffer* store)
    {
        if (offset < m_userOffset) {
            m_current = m_element.GetNext();
        }

        for (;;) {
            if (0 != m_current) {
                if (offset >= m_current->GetStart()) {
                    bool mayInsert =
                        offset <= m_current->GetEnd();
                    if (mayInsert) {
                        m_current = m_current->Split(offset);
                        if (m_current == 0) {
                            m_current = &m_element;
                            return false;
                        }
                        m_current = m_current->Create(
                            offset,
                            type,
                            startPosition,
                            0,
                            count,
                            store);
                        if (m_current == 0) {
                            m_current = &m_element;
                            return false;
                        }

                        break;
                    }
                }
                m_current = m_current->GetNext();
            } else {
                break;
            }
        }

        if (0 != m_current) {
            m_current->AdjustElementCoverage();

            if (m_current->GetEnd() > m_userCount) {
                m_userCount = m_current->GetEnd();
            }
        }

        return true;
    }

    inline UInt32 VertexGeometryBuilder::Element::GetCount() const
    {
        return m_userCount;
    }

    bool VertexGeometryBuilder::Element::IsComplete(UInt32 count) const
    {
        const ElementBuffer* current = m_element.GetNext();
        if (0 != current) {
            do {
                if (count == 0) {
                    break;
                }
                if (current->GetConstStore() == m_element.GetConstStore()){
                    return false;
                }

                UInt32 elementCount = current->GetCount();
                UInt32 currentCount = Math::Minimum(count, elementCount);

                count -= currentCount;

                current = current->GetNext();
            } while ((0 != current) && (current != m_element.GetNext()));
        }
        return true;
    }

    bool VertexGeometryBuilder::Element::GetData(
        void* destination,
        VertexGeometry::VertexDataType type,
        UInt32 count,
        UInt16 stride) const
    {
        const ElementBuffer* current = m_element.GetNext();

        UInt8* typedDestination = FeatStd::Internal::PointerToPointer<UInt8*>(destination);
        if (0 != current) {
            do {
                if (count == 0) {
                    break;
                }

                UInt32 elementCount = current->GetCount();
                UInt32 currentCount = Math::Minimum(count, elementCount);

                Candera::Internal::VertexGeometryTools::DataTypeTraits dstTraits = 
                    Candera::Internal::VertexGeometryTools::GetTypeTraits(type);
                Candera::Internal::VertexGeometryTools::DataTypeTraits srcTraits =
                    Candera::Internal::VertexGeometryTools::GetTypeTraits(current->GetType());
                Candera::Internal::VertexGeometryTools::Converter converter = 
                    Candera::Internal::VertexGeometryTools::GetConverter(dstTraits.type, srcTraits.type);
                converter.ConvertBuffer(
                    typedDestination, dstTraits.count, stride,
                    current->GetConstStartLocation(), srcTraits.count, current->GetStride(),
                    currentCount);

                count -= currentCount;
                typedDestination += currentCount * stride;

                current = current->GetNext();
            } while ((0 != current) && (current != m_element.GetNext()));
        }
        return count == 0;
    }

    VertexGeometryBuilder::Element::Element() :
        m_userOffset(0),
        m_userCount(0),
        m_current(&m_element)
    {
    }

    VertexGeometryBuilder::Element::~Element()
    {
        m_current = 0;
    }

    inline UInt32 VertexGeometryBuilder::Element::GetNewStoreCount() const
    {
        return Math::Maximum(static_cast<UInt32>(c_storeMinCount), m_userCount / c_storeCountIncreaseRate);
    }

    /**************************************************************************
     *  VertexData
     **************************************************************************/
    VertexGeometryBuilder::VertexData* VertexGeometryBuilder::VertexData::Create(StoreBuffer* store)
    {
        VertexData *data = FEATSTD_NEW(VertexData)(store);
        return data;
    }
    void VertexGeometryBuilder::VertexData::Destroy(VertexData* data)
    {
        FEATSTD_DELETE(data);
    }

    void* VertexGeometryBuilder::VertexData::GetElementLocation(
        VertexGeometry::VertexUsage usage,
        UInt8 usageIndex,
        UInt32 cursor,
        VertexGeometry::VertexDataType type,
        UInt32 count)
    {
        VertexElement* element = ForceGetElement(usage, usageIndex);
        if (element == 0) {
            return 0;
        }
        if (!element->m_isUserType) {
            element->m_type = type;
        }
        return element->m_element.GetElementLocation(cursor, type, count);
    }

    bool VertexGeometryBuilder::VertexData::CreateElementBuffer(
        VertexGeometry::VertexUsage usage,
        UInt8 usageIndex,
        UInt32 offset,
        VertexGeometry::VertexDataType type,
        UInt32 startPosition,
        UInt32 count,
        StoreBuffer* store)
    {
        VertexElement* element = ForceGetElement(usage, usageIndex);
        if (element == 0) {
            return false;
        }
        if (!element->m_isUserType) {
            element->m_type = type;
        }
        return element->m_element.CreateElementBuffer(offset, type, startPosition, count, store);
    }

    void VertexGeometryBuilder::VertexData::SetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex, VertexGeometry::VertexDataType type)
    {
        VertexElement* element = ForceGetElement(usage, usageIndex);
        if (element != 0) {
            element->m_type = type;
            element->m_isUserType = true;
        }
    }

    VertexGeometry::VertexDataType VertexGeometryBuilder::VertexData::GetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const
    {
        VertexElement* element = GetElement(usage, usageIndex);
        if (element == 0) {
            return VertexGeometry::Float32_4;
        }
        else {
            return element->m_type;
        }
    }

    void VertexGeometryBuilder::VertexData::RemoveVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex)
    {
        Int index = GetHashIndex(usage, usageIndex);
        VertexElement* element = m_hash[index];
        VertexElement** previous = &m_hash[index];
        while (element != 0) {
            if ((element->m_usage == usage) && (element->m_usageIndex == usageIndex)) {
                *previous = element->m_next;
                element->m_next = 0;

                FEATSTD_DELETE(element);
                break;
            }
            previous = &element->m_next;
            element = element->m_next;
        }
    }

    void VertexGeometryBuilder::VertexData::RemoveVertexElements(VertexGeometry::VertexUsage usage)
    {
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            VertexElement** previous = &m_hash[index];
            while (*previous != 0) {
                if ((*previous)->m_usage == usage) {
                    VertexElement* element = *previous;
                    *previous = (*previous)->m_next;

                    element->m_next = 0;
                    FEATSTD_DELETE(element);
                }
                else {
                    previous = &(*previous)->m_next;
                }
            }
        }
    }

    UInt32 VertexGeometryBuilder::VertexData::GetVertexCount() const
    {
        UInt32 count = 0;
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            VertexElement* element = m_hash[index];
            while (element != 0) {
                UInt32 current = element->m_element.GetCount();
                if (current > count){
                    return current;
                }
                element = element->m_next;
            }
        }
        return count;
    }

    UInt16 VertexGeometryBuilder::VertexData::GetVertexStride() const
    {
        UInt16 stride = 0;
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            VertexElement* element = m_hash[index];
            while (element != 0) {
                stride += VertexGeometry::GetVertexElementSize(element->m_type);
                element = element->m_next;
            }
        }
        return stride;
    }

    UInt16 VertexGeometryBuilder::VertexData::GetVertexElementCount() const
    {
        UInt16 count = 0;
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            VertexElement* element = m_hash[index];
            while (element != 0) {
                count ++;
                element = element->m_next;
            }
        }
        return count;
    }

    bool VertexGeometryBuilder::VertexData::IsComplete(UInt32 count) const
    {
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            VertexElement* element = m_hash[index];
            while (element != 0) {
                if (!element->m_element.IsComplete(count)) {
                    return false;
                }
                element = element->m_next;
            }
        }
        return true;
    }

    bool VertexGeometryBuilder::VertexData::GetVertexElement(void *elementData, VertexGeometry::VertexUsage usage, UInt8 usageIndex, UInt32 count) const
    {
        VertexElement* element = GetElement(usage, usageIndex);
        if (element == 0) {
            return false;
        }

        return element->m_element.GetData(
            elementData,
            element->m_type,
            count,
            VertexGeometry::GetVertexElementSize(element->m_type));
    }

    bool VertexGeometryBuilder::VertexData::GetVertexData(void *vertexData, UInt32 count) const
    {
        UInt16 offset = 0;
        UInt16 stride = GetVertexStride();
        bool success = true;
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            VertexElement* element = m_hash[index];
            while (element != 0) {
                VertexGeometry::VertexDataType type = element->m_type;
                void* elementData = FeatStd::Internal::PointerToPointer<UInt8*>(vertexData) + offset;

                bool elementSuccess = element->m_element.GetData(
                    elementData,
                    type,
                    count,
                    stride);

                success = success && elementSuccess;

                offset += VertexGeometry::GetVertexElementSize(type);
                element = element->m_next;
            }
        }
        return success;
    }

    bool VertexGeometryBuilder::VertexData::GetVertexElementFormat(VertexGeometry::VertexElementFormat* vertexElementFormat, UInt16 count) const
    {
        UInt16 offset = 0;
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            if (count == 0) {
                break;
            }
            VertexElement* element = m_hash[index];
            while (element != 0) {
                if (count == 0) {
                    break;
                }
                VertexGeometry::VertexDataType type = element->m_type;
                vertexElementFormat->Offset = offset;
                vertexElementFormat->Type = type;
                vertexElementFormat->Usage = element->m_usage;
                vertexElementFormat->UsageIndex = element->m_usageIndex;

                offset += VertexGeometry::GetVertexElementSize(type);
                vertexElementFormat ++;
                count --;

                element = element->m_next;
            }
        }
        return count == 0;
    }

    VertexGeometryBuilder::VertexData::~VertexData()
    {
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            FEATSTD_DELETE(m_hash[index]);
            m_hash[index] = 0;
        }

        m_store = 0;
    }

    VertexGeometryBuilder::VertexData::VertexData(StoreBuffer* store) :
        m_store(store)
    {
        for (Int index = 0; index < c_sizeOfVertexHash; index ++) {
            m_hash[index] = 0;
        }
    }

    VertexGeometryBuilder::VertexData::VertexElement* VertexGeometryBuilder::VertexData::ForceGetElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex)
    {
        Int index = GetHashIndex(usage, usageIndex);
        VertexElement* element = m_hash[index];
        while (element != 0) {
            if ((element->m_usage == usage) && (element->m_usageIndex == usageIndex)) {
                return element;
            }
            element = element->m_next;
        }
        element = FEATSTD_NEW(VertexElement)(usage, usageIndex);
        if (element == 0) {
            return 0;
        }
        element->m_element.Initialize(m_store);
        element->m_next = m_hash[index];

        return m_hash[index] = element;
    }

    VertexGeometryBuilder::VertexData::VertexElement* VertexGeometryBuilder::VertexData::GetElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const
    {
        Int index = GetHashIndex(usage, usageIndex);
        VertexElement* element = m_hash[index];
        while (element != 0) {
            if ((element->m_usage == usage) && (element->m_usageIndex == usageIndex)) {
                return element;
            }
            element = element->m_next;
        }
        return 0;
    }

    Int VertexGeometryBuilder::VertexData::GetHashIndex(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const
    {
        return static_cast<Int>( static_cast<UInt>( (static_cast<UInt>(usage) * c_firstPrimeNumber) + (static_cast<UInt>(usageIndex) * c_secondPrimeNumber) ) & static_cast<UInt>(c_vertexHashMask) );
    }

    VertexGeometryBuilder::VertexData::VertexElement::VertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex) :
        m_isUserType(false),
        m_type(VertexGeometry::Float32_4),
        m_usage(usage),
        m_usageIndex(usageIndex),
        m_next(0)
    {
    }

    VertexGeometryBuilder::VertexData::VertexElement::~VertexElement()
    {
        FEATSTD_DELETE(m_next);
        m_next = 0;
    }

    /**************************************************************************
     *  IndexData
     **************************************************************************/
    VertexGeometryBuilder::IndexData* VertexGeometryBuilder::IndexData::Create(StoreBuffer* store)
    {
        IndexData *data = FEATSTD_NEW(IndexData);
        if (data != 0) {
            data->m_element.Initialize(store);
        }
        return data;
    }

    void VertexGeometryBuilder::IndexData::Destroy(IndexData* data)
    {
        FEATSTD_DELETE(data);
    }

    UInt32* VertexGeometryBuilder::IndexData::GetElementLocation(
        UInt32 cursor,
        UInt32 count)
    {
        return FeatStd::Internal::PointerToPointer<UInt32 *>(m_element.GetElementLocation(cursor, VertexGeometry::UInt32_1, count));
    }

    bool VertexGeometryBuilder::IndexData::CreateElementBuffer(
        UInt32 offset,
        UInt32 startPosition,
        UInt32 count,
        StoreBuffer* store)
    {
        return m_element.CreateElementBuffer(offset, VertexGeometry::UInt32_1, startPosition, count, store);
    }

    UInt32 VertexGeometryBuilder::IndexData::GetIndexCount() const
    {
        return m_element.GetCount();
    }

    bool VertexGeometryBuilder::IndexData::IsComplete(UInt32 count) const
    {
         return m_element.IsComplete(count);
    }

    bool VertexGeometryBuilder::IndexData::GetIndexData(void* indexData, VertexGeometry::VertexDataType indexDataType, UInt32 count) const
    {
        return m_element.GetData(
            indexData,
            indexDataType,
            count,
            VertexGeometry::GetVertexElementSize(indexDataType));
    }

    /**************************************************************************
     *  VertexGeometryBuilder
     **************************************************************************/
    VertexGeometryBuilder::VertexGeometryBuilder() :
        m_isValid(true),
        m_isUserMemoryPool(false),
        m_isUserBufferType(false),
        m_isUserBufferUsage(false),
        m_memoryPool(VertexGeometry::VideoMemory),
        m_bufferType(VertexGeometry::ArrayBuffer),
        m_bufferUsage(VertexGeometry::StaticWrite),
        m_maxVertexCount(0xFFffFFff),
        m_maxIndexCount(0xFFffFFff),
        m_vertexCursor(0),
        m_indexCursor(0),
        m_store(StoreBuffer::CreateRootBuffer()),
        m_vertexData(VertexData::Create(m_store)),
        m_indexData(IndexData::Create(m_store))
    {
        if ((m_store == 0) || (m_vertexData == 0) || (m_indexData == 0)) {
            m_isValid = false;
        }
    }

    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::VertexGeometryBuilder::m_vertexData, CANDERA_LINT_REASON_OPERANDNOTNULL)
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::VertexGeometryBuilder::m_indexData, CANDERA_LINT_REASON_OPERANDNOTNULL)
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::VertexGeometryBuilder::m_store, CANDERA_LINT_REASON_OPERANDNOTNULL)

    VertexGeometryBuilder::~VertexGeometryBuilder()
    {
        VertexData::Destroy(m_vertexData);
        m_vertexData = 0;

        IndexData::Destroy(m_indexData);
        m_indexData = 0;

        StoreBuffer::DestroyBuffer(m_store);
        m_store = 0;
    }

    // Vertex manipulation.
    void VertexGeometryBuilder::SetVertexCursor(UInt32 offset)
    {
        m_vertexCursor = offset;
    }
    UInt32 VertexGeometryBuilder::GetVertexCursor() const
    {
        return m_vertexCursor;
    }
    void VertexGeometryBuilder::IncrementVertexCursor(UInt32 steps)
    {
        m_vertexCursor += steps;
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x, Float y, Float z, Float w)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Float32_4, 1);
        if (pointer != 0) {
            Float* fpointer = FeatStd::Internal::PointerToPointer<Float*>(pointer);
            fpointer[0] = x;
            fpointer[1] = y;
            fpointer[2] = z;
            fpointer[3] = w;
        }
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x, Float y, Float z)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Float32_3, 1);
        if (pointer != 0) {
            Float* fpointer = FeatStd::Internal::PointerToPointer<Float*>(pointer);
            fpointer[0] = x;
            fpointer[1] = y;
            fpointer[2] = z;
        }
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x, Float y)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Float32_2, 1);
        if (pointer != 0) {
            Float* fpointer = FeatStd::Internal::PointerToPointer<Float*>(pointer);
            fpointer[0] = x;
            fpointer[1] = y;
        }
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Float x)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Float32_1, 1);
        if (pointer != 0) {
            Float* fpointer = FeatStd::Internal::PointerToPointer<Float*>(pointer);
            fpointer[0] = x;
        }
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x, Int y, Int z, Int w)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Int16_4, 1);
        if (pointer != 0) {
            Int16* ipointer = FeatStd::Internal::PointerToPointer<Int16*>(pointer);
            ipointer[0] = static_cast<Int16>(x);
            ipointer[1] = static_cast<Int16>(y);
            ipointer[2] = static_cast<Int16>(z);
            ipointer[3] = static_cast<Int16>(w);
        }
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x, Int y, Int z)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Int16_3, 1);
        if (pointer != 0) {
            Int16* ipointer = FeatStd::Internal::PointerToPointer<Int16*>(pointer);
            ipointer[0] = static_cast<Int16>(x);
            ipointer[1] = static_cast<Int16>(y);
            ipointer[2] = static_cast<Int16>(z);
        }
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x, Int y)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Int16_2, 1);
        if (pointer != 0) {
            Int16* ipointer = FeatStd::Internal::PointerToPointer<Int16*>(pointer);
            ipointer[0] = static_cast<Int16>(x);
            ipointer[1] = static_cast<Int16>(y);
        }
    }

    void VertexGeometryBuilder::SetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex, Int x)
    {
        if (!m_isValid) {
            return;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, m_vertexCursor, VertexGeometry::Int16_1, 1);
        if (pointer != 0) {
            Int16* ipointer = FeatStd::Internal::PointerToPointer<Int16*>(pointer);
            ipointer[0] = static_cast<Int16>(x);
        }
    }

    bool VertexGeometryBuilder::SetVertexElementRange(
        VertexGeometry::VertexUsage usage,
        UInt8 usageIndex,
        UInt32 offset,
        VertexGeometry::VertexDataType type,
        UInt32 count,
        UInt16 stride,
        const void *buffer)
    {
        if (!m_isValid) {
            return false;
        }
        void* pointer = m_vertexData->GetElementLocation(usage, usageIndex, offset, type, count);
        if (pointer == 0) {
            return false;
        }

        UInt16 typeSize = VertexGeometry::GetVertexElementSize(type);
        if (stride == typeSize) {
            MemoryPlatform::Copy(pointer, buffer, static_cast<SizeType>(typeSize * count));
        }
        else {
            while (count-- > 0) {
                MemoryPlatform::Copy(pointer, buffer, typeSize);
                buffer = FeatStd::Internal::PointerToPointer<const UInt8*>(buffer) +stride;
                pointer = FeatStd::Internal::PointerToPointer<UInt8*>(pointer) + typeSize;
            }
        }

        return true;
    }

    bool VertexGeometryBuilder::SetVertexElementRange(
        VertexGeometry::VertexUsage usage,
        UInt8 usageIndex,
        UInt32 offset,
        VertexGeometry::VertexDataType type,
        UInt32 count,
        UInt16 stride,
        const void* buffer,
        DisposerFn disposerFn)
    {
        if (!m_isValid) {
            return false;
        }
        StoreBuffer *spointer =
            m_store->CreateConstBuffer(
                type,
                count,
                stride,
                buffer,
                disposerFn);

        if (spointer == 0) {
            return false;
        }

        bool success =
            m_vertexData->CreateElementBuffer(
                usage,
                usageIndex,
                offset,
                type,
                0,
                count,
                spointer);

        return success;
    }

    //Index manipulation

    void VertexGeometryBuilder::SetIndexCursor(UInt32 offset)
    {
        m_indexCursor = offset;
    }
    UInt32 VertexGeometryBuilder::GetIndexCursor() const
    {
        return m_indexCursor;
    }
    void VertexGeometryBuilder::IncrementIndexCursor(UInt32 steps)
    {
        m_indexCursor += steps;
    }

    void VertexGeometryBuilder::SetIndexElement(UInt32 i)
    {
        if (!m_isValid) {
            return;
        }
        UInt32* ipointer = m_indexData->GetElementLocation(m_indexCursor, 1);
        if (ipointer != 0) {
            *ipointer = i;
        }

        if (!m_isUserBufferType) {
            m_bufferType = VertexGeometry::IndexedArrayBuffer;
        }
    }

    bool VertexGeometryBuilder::SetIndexElementRange(
        UInt32 offset,
        UInt32 count,
        const UInt16 *buffer)
    {
        if (!m_isValid) {
            return false;
        }
        UInt32* pointer = m_indexData->GetElementLocation(offset, count);
        if (pointer == 0) {
            return false;
        }

        for (UInt32 index = 0; index != count; ++index) {
            pointer[index] = buffer[index];
        }

        if (!m_isUserBufferType) {
            m_bufferType = VertexGeometry::IndexedArrayBuffer;
        }

        return true;
    }

    bool VertexGeometryBuilder::SetIndexElementRange(
        UInt32 offset,
        UInt32 count,
        const void* buffer,
        DisposerFn disposerFn)
    {
        if (!m_isValid) {
            return false;
        }
        StoreBuffer *spointer =
            m_store->CreateConstBuffer(
                VertexGeometry::UInt16_1,
                count,
                sizeof(UInt16),
                buffer,
                disposerFn);

        if (spointer == 0) {
            return false;
        }

        bool success = m_indexData->CreateElementBuffer(offset, 0, count, spointer);

        return success;
    }

    //Buffer manipulation
    bool VertexGeometryBuilder::SpliceGeometry(
        UInt32 vertexOffset,
        UInt32 indexOffset,
        const VertexGeometry* geometry)
    {
        if ((!m_isValid) || (geometry == 0)) {
            return false;
        }

        VertexGeometry::VertexArrayResource vertexArrayResource(geometry->GetVertexArrayResourceHandle());
        const void* vertexArray = vertexArrayResource.GetData();

        if (vertexArray == 0) {
            return false;
        }

        UInt32 vertexCount = geometry->GetVertexCount();
        UInt16 vertexStride = geometry->GetVertexStride();


        VertexGeometry::FormatArrayResource formatArrayResource(geometry->GetFormatArrayResourceHandle());
        const VertexGeometry::VertexElementFormat* vertexElementFormat = formatArrayResource.GetData();
        UInt16 vertexElementCount = geometry->GetVertexElementCount();

        // Go through each element of the source geometry and copy it to the builder.
        for (UInt16 vei = 0; vei < vertexElementCount; vei++) {
            const UInt8* src = FeatStd::Internal::PointerToPointer<const UInt8*>(vertexArray) + vertexElementFormat[vei].Offset;
            UInt8* dst = FeatStd::Internal::PointerToPointer<UInt8*>(
                m_vertexData->GetElementLocation(
                    vertexElementFormat[vei].Usage,
                    vertexElementFormat[vei].UsageIndex,
                    vertexOffset,
                    vertexElementFormat[vei].Type,
                    vertexCount));
            if (dst == 0) {
                return false;
            }
            UInt32 size = VertexGeometry::GetVertexElementSize(vertexElementFormat[vei].Type);
            for (UInt32 i = 0; i < vertexCount; i++) {
                MemoryPlatform::Copy(&dst[i * size], &src[i * vertexStride], size);
            }
        }

        UInt32 indexCount = geometry->GetIndexCount();
        if (indexCount != 0) {
            IndexAccessor indexAccessor(*geometry);
            UInt32* indexLocation = m_indexData->GetElementLocation(indexOffset, indexCount);
            if (indexLocation == 0) {
                return false;
            }

            for (UInt32 i = 0; i < indexCount; i++) {
                indexLocation[i] = indexAccessor.GetIndex(i) + vertexOffset;
            }
        }

        if (!m_isUserBufferUsage) {
            m_bufferUsage = geometry->GetBufferUsage();
        }
        if (!m_isUserMemoryPool) {
            m_memoryPool = geometry->GetMemoryPool();
        }
        if (!m_isUserBufferType) {
            VertexGeometry::BufferType type = geometry->GetBufferType();
            if (m_bufferType == VertexGeometry::ArrayBuffer) {
                m_bufferType = type;
            }
        }

        return true;
    }

    void VertexGeometryBuilder::RemoveVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex)
    {
        if (m_isValid) {
            m_vertexData->RemoveVertexElement(usage, usageIndex);
        }
    }

    void VertexGeometryBuilder::RemoveVertexElements(VertexGeometry::VertexUsage usage)
    {
        if (m_isValid) {
            m_vertexData->RemoveVertexElements(usage);
        }
    }

    void VertexGeometryBuilder::Clear()
    {
        VertexData::Destroy(m_vertexData);
        IndexData::Destroy(m_indexData);
        StoreBuffer::DestroyBuffer(m_store);

        m_isUserMemoryPool = false;
        m_isUserBufferType = false;
        m_isUserBufferUsage = false;
        m_memoryPool = VertexGeometry::VideoMemory;
        m_bufferType = VertexGeometry::ArrayBuffer;
        m_bufferUsage = VertexGeometry::StaticWrite;
        m_maxVertexCount = 0xFFffFFff;
        m_maxIndexCount = 0xFFffFFff;
        m_vertexCursor = 0;
        m_indexCursor = 0;

        m_store = StoreBuffer::CreateRootBuffer();
        m_vertexData = VertexData::Create(m_store);
        m_indexData = IndexData::Create(m_store);

        if ((m_store == 0) || (m_vertexData == 0) || (m_indexData == 0)) {
            m_isValid = false;
        }
    }

    // State manipulation.
    void VertexGeometryBuilder::SetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex, VertexGeometry::VertexDataType type)
    {
        if (m_isValid) {
            m_vertexData->SetVertexElementFormat(usage, usageIndex, type);
        }
    }

    void VertexGeometryBuilder::SetMaxVertexCount(UInt32 size)
    {
        m_maxVertexCount = size;
    }

    void VertexGeometryBuilder::SetMaxIndexCount(UInt32 size)
    {
        m_maxIndexCount = size;
    }

    void VertexGeometryBuilder::SetMemoryPool(VertexGeometry::MemoryPool memoryPool)
    {
        m_memoryPool = memoryPool;
        m_isUserMemoryPool = true;
    }

    void VertexGeometryBuilder::SetBufferType(VertexGeometry::BufferType bufferType)
    {
        m_bufferType = bufferType;
        m_isUserBufferType = true;
    }

    void VertexGeometryBuilder::SetBufferUsage(VertexGeometry::BufferUsage bufferUsage)
    {
        m_bufferUsage = bufferUsage;
        m_isUserBufferUsage = true;
    }

    // State query.

    VertexGeometry::VertexDataType VertexGeometryBuilder::GetVertexElementFormat(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const
    {
        if (m_isValid) {
            return m_vertexData->GetVertexElementFormat(usage, usageIndex);
        }
        return VertexGeometry::Float32_4;
    }

    UInt32 VertexGeometryBuilder::GetMaxVertexCount() const
    {
        return m_maxVertexCount;
    }

    UInt32 VertexGeometryBuilder::GetMaxIndexCount() const
    {
        return m_maxIndexCount;
    }

    VertexGeometry::MemoryPool VertexGeometryBuilder::GetMemoryPool() const
    {
        return m_memoryPool;
    }

    VertexGeometry::BufferType VertexGeometryBuilder::GetBufferType() const
    {
        VertexGeometry::BufferType result = m_bufferType;
        if ((!m_isUserBufferType) && (result != VertexGeometry::ArrayBuffer)) {
            switch (GetIndexDataType()) {
                case VertexGeometry::UInt16_1:
                    result = VertexGeometry::UInt16IndexedArrayBuffer;
                    break;
                case VertexGeometry::UInt8_1:
                    result = VertexGeometry::UInt8IndexedArrayBuffer;
                    break;
                case VertexGeometry::UInt32_1:
                    result = VertexGeometry::UInt32IndexedArrayBuffer;
                    break;
                default:
                    break;
            }
        }

        return result;
    }

    VertexGeometry::BufferUsage VertexGeometryBuilder::GetBufferUsage() const
    {
        return m_bufferUsage;
    }

    UInt32 VertexGeometryBuilder::GetVertexCount() const
    {
        if (!m_isValid) {
            return 0;
        }
        UInt32 count = m_vertexData->GetVertexCount();
        if (count < m_maxVertexCount) {
            return count;
        }
        return m_maxVertexCount;
    }

    UInt32 VertexGeometryBuilder::GetIndexCount() const
    {
        if (!m_isValid) {
            return 0;
        }
        if (m_bufferType == VertexGeometry::ArrayBuffer) {
            return 0;
        }
        UInt32 count = m_indexData->GetIndexCount();
        if (count < m_maxIndexCount) {
            return count;
        }
        return m_maxIndexCount;
    }

    UInt32 VertexGeometryBuilder::GetVertexDataSize() const
    {
        return GetVertexStride() * GetVertexCount();
    }

    UInt16 VertexGeometryBuilder::GetVertexStride() const
    {
        if (!m_isValid) {
            return 0;
        }
        return m_vertexData->GetVertexStride();
    }

    UInt16 VertexGeometryBuilder::GetVertexElementCount() const
    {
        if (!m_isValid) {
            return 0;
        }
        return m_vertexData->GetVertexElementCount();
    }

    bool VertexGeometryBuilder::IsComplete() const
    {
        if (!m_isValid) {
            return false;
        }
        return m_vertexData->IsComplete(GetVertexCount()) &&
            m_indexData->IsComplete(GetIndexCount());
    }

    // Vertex Geometry building.

    bool VertexGeometryBuilder::GetVertexGeometry(
        void* vertexData,
        VertexGeometry::VertexElementFormat* vertexElementFormat,
        UInt16* indexData) const
    {
        if (!m_isValid){
            return false;
        }

        bool success = m_vertexData->GetVertexData(vertexData, GetVertexCount());
        success = success && m_vertexData->GetVertexElementFormat(vertexElementFormat, GetVertexElementCount());
        success = success && m_indexData->GetIndexData(indexData, VertexGeometry::UInt16_1, GetIndexCount());

        return success;
    }

    bool VertexGeometryBuilder::GetVertexGeometry(
        void* vertexData,
        VertexGeometry::VertexElementFormat* vertexElementFormat,
        void* indexData) const
    {
        if (!m_isValid) {
            return false;
        }

        bool success = m_vertexData->GetVertexData(vertexData, GetVertexCount());
        success = success && m_vertexData->GetVertexElementFormat(vertexElementFormat, GetVertexElementCount());
        success = success && m_indexData->GetIndexData(indexData, GetIndexDataType(), GetIndexCount());

        return success;
    }

    void* VertexGeometryBuilder::GetVertexData() const
    {
        if (!m_isValid){
            return 0;
        }

        UInt32 size = GetVertexDataSize();
        if (size == 0) {
            return 0;
        }

        void* data = FEATSTD_NEW_ARRAY(UInt8, size);
        if ((data == 0) || (!m_vertexData->GetVertexData(data, GetVertexCount()))) {
            FEATSTD_DELETE_ARRAY_T(data, UInt8);
            return 0;
        }

        return data;
    }

    void* VertexGeometryBuilder::GetVertexElement(VertexGeometry::VertexUsage usage, UInt8 usageIndex) const
    {
        if (!m_isValid){
            return 0;
        }

        UInt32 size = VertexGeometry::GetVertexElementSize(m_vertexData->GetVertexElementFormat(usage, usageIndex));
        if (size == 0) {
            return 0;
        }

        void* data = FEATSTD_NEW_ARRAY(UInt8, static_cast<SizeType>(size * GetVertexCount()));
        if ((data == 0) || (!m_vertexData->GetVertexElement(data, usage, usageIndex, GetVertexCount()))) {
            FEATSTD_DELETE_ARRAY_T(data, UInt8);
            return 0;
        }

        return data;
    }

    VertexGeometry::VertexElementFormat* VertexGeometryBuilder::GetVertexElementFormat() const
    {
        if (!m_isValid){
            return 0;
        }

        UInt16 count = GetVertexElementCount();
        if (count == 0) {
            return 0;
        }

        VertexGeometry::VertexElementFormat* data = FEATSTD_NEW_ARRAY(VertexGeometry::VertexElementFormat, count);
        if ((data == 0) || (!m_vertexData->GetVertexElementFormat(data, count))) {
            FEATSTD_DELETE_ARRAY(data);
            return 0;
        }

        return data;
    }

    VertexGeometry::VertexDataType VertexGeometryBuilder::GetIndexDataType() const
    {
        if (GetVertexCount() <= FeatStd::Internal::Limits<UInt8>::cMaxValue) {
            return VertexGeometry::UInt8_1;
        }
        if (GetVertexCount() <= FeatStd::Internal::Limits<UInt16>::cMaxValue) {
            return VertexGeometry::UInt16_1;
        }
        return VertexGeometry::UInt32_1;
    }

    UInt16* VertexGeometryBuilder::GetIndexData() const
    {
        if (!m_isValid){
            return 0;
        }

        UInt32 count = GetIndexCount();
        if (count == 0) {
            return 0;
        }

        UInt16* data = FEATSTD_NEW_ARRAY(UInt16, count);
        if (!m_indexData->GetIndexData(data, VertexGeometry::UInt16_1, count)) {
            FEATSTD_DELETE_ARRAY(data);
            return 0;
        }

        return data;
    }

    void* VertexGeometryBuilder::GetIndexData(VertexGeometry::VertexDataType indexDataType) const
    {
        if (!m_isValid) {
            return 0;
        }

        UInt32 count = GetIndexCount();
        if (count == 0) {
            return 0;
        }

        void* data = FEATSTD_NEW_ARRAY(UInt8, static_cast<SizeType>(count)* static_cast<SizeType>(VertexGeometry::GetVertexElementSize(indexDataType)));
        if (!m_indexData->GetIndexData(data, indexDataType, count)) {
            FEATSTD_DELETE_ARRAY_T(data, UInt8);
            return 0;
        }

        return data;
    }

    VertexGeometry* VertexGeometryBuilder::GetVertexGeometry() const
    {
        if (!m_isValid){
            return 0;
        }

        void* vertexData = GetVertexData();
        VertexGeometry::VertexElementFormat* vertexElementFormat = GetVertexElementFormat();
        void* indexData = 0;
        VertexGeometry::BufferType bufferType = GetBufferType();
        VertexGeometry::IndexBufferDisposer::DisposerFunction indexDisposerFn = AdaptedVertexGeometryDisposer::Dispose;
        UInt32 indexArraySize = GetIndexCount();

        switch (bufferType) {
            case VertexGeometry::UInt16IndexedArrayBuffer:
                indexData = GetIndexData(VertexGeometry::UInt16_1);
                indexArraySize *= 2;
                break;
            case VertexGeometry::UInt8IndexedArrayBuffer:
                indexData = GetIndexData(VertexGeometry::UInt8_1);
                break;
            case VertexGeometry::UInt32IndexedArrayBuffer:
                indexData = GetIndexData(VertexGeometry::UInt32_1);
                indexArraySize *= 4;
                break;
            default:
                indexDisposerFn = 0;
                indexArraySize = 0;
                break;
        }

        bool generationFailed =
            (vertexData == 0) ||
            (vertexElementFormat == 0) ||
            ((indexData == 0) && (bufferType != VertexGeometry::ArrayBuffer));

        if (generationFailed) {
            AdaptedVertexGeometryDisposer::Dispose(vertexData);
            FEATSTD_DELETE_ARRAY(vertexElementFormat);
            AdaptedVertexGeometryDisposer::Dispose(indexData);

            return 0;
        }

        VertexGeometry* geometry = FEATSTD_NEW(VertexGeometry)(
            VertexGeometry::VertexArrayResource::CreateHandle(vertexData, AdaptedVertexGeometryDisposer::Dispose, GetVertexCount() * GetVertexStride()),
            VertexGeometry::IndexArrayResource::CreateHandle(indexData, indexDisposerFn, indexArraySize),
            VertexGeometry::FormatArrayResource::CreateHandle(vertexElementFormat, VertexGeometry::VertexElementFormatDisposer::Dispose, static_cast<UInt32>(GetVertexElementCount()) * sizeof(VertexGeometry::VertexElementFormat)),
            GetVertexStride(),
            m_memoryPool,
            bufferType,
            m_bufferUsage);

        if (geometry == 0) {
            AdaptedVertexGeometryDisposer::Dispose(vertexData);
            FEATSTD_DELETE_ARRAY(vertexElementFormat);
            AdaptedVertexGeometryDisposer::Dispose(indexData);

            return 0;
        }

        return geometry;
    }
} // namespace Candera
