//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexGeometry.h"

#include <Candera/Engine3D/Core/VertexGeometryAccessors.h>
#include <Candera/Engine3D/Core/ShaderParamNames.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Macros.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

VertexGeometry::VertexGeometry(const void* vertexArray,
                               VertexGeometryDisposerFn vertexGeometryDisposerFn,
                               const VertexGeometry::VertexElementFormat* vertexElementFormat,
                               VertexElementFormatDisposerFn vertexElementFormatDisposerFn,
                               const UInt16* indexBuffer,
                               IndexBufferDisposerFn indexBufferDisposerFn,
                               UInt32 vertexCount,
                               UInt16 vertexStride,
                               UInt16 vertexElementCount,
                               UInt32 indexCount,
                               MemoryPool memoryPool,
                               BufferType bufferType,
                               BufferUsage bufferUsage) :
    Base(),
    m_vertexStride(vertexStride),
    m_memoryPool(memoryPool),
    m_bufferType(bufferType),
    m_bufferUsage(bufferUsage),
    m_vertexArrayResourceHandle(VertexArrayResource::CreateHandle(vertexArray, vertexGeometryDisposerFn, vertexCount * vertexStride)),
    m_indexArrayResourceHandle(IndexArrayResource::CreateHandle(indexBuffer, indexBufferDisposerFn, indexCount * sizeof(UInt16))),
    m_formatArrayResourceHandle(FormatArrayResource::CreateHandle(vertexElementFormat, vertexElementFormatDisposerFn, static_cast<UInt32>(sizeof(VertexElementFormat)) * vertexElementCount))
{
    FEATSTD_DEBUG_ASSERT(bufferType != UInt8IndexedArrayBuffer);
    FEATSTD_DEBUG_ASSERT(bufferType != UInt32IndexedArrayBuffer);
    FEATSTD_COMPILETIME_ASSERT(static_cast<UInt32>(VertexUsageCount-1 /* enum starts with 0x1 */) * 8U == static_cast<UInt32>(ShaderParamNames::AttributeSemanticCount));
}

VertexGeometry::VertexGeometry(void* vertexArray,
                               VertexGeometryDisposerFn vertexGeometryDisposerFn,
                               VertexGeometry::VertexElementFormat* vertexElementFormat,
                               VertexElementFormatDisposerFn vertexElementFormatDisposerFn,
                               UInt16* indexBuffer,
                               IndexBufferDisposerFn indexBufferDisposerFn,
                               UInt32 vertexCount,
                               UInt16 vertexStride,
                               UInt16 vertexElementCount,
                               UInt32 indexCount,
                               MemoryPool memoryPool,
                               BufferType bufferType,
                               BufferUsage bufferUsage) :
    Base(),
    m_vertexStride(vertexStride),
    m_memoryPool(memoryPool),
    m_bufferType(bufferType),
    m_bufferUsage(bufferUsage),
    m_vertexArrayResourceHandle(VertexArrayResource::CreateHandle(vertexArray, vertexGeometryDisposerFn, vertexCount * vertexStride)),
    m_indexArrayResourceHandle(IndexArrayResource::CreateHandle(indexBuffer, indexBufferDisposerFn, indexCount * sizeof(UInt16))),
    m_formatArrayResourceHandle(FormatArrayResource::CreateHandle(vertexElementFormat, vertexElementFormatDisposerFn, static_cast<UInt32>(sizeof(VertexElementFormat)) * vertexElementCount))
    {
        FEATSTD_DEBUG_ASSERT(bufferType != UInt8IndexedArrayBuffer);
        FEATSTD_DEBUG_ASSERT(bufferType != UInt32IndexedArrayBuffer);
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, indexBuffer, CANDERA_LINT_REASON_NONCONST)
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, vertexArray, CANDERA_LINT_REASON_NONCONST)
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, vertexElementFormat, CANDERA_LINT_REASON_NONCONST)
    }

    VertexGeometry::VertexGeometry(const ResourceDataHandle& vertexArrayResourceHandle,
                                   const ResourceDataHandle& indexArrayResourceHandle,
                                   const ResourceDataHandle& formatArrayResourceHandle,
                                   UInt16 vertexStride,
                                   MemoryPool memoryPool,
                                   BufferType bufferType,
                                   BufferUsage bufferUsage):
    Base(),
    m_vertexStride(vertexStride),
    m_memoryPool(memoryPool),
    m_bufferType(bufferType),
    m_bufferUsage(bufferUsage),
    m_vertexArrayResourceHandle(vertexArrayResourceHandle),
    m_indexArrayResourceHandle(indexArrayResourceHandle),
    m_formatArrayResourceHandle(formatArrayResourceHandle)
    {

    }
VertexGeometry::~VertexGeometry()
{
    DisposeVertexData();
    FormatArrayResource::DisposeData(m_formatArrayResourceHandle);
}

UInt8 VertexGeometry::GetMaxUsageIndex() const
{
    UInt8 maxIndex = 0;

    VertexGeometry::FormatArrayResource formatArrayResource(GetFormatArrayResourceHandle());
    const VertexGeometry::VertexElementFormat* formatArray = formatArrayResource.GetData();
    if (formatArray == 0) {
        return 0;
    }
    const UInt16 formatCount = GetVertexElementCount();
    for (UInt16 i = 0; i < formatCount; i++) {
        if (formatArray[i].UsageIndex > maxIndex) {
            maxIndex = formatArray[i].UsageIndex;
        }
    }

    return maxIndex;
}

struct VertexDataTypeTraits {
    UInt8 m_size;
    UInt8 m_count;
    bool m_isNormailized;
};

static const VertexDataTypeTraits c_dataTypeTraits[] = {
        {4, 1, false},    //   Float32_1
        {4, 2, false},    //   Float32_2
        {4, 3, false},    //   Float32_3
        {4, 4, false},    //   Float32_4
        {1, 1, false},    //   UInt8_1
        {1, 1, true},     //   UInt8_1_N
        {1, 2, false},    //   UInt8_2
        {1, 2, true},     //   UInt8_2_N
        {1, 3, false},    //   UInt8_3
        {1, 3, true},     //   UInt8_3_N
        {1, 4, false},    //   UInt8_4
        {1, 4, true},     //   UInt8_4_N
        {1, 1, false},    //   Int8_1
        {1, 1, true},     //   Int8_1_N
        {1, 2, false},    //   Int8_2
        {1, 2, true},     //   Int8_2_N
        {1, 3, false},    //   Int8_3
        {1, 3, true},     //   Int8_3_N
        {1, 4, false},    //   Int8_4
        {1, 4, true},     //   Int8_4_N
        {2, 1, false},    //   UInt16_1
        {2, 1, true},     //   UInt16_1_N
        {2, 2, false},    //   UInt16_2
        {2, 2, true},     //   UInt16_2_N
        {2, 3, false},    //   UInt16_3
        {2, 3, true},     //   UInt16_3_N
        {2, 4, false},    //   UInt16_4
        {2, 4, true},     //   UInt16_4_N
        {2, 1, false},    //   Int16_1
        {2, 1, true},     //   Int16_1_N
        {2, 2, false},    //   Int16_2
        {2, 2, true},     //   Int16_2_N
        {2, 3, false},    //   Int16_3
        {2, 3, true},     //   Int16_3_N
        {2, 4, false},    //   Int16_4
        {2, 4, true},     //   Int16_4_N
        {2, 1, false},    //   Float16_1
        {2, 2, false},    //   Float16_2
        {2, 3, false},    //   Float16_3
        {2, 4, false},    //   Float16_4
        {4, 1, false},    //   UInt32_1
        {4, 1, true},     //   UInt32_1_N
        {4, 2, false},    //   UInt32_2
        {4, 2, true},     //   UInt32_2_N
        {4, 3, false},    //   UInt32_3
        {4, 3, true},     //   UInt32_3_N
        {4, 4, false},    //   UInt32_4
        {4, 4, true},     //   UInt32_4_N
        {4, 1, false},    //   Int32_1
        {4, 1, true},     //   Int32_1_N
        {4, 2, false},    //   Int32_2
        {4, 2, true},     //   Int32_2_N
        {4, 3, false},    //   Int32_3
        {4, 3, true},     //   Int32_3_N
        {4, 4, false},    //   Int32_4
        {4, 4, true},     //   Int32_4_N
};

UInt8 VertexGeometry::GetVertexElementSize(VertexDataType vertexDataType)
{
    return c_dataTypeTraits[vertexDataType].m_size * c_dataTypeTraits[vertexDataType].m_count;
}

bool VertexGeometry::IsVertexElementNormalized(VertexDataType vertexDataType)
{
    return c_dataTypeTraits[vertexDataType].m_isNormailized;
}

void VertexGeometry::DisposeVertexData()
{
    VertexArrayResource::DisposeData(m_vertexArrayResourceHandle);
    IndexArrayResource::DisposeData(m_indexArrayResourceHandle);
}


const void* VertexGeometry::GetVertexArray() const
{
    return VertexArrayResource(GetVertexArrayResourceHandle()).GetData();
}

void* VertexGeometry::GetVertexArray()
{
    return VertexArrayResource(GetVertexArrayResourceHandle()).GetMutableData();
}

const UInt16* VertexGeometry::GetIndexBuffer() const
{
    if (m_bufferType != IndexedArrayBuffer) {
        if (m_bufferType != ArrayBuffer) {
            FEATSTD_LOG_WARN("GetIndexBuffer() returns 0 because index is not on 16 bits.");
        }
        return 0;
    }

    return ResourceObject<const UInt16>(GetIndexArrayResourceHandle()).GetData();
}

UInt16* VertexGeometry::GetIndexBuffer()
{
    if (m_bufferType != IndexedArrayBuffer) {
        if (m_bufferType != ArrayBuffer) {
            FEATSTD_LOG_WARN("GetIndexBuffer() returns 0 because index is not on 16 bits.");
        }
        return 0;
    }

    return ResourceObject<const UInt16>(GetIndexArrayResourceHandle()).GetMutableData();
}

const VertexGeometry::VertexElementFormat* VertexGeometry::GetVertexElementFormat() const
{
    return FormatArrayResource(GetFormatArrayResourceHandle()).GetData();
}

VertexGeometry::VertexElementFormat* VertexGeometry::GetVertexElementFormat()
{
    return FormatArrayResource(GetFormatArrayResourceHandle()).GetMutableData();
}

bool VertexGeometry::IsMutable() const
{
    return ((m_vertexArrayResourceHandle.m_isMutable == 1) && (m_indexArrayResourceHandle.m_isMutable == 1));
}

UInt32 VertexGeometry::GetIndexCount() const
{
    UInt32 result = 0;

    switch (m_bufferType) {
        case Candera::VertexGeometry::UInt16IndexedArrayBuffer:
            result = GetIndexArrayResourceHandle().m_size / sizeof(UInt16);
            break;
        case Candera::VertexGeometry::UInt8IndexedArrayBuffer:
            result = GetIndexArrayResourceHandle().m_size / sizeof(UInt8);
            break;
        case Candera::VertexGeometry::UInt32IndexedArrayBuffer:
            result = GetIndexArrayResourceHandle().m_size / sizeof(UInt32);
            break;
        default:
            break;
    }

    return result;
}

} // namespace Candera
