//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexBuffer.h"
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    VertexBuffer::VertexBuffer() :
        Base(),
        m_vertexGeometry(0),
        m_vertexGeometryDisposerFn(0),
        m_primitiveType(Triangles),
        m_elementStart(-1),
        m_elementCount(-1)
    {
        MemoryPlatform::Set(m_vertexArrayMemoryHandle, 0, sizeof(m_vertexArrayMemoryHandle));
        MemoryPlatform::Set(m_indexBufferMemoryHandle, 0, sizeof(m_indexBufferMemoryHandle));

        m_triangleIterator.currentIndex = 0;
        m_triangleIterator.triangle.index[0] = 0;
        m_triangleIterator.triangle.index[1] = 0;
        m_triangleIterator.triangle.index[2] = 0;
    }

    VertexBuffer::SharedPointer VertexBuffer::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        VertexBuffer* ptr = FEATSTD_NEW(VertexBuffer)();
        if (ptr == 0) {
            FEATSTD_LOG_ERROR("VertexBuffer create failed, out of memory.");
        }
        SharedPointer sharedPointer(ptr);
        return sharedPointer;
    }

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1579, "vertexGeometry is disposed by the disposer; the disposer has external lifetime")
    VertexBuffer::~VertexBuffer()
    {
        static_cast<void>(Unload(ForceAll));
        DisposeVertexGeometry();
    }

    bool VertexBuffer::Activate()
    {
        return RenderDevice::ActivateVertexBuffer(*this);
    }

    bool VertexBuffer::Update(UInt32 index, UInt32 count)
    {
        bool result = false;
        if ((m_vertexGeometry != 0) && (m_vertexGeometry ->GetMemoryPool() == VertexGeometry::VideoMemory) && IsUploaded()) {
            VertexGeometry::VertexArrayResource vertexArrayResourceObject(m_vertexGeometry->GetVertexArrayResourceHandle());
            const UInt8* data = FeatStd::Internal::PointerToPointer<const UInt8*>(vertexArrayResourceObject.GetData());
            if (data != 0) {
                Int offset = static_cast<Int>(index * m_vertexGeometry->GetVertexStride());
                Int size = static_cast<Int>(count * m_vertexGeometry->GetVertexStride());
                result = RenderDevice::SetVertexBufferSubData(*this, offset, size, data);
            }
        }
        return result;
    }

    bool VertexBuffer::Update(Int offset, Int size, const UInt8* data)
    {
        bool result = false;
        if ((m_vertexGeometry != 0) && (m_vertexGeometry->GetMemoryPool() == VertexGeometry::VideoMemory)
                && IsUploaded() && (data != 0)) {
            result = RenderDevice::SetVertexBufferSubData(*this, offset, size, data);
        }
        return result;
    }

    bool VertexBuffer::SetVertexGeometry(VertexGeometry* vertexGeometry, VertexGeometryDisposerFn disposerFn /* = 0 */)
    {
        if (IsUploaded()) {
            return false; // vertexBuffer data is already uploaded
        }
        DisposeVertexGeometry();
        m_vertexGeometry = vertexGeometry;
        m_vertexGeometryDisposerFn = disposerFn;
        return true;
    }

    static void SetTriangle(VertexBuffer::Triangle& triangle, const VertexBuffer::Primitive& primitive)
    {
        triangle.index[0] = static_cast<UInt16>(primitive.m_index[0]);
        triangle.index[1] = static_cast<UInt16>(primitive.m_index[1]);
        triangle.index[2] = static_cast<UInt16>(primitive.m_index[2]);
    }

    const VertexBuffer::Triangle* VertexBuffer::GetFirstTriangle()
    {
        if ((m_primitiveType != Triangles) &&
            (m_primitiveType != TriangleFan) &&
            (m_primitiveType != TriangleStrip)) {
            return 0;
        }
        PrimitiveIterator it = GetPrimitiveIterator();
        m_triangleIterator.currentIndex = 0;
        if (!it.IsValid()) {
            return 0;
        }

        SetTriangle(m_triangleIterator.triangle, *it);
        return &m_triangleIterator.triangle;
    }

    const VertexBuffer::Triangle* VertexBuffer::GetNextTriangle()
    {
        if ((m_primitiveType != Triangles) &&
            (m_primitiveType != TriangleFan) &&
            (m_primitiveType != TriangleStrip)) {
            return 0;
        }
        PrimitiveIterator it = GetPrimitiveIterator();
        ++m_triangleIterator.currentIndex;
        for (UInt32 index = 0; index < m_triangleIterator.currentIndex; ++index) {
            ++it;
        }
        if (!it.IsValid()) {
            return 0;
        }

        SetTriangle(m_triangleIterator.triangle, *it);
        return &m_triangleIterator.triangle;
    }

    const void* VertexBuffer::GetVertex(UInt32 index) const
    {
        const void* result = 0;

        if (m_vertexGeometry != 0) {
            VertexGeometry::VertexArrayResource vertexArrayResource(m_vertexGeometry->GetVertexArrayResourceHandle());
            const void* vertices = vertexArrayResource.GetData();
            if ((vertices != 0) && (index < m_vertexGeometry->GetVertexCount())) {
                result = FeatStd::Internal::PointerAdd(vertices, static_cast<OffsetType>(index * static_cast<UInt32>(m_vertexGeometry->GetVertexStride())));
            }
        }

        return result;
    }

    void* VertexBuffer::GetMutableVertex(UInt32 index)
    {
        void* result = 0;

        if (m_vertexGeometry != 0) {
            VertexGeometry::VertexArrayResource vertexArrayResource(m_vertexGeometry->GetVertexArrayResourceHandle());
            void* vertices = vertexArrayResource.GetMutableData();
            if ((vertices != 0) && (index < m_vertexGeometry->GetVertexCount())) {
                result = FeatStd::Internal::PointerAdd(vertices, static_cast<OffsetType>(index * static_cast<UInt32>(m_vertexGeometry->GetVertexStride())));
            }
        }

        return result;
    }

    SizeType VertexBuffer::GetElementCount() const
    {
        if (0 > m_elementCount) {
            const VertexGeometry* geometry = m_vertexGeometry;
            if (0 != geometry) {
                if (geometry->GetBufferType() == VertexGeometry::ArrayBuffer) {
                    return static_cast<SizeType>(geometry->GetVertexCount());
                }

                return static_cast<SizeType>(geometry->GetIndexCount());
            }

            return 0;
        }

        return m_elementCount;
    }

    bool VertexBuffer::UploadInternal(LoadingHint loadingHint)
    {
        bool result = false;
        if ((m_vertexGeometry != 0) && (m_vertexGeometry->GetMemoryPool() == VertexGeometry::VideoMemory)) {
            result = Renderer::UploadVertexBuffer(*this, loadingHint);
        }
        else {
            result = true; // VertexGeometry is in SystemMemory, everything is fine and no need to upload to VRAM
        }
        return result;
    }

    bool VertexBuffer::UnloadInternal(LoadingHint loadingHint)
    {
        bool result = false;
        if ((m_vertexGeometry != 0) && (m_vertexGeometry->GetMemoryPool() == VertexGeometry::VideoMemory)) {
            result = Renderer::UnloadVertexBuffer(*this, loadingHint);
        }
        else {
            result = true; // VertexGeometry is in SystemMemory, everything is fine and no need to unload VRAM
        }
        return result;
    }

    void VertexBuffer::DisposeInternal()
    {
        if ((m_vertexGeometry != 0) && (m_vertexGeometry->GetMemoryPool() == VertexGeometry::VideoMemory)) {
            m_vertexGeometry->DisposeVertexData();
        }
    }

    void VertexBuffer::Render()
    {
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::VertexBuffer::Render, "function has side effects and is therefore considered non-const")

        if (!Renderer::Draw(*this)) {
            FEATSTD_LOG_ERROR("VertexBuffer renderer, failed to draw.");
        }
    }

    VertexGeometry* VertexBuffer::GetMutableVertexGeometry()
    {
        return ((m_vertexGeometry != 0) &&
                (m_vertexGeometry->GetVertexArrayResourceHandle().m_isMutable == 1) &&
                (m_vertexGeometry->GetIndexArrayResourceHandle().m_isMutable == 1) &&
                (m_vertexGeometry->GetFormatArrayResourceHandle().m_isMutable == 1)) ? m_vertexGeometry : 0;
    }

    UInt VertexBuffer::GetSize() const
    {
        return static_cast<UInt>(m_vertexGeometry->GetVertexArrayResourceHandle().m_size + m_vertexGeometry->GetIndexArrayResourceHandle().m_size);
    }

    void VertexBuffer::DisposeVertexGeometry()
    {
        if ((m_vertexGeometryDisposerFn != 0) && (m_vertexGeometry != 0)) {
            m_vertexGeometryDisposerFn(m_vertexGeometry);
        }
        m_vertexGeometry = 0;
        m_vertexGeometryDisposerFn = 0;
    }

    Handle VertexBuffer::GetVertexArrayMemoryHandle() const
    {
        return m_vertexArrayMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
    }

    Handle VertexBuffer::GetIndexBufferMemoryHandle() const
    {
        return m_indexBufferMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
    }

    void VertexBuffer::SetIndexArrayMemoryHandle(Handle handle)
    {
        m_indexBufferMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
    }

    void VertexBuffer::SetVertexArrayMemoryHandle(Handle handle)
    {
        m_vertexArrayMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
    }

    VertexBuffer::PrimitiveIterator VertexBuffer::GetPrimitiveIterator() const
    {
        if (GetVertexGeometry() == 0) {
            InternalIterator invalidIt = {IndexAccessor(), 0, m_primitiveType};
            return PrimitiveIterator(invalidIt, invalidIt);
        }

        IndexAccessor indexAccessor(*GetVertexGeometry());
        InternalIterator first = {indexAccessor, 0, m_primitiveType};
        InternalIterator last = first;
        last.m_index = indexAccessor.GetCount();

        switch (m_primitiveType) {
            case VertexBuffer::Triangles:
                last.m_index /= 3;
                break;
            case VertexBuffer::TriangleStrip:
            case VertexBuffer::TriangleFan:
                last.m_index -= 2;
                break;
            case VertexBuffer::Lines:
                last.m_index /= 2;
                break;
            case VertexBuffer::LineStrip:
                last.m_index -= 1;
                break;
            default:
                break;
        }

        return PrimitiveIterator(first, last);
    }

    VertexBuffer::Primitive VertexBuffer::InternalIterator::operator*() const
    {
        VertexBuffer::Primitive result = {{0, 0, 0}};

        switch (m_primitiveType) {
            case VertexBuffer::Triangles:
                result.m_index[0] = m_indexAccessor.GetIndex(3 * m_index);
                result.m_index[1] = m_indexAccessor.GetIndex(3 * m_index + 1);
                result.m_index[2] = m_indexAccessor.GetIndex(3 * m_index + 2);
                break;
            case VertexBuffer::Lines:
                result.m_index[0] = m_indexAccessor.GetIndex(2 * m_index);
                result.m_index[1] = m_indexAccessor.GetIndex(2 * m_index + 1);
                break;
            case VertexBuffer::TriangleStrip:
            case VertexBuffer::TriangleFan:
                result.m_index[2] = m_indexAccessor.GetIndex(m_index + 2);
                CANDERA_LINT_FALLTHROUGH()
            case VertexBuffer::LineStrip:
                CANDERA_LINT_FALLTHROUGH()
            case VertexBuffer::LineLoop:
                result.m_index[1] = m_indexAccessor.GetIndex(m_index + 1);
                CANDERA_LINT_FALLTHROUGH()
            case VertexBuffer::Points:
                result.m_index[0] = m_indexAccessor.GetIndex(m_index);
                break;
            default:
                break;
        }

        if (m_primitiveType == VertexBuffer::TriangleFan) {
            result.m_index[0] = m_indexAccessor.GetIndex(0);
        }
        if ((m_primitiveType == VertexBuffer::LineLoop) && (m_index == (m_indexAccessor.GetCount() - 1))) {
            result.m_index[1] = m_indexAccessor.GetIndex(0);
        }

        return result;
    }

    bool VertexBuffer::InternalIterator::operator!=(const VertexBuffer::InternalIterator& other) const
    {
        return (m_index != other.m_index); //Enough since the comparison is done only between instances of objects creted from the same VertexBuffer.
    }

    VertexBuffer::InternalIterator& VertexBuffer::InternalIterator::operator++()
    {
        ++m_index;
        return *this;
    }

} // namespace Candera
