//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(UniformNode_h)
#define UniformNode_h

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/System/Container/LinkedList.h>

namespace Candera {
    namespace Internal {

        /**
         *   @brief  A UniformNode stores shader parameters for uniforms.
         */
        class UniformNode
        {
            public:
                /**
                 *  Constructs a UniformNode object.
                 *  @param name specifies the uniform name referenced to in shader.
                 *  @param type            Defines the data type of the uniform data given.
                 *  @param data            Defines the pointer to the location where the data is stored.
                 *  @param count           Specifies the number of array elements to load from the data given. For single elements this value is always 1.
                 *  @param isDataVolatile  If true, the data pointer is not stored, but its content is copied instead.
                 */
                UniformNode(const Char* name, Shader::UniformType type, void* data, Int32 count, bool isDataVolatile);

                /**
                 *  Destroys this UniformNode object.
                 */
                ~UniformNode();

                /**
                 *  Sets the pointer to uniform data location.
                 *  param data Pointer to uniform data location to be set.
                 *  @param data        Data pointer to set.
                 *  @param isVolatile  If true, the data pointer is not stored, but its content is copied instead.
                 */
                void SetData(void* data, bool isVolatile);

                /**  
                 *  Retrieves pointer to uniform data location.
                 *  @return Pointer to uniform data location.
                 */
                const void* GetData() const { return m_data; } 

                /**  
                 *  Retrieves pointer to uniform data location.
                 *  @return Pointer to uniform data location.
                 */
                void* GetData() { return m_data; }

                /**
                 *  Retrieves name of uniform.
                 *  @return Name of the uniform.
                 */
                const Char* GetName() const { return m_name; }

                /**
                 *  Retrieves type of uniform.
                 *  @return Type of the uniform.
                 */
                Shader::UniformType GetType() const { return m_type; }

                /**
                 *  Retrieves the number of array elements to load from the data location.
                 *  @return Number of array elements to load from the data location.
                 */
                Int32 GetCount() const { return m_count; }

                /**
                 *  Intrusive UniformNode pointer.
                 */
                Candera::Internal::LinkedListNode<UniformNode> ListNode;

            private:
                //Default constructor is not defined.
                UniformNode();

                const Int32 m_count;                // Number of data elements specified by uniform type.
                const Shader::UniformType m_type;   // Type of uniform.
                const Char* m_name;                 // Name of uniform.
                void* m_data;                       // Uniform data location.
                const SizeType m_size;              // Size of the data in bytes.
                bool m_isOwner;                     // Indicates if m_name and m_data is owned by UniformNode

                //forbid assignment by commenting it out
                UniformNode& operator=(const UniformNode&);
        };

    }

}

#endif // UniformNode_h
