//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_UNIFORMBUFFER_H
#define CANDERA_UNIFORMBUFFER_H

#include <Candera/Engine3D/Core/DeviceObject.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief   The class UniformBuffer serves as a base for objects that contain/represent uniform buffer
 *          objects in system memory. They can be uploaded to, updated in, and unloaded from the GPU
 *          via the Renderer.
 */
class UniformBuffer : public DeviceObject
{
    FEATSTD_TYPEDEF_BASE(DeviceObject);

public:
    /**
     *  Constructor.
     */
    UniformBuffer();

    /**
     *  Destructor.
     */
    virtual ~UniformBuffer() override;

    /**
     *  Return if the uniform buffer is dirty, i.e. the data in system memory does not match
     *  the data in VRAM.
     *  @return  True if the uniform buffer is dirty.
     */
    bool IsDirty() const { return m_isDirty; }

    /**
     *  Set the dirty flag of the uniform buffer. Whenever the data of the uniform buffer in
     *  system memory changes, set the dirty flag to indicate the change. The flag will be
     *  reset by the Renderer, once the data has been updated in VRAM.
     */
    void SetDirty() { m_isDirty = true; }

protected:

    /**
     *  Return a pointer to the data buffer to be uploaded/updated to/in VRAM.
     *  @return  A pointer to the data buffer to be uploaded/updated to/in VRAM.
     */
    virtual const void* GetData() const = 0;

    /**
     *  Return the size of the data buffer to be uploaded/updated to/in VRAM.
     *  @return  The size of the data buffer to be uploaded/updated to/in VRAM.
     */
    virtual SizeType GetSize() const = 0;

    /**
     *  Upload the buffer object to VRAM
     *  @param loadingHint  See DeviceObject::LoadingHint for details.
     *  @return  True, if uploading the buffer object to VRAM succeeded. False, otherwise.
     */
    virtual bool UploadInternal(LoadingHint loadingHint) override;

    /**
     *  Unload the buffer object from VRAM
     *  @param loadingHint  See DeviceObject::LoadingHint for details.
     *  @return  True, if unloading the buffer object from VRAM succeeded. False, otherwise.
     */
    virtual bool UnloadInternal(LoadingHint loadingHint) override;

    /**
     *  Empty implementation of DisposeInternal.
     */
    virtual void DisposeInternal() override {};

    /**
     *  Copy constructor
     */
    UniformBuffer(const UniformBuffer& rhs);

private:
    friend class Renderer;
    Handle m_deviceHandle[CANDERA_MAX_CONTEXT_COUNT];
    bool m_isDirty;
};

}

#endif
