//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSFORMABLE_H)
#define CANDERA_TRANSFORMABLE_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <FeatStd/Util/Optional.h>

namespace Candera {

/**  @addtogroup Core3D
 *   @{
 */

/**
 *  @brief Defines common methods for manipulating node and texture transformations in local coordinate space (object space).
 *         The transformation consists of following components: position, rotation, scale and a generic transform matrix.
 */
class Transformable : public CanderaObject
{
    FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        /**
         *  Constructor constructs a Transformable object with default values.
         */
        Transformable();

        /**
         *  Copy-Constructor constructs a new Transformable object by copying the contents of the Transformable object given.
         *  @param transformable defines the values to copy in this Transformable object.
         */
        Transformable(const Transformable& transformable);

        /**
         *  Destructor
         */
        virtual ~Transformable() override;

        /**
         *  Sets the local position of this object.
         *  @param x X-coordinate of the position to set.
         *  @param y Y-coordinate of the position to set.
         *  @param z Z-coordinate of the position to set.
         */
        void SetPosition(Float x, Float y, Float z);

        /**
         *  Sets the local position of this object.
         *  @param position The Position vector describing the position components (x, y, z) to set.
         */
        void SetPosition(const Vector3& position);

        /**
         *  Retrieve the position as a vector describing the position components (x, y, z).
         *  @return The position as a vector.
         */
        const Vector3& GetPosition() const { return m_position; }

        /**
         *  Sets the local rotation of this object.
         *  @param pitch Pitch-component of the rotation to set.
         *  @param yaw Yaw-component of the rotation to set.
         *  @param roll Roll-component of the rotation to set.
         */
        void SetRotation(Float pitch, Float yaw, Float roll);

        /**
         *  Sets the local rotation of this object.
         *  @param rotation The rotation vector describing the rotation components (pitch, yaw, roll) to set.
         */
        void SetRotation(const Vector3& rotation);

        /**
         *  Retrieve the rotation as a vector describing the rotation components (pitch, yaw, roll).
         *  @return The rotation as a vector
         */
        const Vector3& GetRotation() const { return m_rotation; }

        /**
         *  Sets the local scale factor of this object.
         *  @param xScale Scale factor in x-direction of the scale to set.
         *  @param yScale Scale factor in y-direction of the scale to set.
         *  @param zScale Scale factor in z-direction of the scale to set.
         */
        void SetScale(Float xScale, Float yScale, Float zScale);

        /**
         *  Sets the local scale factor of this object.
         *  @param scale The scale factor vector describing the scale components (xScale, yScale, zScale) to set.
         */
        void SetScale(const Vector3& scale);

        /**
         *  Retrieve the scale factor as a vector describing the scale components (xScale, yScale, zScale).
         *  @return The scale factor as a vector.
         */
        const Vector3& GetScale() const { return m_scale; }

        /**
        * Sets the pivot point for this object in local coordinate space. The pivot point
        * is the point where rotation and scale is centered around.
        * @param xPivot     The local x coordinate of the pivot point.
        * @param yPivot     The local y coordinate of the pivot point.
        * @param zPivot     The local z coordinate of the pivot point.
        */
        void SetPivotPoint(Float xPivot, Float yPivot, Float zPivot);

        /**
        * Sets the pivot point for this object in local coordinate space. The pivot point
        * is the point where rotation and scale is centered around.
        * @param pivot     The pivot point in local coordinate space.
        */
        void SetPivotPoint(const Vector3& pivot);

        /**
        * Gets the pivot point for this object in local coordinate space. The pivot point
        * is the point where rotation and scale is centered around.
        * @return  The pivot point in local coordinate space.
        */
        const Vector3& GetPivotPoint() const { return m_pivotPoint; }

        /**
         *  Sets the local generic transformation matrix component of this Transformable, without affecting the
         *  separately stored position, rotation and scale components.
         *  @param transform Transformation matrix to set. Position, rotation and scale components are not affected.
         */
        void SetGenericTransform(const Matrix4& transform);

        /**
        *  Sets the optional local generic transformation matrix component of this Transformable, without affecting the
        *  separately stored position, rotation and scale components.
        *  This is the optional version.
        *  @param transform OptionalTransformation matrix to set. Position, rotation and scale components are not affected.
        */
        void SetGenericTransform(const FeatStd::Optional<Matrix4>& transform);

        /**
         *  Retrieves the local transformation matrix component. This does not include position, rotation, and scaling components.
         *  @return The local transformation matrix component. This does not include position, rotation, and scaling components.
         */
        const Matrix4& GetGenericTransform() const;

        /**
         *  Retrieves the local composite transformation matrix. The composite transformation matrix is a concatenation of the
         *  position, rotation, scaling, and transformation matrix components.
         *  Formal logic: CompositeTransform = Scale *  Rotation *  Position *  GenericTransform.
         *  @return The local composite transformation matrix, computed from Scale, Rotation, Position and GenericTransform matrices.
         */
        const Matrix4& GetCompositeTransform() const;

        /**
         *  Concatenates the given movement (translation) with current position component.
         *  @param translation Vector describing the movement(translation) to apply.
         */
        void Translate(const Vector3& translation);

        /**
         *  Multiplies the given scale factors with current scale component.
         *  @param scale Vector describing the scale factors to apply.
         */
        void Scale(const Vector3& scale);

        /**
         *  Concatenates the given rotation angles with current rotation component.
         *  Pitch defines the rotation around the x-axis, yaw around the y-axis, and roll the around z-axis.
         *  @param rotation Vector describing the rotation components (pitch, yaw, roll) to apply.
         */
        void Rotate(const Vector3& rotation);

        /**
        * Translates the pivot point from its current position by the input Vector3 value.
        * The translation is applied locally and relatively to the object's rotation.
        * @param translation The translation that shall be applied.
        */
        void TranslatePivotPoint(const Vector3& translation);

        /**
         *  Assignment operator
         *  @param transformable Transformable to assign.
         *  @return The assigned Transformable.
         */
        Transformable& operator=(const Transformable& transformable);

        FEATSTD_RTTI_DECLARATION();

    protected:
        /**
         *  Determine if the cache for local transform cache is valid.
         *  @return Whether composite transform cache needs to be updated (false) or not(true).
         */
        bool IsCompositeTransformCacheValid() const { return m_isCompositeTransformCacheValid; }

        /**
         *  Called when one of the transforms (translation, rotation, scale, general transform) has changed.
         *  Useful for updating dependent properties. Implementing classes will probably want to propagate the call up the hierarchy.
         */
        virtual void OnCompositeTransformChanged();

        // Composite Transform Cache
        mutable bool m_isCompositeTransformCacheValid;

    private:
        Vector3 m_position;                         // Local position.
        Vector3 m_rotation;                         // Local rotation.
        Vector3 m_scale;                            // Local scale.
        Vector3 m_pivotPoint;                       // Local Pivot Point.
        FeatStd::Optional<Matrix4> m_transform;     // A generic 4x4 transform matrix in object space.
        mutable Matrix4 m_compositeTransformCache;

        /**
         *  Invalidate composite transform cache and notify descendant objects in order to update their world transform cache.
         */
        void InvalidateCompositeCacheAndNotifyDerivedClasses();
};

/**  @} */ // end of Core3D

} // namespace Candera

#endif  // CANDERA_TRANSFORMABLE_H
