//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Transformable.h"
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(Transformable, CanderaObject)

Transformable::Transformable() :
    Base(),
    m_isCompositeTransformCacheValid(false),
    m_position(Vector3(0.0F, 0.0F, 0.0F)),
    m_rotation(Vector3(0.0F, 0.0F, 0.0F)),
    m_scale(Vector3(1.0F, 1.0F, 1.0F)),
    m_pivotPoint(Vector3(0.0F, 0.0F, 0.0F))
{
}


Transformable::Transformable(const Transformable &transformable) :
    Base(transformable),
    m_isCompositeTransformCacheValid(transformable.m_isCompositeTransformCacheValid),
    m_position(transformable.m_position),
    m_rotation(transformable.m_rotation),
    m_scale(transformable.m_scale),
    m_transform(transformable.m_transform),
    m_compositeTransformCache(transformable.m_compositeTransformCache)
{
}


Transformable::~Transformable()
{
    // Nothing to do.
}


void Transformable::SetPosition(Float x, Float y, Float z)
{
    m_position.SetX(x);
    m_position.SetY(y);
    m_position.SetZ(z);
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}


void Transformable::SetPosition(const Vector3& position)
{
    m_position = position;
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}


void Transformable::SetRotation(Float pitch, Float yaw, Float roll)
{
    m_rotation.SetX(pitch);
    m_rotation.SetY(yaw);
    m_rotation.SetZ(roll);
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}


void Transformable::SetRotation(const Vector3& rotation)
{
    m_rotation = rotation;
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}


void Transformable::SetScale(Float xScale, Float yScale, Float zScale)
{
    m_scale.SetX(xScale);
    m_scale.SetY(yScale);
    m_scale.SetZ(zScale);
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}

void Transformable::SetScale(const Vector3& scale)
{
    m_scale = scale;
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}



void Transformable::SetPivotPoint(Float xPivot, Float yPivot, Float zPivot)
{
    m_pivotPoint.SetX(xPivot);
    m_pivotPoint.SetY(yPivot);
    m_pivotPoint.SetZ(zPivot);
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}

void Transformable::SetPivotPoint(const Vector3& pivot)
{
    m_pivotPoint = pivot;
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}

void Transformable::SetGenericTransform(const Matrix4& transform)
{
    if (transform != GetGenericTransform()) {
        m_transform = transform;
        InvalidateCompositeCacheAndNotifyDerivedClasses();
    }
}

void Transformable::SetGenericTransform(const FeatStd::Optional<Matrix4>& transform)
{
    if (transform != m_transform) {
        m_transform = transform;
        InvalidateCompositeCacheAndNotifyDerivedClasses();
    }
}

const Candera::Matrix4& Transformable::GetGenericTransform() const
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(Candera::Matrix4, s_transformIdentity);
    if (m_transform.IsSet()) {
        return *m_transform;
    }
    return s_transformIdentity;
}

const Matrix4& Transformable::GetCompositeTransform() const
{
    // Compute composite transform and update composite transformation cache only, if not up-to-date.
    if (!m_isCompositeTransformCacheValid) {
        // Calculate composite transformation matrix, which is: S * R * T * G (Scale * Rotation * Translation * GenericTransformation).
        // Instead of creating and multiplying according matrices with S*R*T*G formula with performance expensive 4x4 matrix multiplications,
        // this is done by setting the transformation values in a composite matrix, directly. See below a simple sketch in which dedicated
        // rows and columns the transform values are represented in matrices:
        //
        //  Scale (S):           Rotation (R):        Translation (T):     Generic Transformation (G):
        //  -----------------    -----------------    -----------------    -----------------
        //  | S | S | S | 0 |    | R | R | R | 0 |    | 0 | 0 | 0 | 0 |    | G | G | G | G |
        //  | S | S | S | 0 |    | R | R | R | 0 |    | 0 | 0 | 0 | 0 |    | G | G | G | G |
        //  | S | S | S | 0 |    | R | R | R | 0 |    | 0 | 0 | 0 | 0 |    | G | G | G | G |
        //  | 0 | 0 | 0 | 1 |    | 0 | 0 | 0 | 1 |    | T | T | T | 1 |    | G | G | G | G |
        //  -----------------    -----------------    -----------------    -----------------

        // First step: Init the composite matrix with the rotation matrix.
        m_compositeTransformCache.SetRotationXYZ(m_rotation.GetX(), m_rotation.GetY(), m_rotation.GetZ());

        // Second step: Multiply rotation values with scaling values.
        Float scaleFactor = m_scale.GetX();
        m_compositeTransformCache(0,0) *= scaleFactor;
        m_compositeTransformCache(0,1) *= scaleFactor;
        m_compositeTransformCache(0,2) *= scaleFactor;

        scaleFactor = m_scale.GetY();
        m_compositeTransformCache(1,0) *= scaleFactor;
        m_compositeTransformCache(1,1) *= scaleFactor;
        m_compositeTransformCache(1,2) *= scaleFactor;

        scaleFactor = m_scale.GetZ();
        m_compositeTransformCache(2,0) *= scaleFactor;
        m_compositeTransformCache(2,1) *= scaleFactor;
        m_compositeTransformCache(2,2) *= scaleFactor;

        // Third step: Set translation in composite transform matrix.
        m_compositeTransformCache(3,0)= m_position.GetX() + m_pivotPoint.GetX();
        m_compositeTransformCache(3,1)= m_position.GetY() + m_pivotPoint.GetY();
        m_compositeTransformCache(3,2)= m_position.GetZ() + m_pivotPoint.GetZ();

        // Fourth step: Apply pivot point to already calculated composite transform cache.
        if (m_pivotPoint.GetX() != 0.0F) {
            m_compositeTransformCache(3, 0) -=  m_pivotPoint.GetX() * m_compositeTransformCache(0, 0);
            m_compositeTransformCache(3, 1) -=  m_pivotPoint.GetX() * m_compositeTransformCache(0, 1);
            m_compositeTransformCache(3, 2) -=  m_pivotPoint.GetX() * m_compositeTransformCache(0, 2);
        }

        if (m_pivotPoint.GetY() != 0.0F) {
            m_compositeTransformCache(3, 0) -= m_pivotPoint.GetY() * m_compositeTransformCache(1, 0);
            m_compositeTransformCache(3, 1) -= m_pivotPoint.GetY() * m_compositeTransformCache(1, 1);
            m_compositeTransformCache(3, 2) -= m_pivotPoint.GetY() * m_compositeTransformCache(1, 2);
        }

        if (m_pivotPoint.GetZ() != 0.0F) {
            m_compositeTransformCache(3, 0) -= m_pivotPoint.GetZ() * m_compositeTransformCache(2, 0);
            m_compositeTransformCache(3, 1) -= m_pivotPoint.GetZ() * m_compositeTransformCache(2, 1);
            m_compositeTransformCache(3, 2) -= m_pivotPoint.GetZ() * m_compositeTransformCache(2, 2);
        }

        if (m_transform.IsSet()) {
            // Finally, multiply with generic transformation matrix.
            m_compositeTransformCache *= *m_transform;
        }

        // Reset composite transform cache to valid state.
        m_isCompositeTransformCacheValid = true;
    }
    return m_compositeTransformCache;
}


void Transformable::Translate(const Vector3& translation)
{
    m_position += translation;
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}


void Transformable::Scale(const Vector3& scale)
{
    m_scale *= scale;
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}


void Transformable::Rotate(const Vector3& rotation)
{
    m_rotation += rotation;
    InvalidateCompositeCacheAndNotifyDerivedClasses();
}


void Transformable::TranslatePivotPoint(const Vector3& translation)
{
    SetPivotPoint(GetPivotPoint() + translation);
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(534, "Ignoring return value of function.")
    GetCompositeTransform();
}

Transformable& Transformable::operator=(const Transformable& transformable)
{
    if (this == &transformable){
        return *this;
    }
    Base::operator=(transformable);
    m_position  = transformable.m_position;
    m_rotation  = transformable.m_rotation;
    m_scale     = transformable.m_scale;
    m_transform = transformable.m_transform;
    m_pivotPoint = transformable.m_pivotPoint;
    m_isCompositeTransformCacheValid = transformable.m_isCompositeTransformCacheValid;
    m_compositeTransformCache = transformable.m_compositeTransformCache;
    return *this;
}

void Transformable::OnCompositeTransformChanged()
{
    // Nothing to do.
}

void Transformable::InvalidateCompositeCacheAndNotifyDerivedClasses()
{
    // There is no need to traverse the whole sub tree if the cache is invalid.
    // Which means that the tree has already been traversed. New nodes that are added
    // to the tree are invalidated automatically.
    if (m_isCompositeTransformCacheValid) {
        m_isCompositeTransformCacheValid = false;
        OnCompositeTransformChanged();
    }
}
} // namespace Candera
