//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StereoProjection.h"
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

StereoProjection::StereoProjection():
    PerspectiveProjection(),
    m_eyeSeparation(0.035F),
    m_convergenceDistance(1.0F)
{
}

StereoProjection::SharedPointer StereoProjection::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

    StereoProjection* ptr = FEATSTD_NEW(StereoProjection)();
    if (ptr == 0) {
        FEATSTD_LOG_ERROR("StereoscopicProjection::Create() failed, out of memory.");
    }
    StereoProjection::SharedPointer sharedPointer(ptr);
    return sharedPointer;
}

StereoProjection::~StereoProjection()
{
}

Projection::SharedPointer StereoProjection::Clone() const
{
    return SharedPointer(FEATSTD_NEW(StereoProjection)(*this));
}

StereoProjection::StereoProjection(const StereoProjection& other):
    PerspectiveProjection(other),
    m_eyeSeparation(other.m_eyeSeparation),
    m_convergenceDistance(other.m_convergenceDistance)
{
}

FEATSTD_RTTI_DEFINITION(StereoProjection, PerspectiveProjection)

void StereoProjection::SetEyeSeparation(Float separation)
{
    m_eyeSeparation = separation;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

void StereoProjection::SetConvergenceDistance(Float distance)
{
    m_convergenceDistance = distance;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

void StereoProjection::UpdateProjectionMatrix() const
{
    if (m_isMatrixUpToDate) {
        return;
    }
    Matrix4 stereoscopicProjection = Math3D::SetStereoProjection(stereoscopicProjection,
        GetNearZ(), GetFarZ(), GetFovYDegrees(), GetAspectRatio(), GetEyeSeparation(), GetConvergenceDistance());
    SetProjectionMatrix(stereoscopicProjection);
    m_isMatrixUpToDate = true;
    

}
}

