//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StereoCamera.h"
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/Core/StereoProjection.h>
#include <Candera/Engine3D/Core/OrthographicProjection.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Cloning/TreeCloner.h>
#include <Candera/System/Mathematics/Matrix3.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Container/Vector.h>

namespace Candera {
    void StereoCamera::StereoCameraSceneListener::OnSceneActivated(Scene* scene)
    {
        if (scene != 0) {
            if (m_stereoCam != 0) {
                m_stereoCam->Update();
            }
        }
    }

    StereoCamera* StereoCamera::Create()
    {
        return FEATSTD_NEW(StereoCamera);
    }

    StereoCamera* StereoCamera::Create(const Camera* camera)
    {
        StereoCamera* ptr = 0;
        if (camera != 0) {
            Node* leftNode = camera->Clone();
            Camera* left = Dynamic_Cast<Camera*>(leftNode);
            Node* rightNode = camera->Clone();
            Camera* right = Dynamic_Cast<Camera*>(rightNode);

            ptr = FEATSTD_NEW(StereoCamera);
            bool success = (ptr != 0) && (right != 0) && (left != 0);
            if (success) {
                success = ptr->SetLeftEye(left);
                if (success) {
                    left->SetAppearance(camera->GetAppearance());
                }
                success = ptr->SetRightEye(right);
                if (success) {
                
                    right->SetAppearance(camera->GetAppearance());
                }

                static_cast<Transformable&>(*ptr) = *camera;
                
                for (const Node* child = camera->GetFirstChild(); child != 0; child = child->GetNextSibling()) {
                    //Deep clone children.
                    TreeCloner traverser;
                    Node* clone = traverser.CreateClone(*child);
                    static_cast<void>(ptr->AddChild(clone));
                }

                ptr->SetRenderingEnabled(camera->IsRenderingEnabled());
                ptr->SetAlphaValue(camera->GetAlphaValue());
                ptr->SetCenter(camera->GetCenter());
                ptr->SetRadius(camera->GetRadius());
                Vector3 minBox;
                Vector3 maxBox;
                camera->GetBoundingBox(minBox, maxBox);
                ptr->SetBoundingBox(minBox, maxBox);
                ptr->SetRenderOrderBinAssignment(camera->GetRenderOrderBinAssignment());
                ptr->SetRenderOrderRank(camera->GetRenderOrderRank());
                ptr->SetRenderBenchmark(camera->GetRenderBenchmark());
                ptr->SetIntersectionTestEnabled(camera->IsIntersectionTestEnabled());
                ptr->SetScopeMask(camera->GetScopeMask());
            }
            else {
                if (leftNode != 0) {
                    leftNode->Dispose();
                }
                if (rightNode != 0) {
                    rightNode->Dispose();
                }
                if (ptr != 0) {
                    FEATSTD_DELETE(ptr);
                    ptr = 0;
                }
            }
        }

        return ptr;
    }

    StereoCamera::StereoCamera():
        m_isCameraUpdateNeeded(true),
        m_leftEye(0),
        m_rightEye(0),
        m_eyeSeparation(0.0F),
        m_convergenceDistance(1.0F)
    {
        m_listener.SetStereoCamera(this);
    }

    StereoCamera::~StereoCamera()
    {
        //No need to dispose cameras manually, as they are part of the scene graph.
        m_leftEye = 0;
        m_rightEye = 0;

        Scene* scene = GetScene();
        if (scene != 0) {
            static_cast<void>(scene->RemoveSceneListener(&m_listener));
        }
    }

    bool StereoCamera::SetLeftEye(Camera* leftEye)
    {
        bool isSuccess = true;

        //Remove left eye child, if there is any.
        if (m_leftEye != 0) {
            static_cast<void>(RemoveChild(m_leftEye));
            m_leftEye = 0;
        }

        //Add child if not already done.
        if (!IsParentOf(leftEye)) {
            if (leftEye != 0) {
                isSuccess = AddChild(leftEye);
            }
        }

        if (isSuccess) {
            m_leftEye = leftEye;
            m_isCameraUpdateNeeded = true;
        }

        return isSuccess;
    }

    bool StereoCamera::SetRightEye(Camera* rightEye)
    {
        bool isSuccess = true;

        //Remove right eye child, if there is any.
        if (m_rightEye != 0) {
            static_cast<void>(RemoveChild(m_rightEye));
            m_rightEye = 0;
        }

        //Add child if not already done.
        if (!IsParentOf(rightEye)) {
            if (rightEye != 0) {
                isSuccess = AddChild(rightEye);
            }
        }

        if (isSuccess) {
            m_rightEye = rightEye;
            m_isCameraUpdateNeeded = true;
        }

        return isSuccess;
    }

    void StereoCamera::SetEyeSeparation(Float separation)
    {
        m_eyeSeparation = separation;
        m_isCameraUpdateNeeded = true;
    }

    void StereoCamera::SetConvergenceDistance(Float distance)
    {
        m_convergenceDistance = distance;
        m_isCameraUpdateNeeded = true;
    }

    void StereoCamera::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    StereoCamera* StereoCamera::Clone() const
    {
        return FEATSTD_NEW(StereoCamera)(*this);
    }

    StereoCamera::StereoCamera(const Candera::StereoCamera& other) :
        Base(other),
        ProjectionListener(other),
        m_isCameraUpdateNeeded(true),
        m_leftEye(0),
        m_rightEye(0),
        m_eyeSeparation(other.m_eyeSeparation),
        m_convergenceDistance(other.m_convergenceDistance)
    {
    }

    void StereoCamera::OnProjectionChanged(Projection* /*projection*/)
    {
        m_isCameraUpdateNeeded = true;
    }

    void StereoCamera::Update()
    {
        if (m_isCameraUpdateNeeded) {
            if ((m_leftEye != 0) && (m_rightEye != 0)) {
                Float singleEyeSeparation = m_eyeSeparation * 0.5F;

                //Update cameras for both eyes, and set the according stereo projection.

                StereoProjection::SharedPointer leftProjection = StereoProjection::Create();
                if (m_leftEye->GetProjection() != 0) {
                    if (m_leftEye->GetProjection()->IsTypeOf(PerspectiveProjection::GetTypeId())) {
                        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
                        PerspectiveProjection* projection = static_cast<PerspectiveProjection*>(m_leftEye->GetProjection().GetPointerToSharedInstance());
                        leftProjection->SetAspectRatio(projection->GetAspectRatio());
                        leftProjection->SetFovYDegrees(projection->GetFovYDegrees());
                        leftProjection->SetNearZ(projection->GetNearZ());
                        leftProjection->SetFarZ(projection->GetFarZ());
                    }
                    else if (m_leftEye->GetProjection()->IsTypeOf(OrthographicProjection::GetTypeId())) {
                        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
                        OrthographicProjection* projection = static_cast<OrthographicProjection*>(m_leftEye->GetProjection().GetPointerToSharedInstance());
                        leftProjection->SetNearZ(projection->GetNearZ());
                        leftProjection->SetFarZ(projection->GetFarZ());
                    }
                    else {
                        //do nothing
                    }
                }

                leftProjection->SetEyeSeparation(-singleEyeSeparation);
                leftProjection->SetConvergenceDistance(m_convergenceDistance);
                m_leftEye->SetProjection(leftProjection);
                static_cast<void>(m_leftEye->SetLookAtVector(Vector3(0.0F, 0.0F, -1.0F))); //Restore default look at and up vector, should be the same for both cameras.
                static_cast<void>(m_leftEye->SetUpVector(Vector3(0.0F, 1.0F, 0.0F)));
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::Rectangle::SetPosition, False positive because here SetPosition is not called from Candera::Rectangle.)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Rectangle::SetPosition, False positive because here SetPosition is not called from Candera::Rectangle.)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, float = struct, False positive because an overloaded function SetPosition is called that accepts Vector3.)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Rectangle = Candera::Camera, False positive because here no function matches with Candera::Rectangle.)
                m_leftEye->SetPosition(m_leftEye->GetRightVector() * (-singleEyeSeparation)/*leftProjection->GetViewSpaceEyeSeparation()*/);

                StereoProjection::SharedPointer rightProjection = StereoProjection::Create();
                if (m_rightEye->GetProjection() != 0) {
                    if (m_rightEye->GetProjection()->IsTypeOf(PerspectiveProjection::GetTypeId())) {
                        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
                        PerspectiveProjection* projection = static_cast<PerspectiveProjection*>(m_rightEye->GetProjection().GetPointerToSharedInstance());
                        rightProjection->SetAspectRatio(projection->GetAspectRatio());
                        rightProjection->SetFovYDegrees(projection->GetFovYDegrees());
                        rightProjection->SetNearZ(projection->GetNearZ());
                        rightProjection->SetFarZ(projection->GetFarZ());
                    }
                    else if (m_rightEye->GetProjection()->IsTypeOf(OrthographicProjection::GetTypeId())) {
                        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
                        OrthographicProjection* projection = static_cast<OrthographicProjection*>(m_rightEye->GetProjection().GetPointerToSharedInstance());
                        rightProjection->SetNearZ(projection->GetNearZ());
                        rightProjection->SetFarZ(projection->GetFarZ());
                    }
                    else {
                        // Only orthographic and perspective projection supported.
                    }
                }

                rightProjection->SetEyeSeparation(singleEyeSeparation);
                rightProjection->SetConvergenceDistance(m_convergenceDistance);
                m_rightEye->SetProjection(rightProjection);
                static_cast<void>(m_rightEye->SetLookAtVector(Vector3(0.0F, 0.0F, -1.0F))); //Restore default look at and up vector, should be the same for both cameras.
                static_cast<void>(m_rightEye->SetUpVector(Vector3(0.0F, 1.0F, 0.0F)));
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::Rectangle::SetPosition, False positive because here SetPosition is not called from Candera::Rectangle.)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Rectangle::SetPosition, False positive because here SetPosition is not called from Candera::Rectangle.)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, float = struct, False positive because an overloaded function SetPosition is called that accepts Vector3.)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Rectangle = Candera::Camera, False positive because here no function matches with Candera::Rectangle.)
                m_rightEye->SetPosition(m_rightEye->GetRightVector() * singleEyeSeparation/*rightProjection->GetViewSpaceEyeSeparation()*/);

                m_isCameraUpdateNeeded = false;
            }
        }
    }

    /**
        *  Overrides Node::OnAncestorAdded
        *  @param scene Scene to which LOD scene listener shall be added.
        */
    void StereoCamera::OnAncestorAdded(Scene* scene)
    {
        if (scene != 0) {
            static_cast<void>(scene->AddSceneListener(&m_listener));
        }
    }

    /**
        *  Overrides Node::OnAncestorRemoved
        *  @param scene Scene to remove LOD scene listener from.
        */
    void StereoCamera::OnAncestorRemoved(Scene* scene)
    {
        if (scene != 0) {
            static_cast<void>(scene->RemoveSceneListener(&m_listener));
        }
    }

    FEATSTD_RTTI_DEFINITION(StereoCamera, Base)
} // namespace Candera

