//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SkyBox.h"
#include <Candera/Engine3D/Core/VertexGeometryBuilder.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    static const Float c_cubeData[8][3] = {
        { -5.0F, 5.0F, 5.0F},  //0
        { 5.0F, 5.0F, 5.0F},  //1
        { -5.0F, -5.0F, 5.0F}, //2
        { 5.0F, -5.0F, 5.0F},  //3
        { -5.0F, 5.0F, -5.0F}, //4
        { 5.0F, 5.0F, -5.0F},  //5
        { -5.0F, -5.0F, -5.0F},//6
        { 5.0F, -5.0F, -5.0F}  //7
    };

    static const UInt16 c_cubeIndexData[] = {3, 2, 1, 0, 4, 2, 6, 3, 7, 1, 5, 4, 7, 6 };

    static UInt8 s_upColor[4] = {0, 200, 255, 255};
    static UInt8 s_downColor[4] = {0, 210, 50, 255};
    static UInt8 s_leftColor[4] = {255, 0, 0, 255};
    static UInt8 s_rightColor[4] = {0, 0, 255, 255};
    static UInt8 s_frontColor[4] = {0, 0, 0, 255};
    static UInt8 s_backColor[4] = {255, 255, 255, 255};
    static Bitmap::SharedPointer s_defaultFaceUp = Bitmap::Create(1,1,Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, s_upColor,0);
    static Bitmap::SharedPointer s_defaultFaceDown = Bitmap::Create(1,1,Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, s_downColor,0);
    static Bitmap::SharedPointer s_defaultFaceLeft = Bitmap::Create(1,1,Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, s_leftColor,0);
    static Bitmap::SharedPointer s_defaultFaceRight = Bitmap::Create(1,1,Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, s_rightColor,0);
    static Bitmap::SharedPointer s_defaultFaceFront = Bitmap::Create(1,1,Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, s_frontColor,0);
    static Bitmap::SharedPointer s_defaultFaceBack = Bitmap::Create(1,1,Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, s_backColor,0);

    SharedPointer<SkyBox> SkyBox::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        SkyBox* ptr = FEATSTD_NEW(SkyBox);

        if (ptr == 0) {
            FEATSTD_LOG_ERROR("SkyBox create failed, out of memory.");
            return SharedPointer<SkyBox>();
        }

        ptr->CreateSkyBoxMesh();
        SharedPointer<SkyBox> sharedPointer(ptr);
        return sharedPointer;
    }

    SkyBox::~SkyBox()
    {
        if (m_skyBoxMesh != 0) {
            m_skyBoxMesh->Dispose();
            m_skyBoxMesh = 0;
        }
    }

    bool SkyBox::Activate()
    {
        bool result = false;

        if (m_skyBoxMesh != 0) {
            m_skyBoxMesh->SetPosition(RenderDevice::GetActiveCamera()->GetWorldPosition());
            m_skyBoxMesh->Render();
            result = true;
        }

        return result;
    }

    bool SkyBox::Upload()
    {
        return (m_skyBoxMesh != 0) ? m_skyBoxMesh->Upload() : true;
    }

    bool SkyBox::Unload()
    {
        return (m_skyBoxMesh != 0) ? m_skyBoxMesh->Unload() : true;
    }

    void SkyBox::SetSkyBoxMesh(Mesh* skybox) {
        m_skyBoxMesh = skybox;
    }

    void SkyBox::CreateSkyBoxMesh()
    {
        VertexGeometryBuilder builder;
        bool success;

        builder.Clear();
        success = builder.SetVertexElementRange(VertexGeometry::Position, 0, 0, VertexGeometry::Float32_3,
            8, 3*sizeof(Float), c_cubeData );
        success = success && builder.SetIndexElementRange(0, sizeof(c_cubeIndexData) / sizeof(UInt16), c_cubeIndexData);

        SharedPointer<VertexBuffer> vb = VertexBuffer::Create();
        success = success && (vb != 0);
        if (success) {
            VertexGeometry* cubeGeometry = builder.GetVertexGeometry();
            success = vb->SetVertexGeometry(cubeGeometry, VertexBuffer::VertexGeometryDisposer::Dispose);
            vb->SetPrimitiveType(VertexBuffer::TriangleStrip);
        }

        SharedPointer<Appearance> appearance = Appearance::Create();
        SharedPointer<RenderMode> renderMode = RenderMode::Create();
        success = success && (appearance != 0) && (renderMode != 0);
        if (success) {
            appearance->SetRenderMode(renderMode);
            appearance->GetRenderMode()->SetCulling(RenderMode::BackFaceCulling);
            appearance->GetRenderMode()->SetDepthTestEnabled(false);
            appearance->GetRenderMode()->SetDepthWriteEnabled(false);
        }

        SharedPointer<CubeMapTextureImage> cubeTextureImage = CubeMapTextureImage::Create();
        success = success && (cubeTextureImage != 0);
        success = success && cubeTextureImage->SetBitmapFace(CubeMapTextureImage::CubeMapPositiveXFace, s_defaultFaceRight);
        success = success && cubeTextureImage->SetBitmapFace(CubeMapTextureImage::CubeMapNegativeXFace, s_defaultFaceRight);
        success = success && cubeTextureImage->SetBitmapFace(CubeMapTextureImage::CubeMapPositiveYFace, s_defaultFaceRight);
        success = success && cubeTextureImage->SetBitmapFace(CubeMapTextureImage::CubeMapNegativeYFace, s_defaultFaceRight);
        success = success && cubeTextureImage->SetBitmapFace(CubeMapTextureImage::CubeMapPositiveZFace, s_defaultFaceRight);
        success = success && cubeTextureImage->SetBitmapFace(CubeMapTextureImage::CubeMapNegativeZFace, s_defaultFaceRight);

        SharedPointer<Texture> cubeMapTexture = Texture::Create();
        success = success && (cubeTextureImage != 0);
        if (success) {
            cubeMapTexture->SetTextureImage(cubeTextureImage);
            cubeMapTexture->SetWrapModeU(Texture::ClampToEdge);
            cubeMapTexture->SetWrapModeV(Texture::ClampToEdge);
            success = appearance->SetTexture(cubeMapTexture);
        }

        SharedPointer<GenericShaderParamSetter> skyBoxSPS = GenericShaderParamSetter::Create();
        success = success && (skyBoxSPS != 0);
        if (success) {
            skyBoxSPS->SetLightActivationEnabled(false);
            appearance->SetShaderParamSetter(skyBoxSPS);
        }

        if (success) {
            m_skyBoxMesh = Mesh::Create();
            if (m_skyBoxMesh != 0) {
                m_skyBoxMesh->SetVertexBuffer(vb);
                m_skyBoxMesh->SetAppearance(appearance);
            }
        }
    }
} // namespace Candera
