//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SHADER_H)
#define CANDERA_SHADER_H

#include <Candera/Engine3D/Core/3DStringBufferAppenders.h>
#include <Candera/Engine3D/Core/DeviceObject.h>
#include <Candera/Engine3D/Core/ShaderParamNames.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/EngineBase/Common/InstanceId.h>
#include <Candera/EngineBase/Common/ResourceObject.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <FeatStd/Util/Guid.h>

namespace Candera {
    /**  @addtogroup Core3D
     *   @{
     */

    namespace Internal { class UniformNode; }
    class VertexBuffer;

    /**
     *  @brief   The class Shader is an Appearance component representing a graphical processing unit (GPU) program, thus a
     *           vertex and fragment shader pair. The Shader program can be parametrized with uniforms and attributes.
     *           It's guaranteed by the shader that the program is only created once in VRAM.
     *           To enable sharing of Shader objects and to prevent multiple uploads the shader implements a reference counting,
     *           it's only upload once and delete only if the ref count gets 0.
     *
     */
    class Shader : public DeviceObject {
            FEATSTD_TYPEDEF_BASE(DeviceObject);

        public:
            /**
             * ShaderResource is a ResourceObject that provides access to the
             *  vertex/fragment shader data.
             *
             * To create a resource handle, call ShaderResource::CreateHandle with the following parameters:
             * - data: Depending on the configuration, can be source-text or device dependent binary.
             * - disposer: The function which defines the dispose method for the data.
             *    The function will be called on Shader::Dispose call or on Shader destruction.
             * - size: Size of the data in bytes.
             *
             * @see ResourceObject
             */
            typedef ResourceObject<const void> ShaderResource;
            typedef ShaderResource::DisposerFunction DisposerFn;

            /**
             *  Creates an instance of this class.
             *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
             */
            static MemoryManagement::SharedPointer<Shader> Create();

            /**
             *  Destructor
             */
            virtual ~Shader() override;

            /**
             *  UniformType defines the data type of an uniform.
             */
            enum UniformType {
                Float,       ///< Float.
                FloatVec2,   ///< Float 2d vector.
                FloatVec3,   ///< Float 3d vector.
                FloatVec4,   ///< Float 4d vector.

                Integer,     ///< Integer.
                IntegerVec2, ///< Integer 2d vector.
                IntegerVec3, ///< Integer 3d vector.
                IntegerVec4, ///< Integer 4d vector.

                Bool,        ///< Boolean.
                BoolVec2,    ///< Boolean 2d vector.
                BoolVec3,    ///< Boolean 3d vector.
                BoolVec4,    ///< Boolean 4d vector.

                FloatMat2,   ///< 2x2 Float matrix.
                FloatMat3,   ///< 3x3 Float matrix.
                FloatMat4,   ///< 4x4 Float matrix.

                Sampler2D,   ///< 2d texture.
                SamplerCube  ///< Cube texture.
            };

            /**
             * ConstantAttributeType defines the data type a constant attribute can have.
             * Note: Constant attributes can only have float values.
             */
            enum ConstantAttributeType {
                FloatAttribute,     ///< Float.
                FloatVec2Attribute, ///< Float 2d vector.
                FloatVec3Attribute, ///< Float 3d vector.
                FloatVec4Attribute  ///< Float 4d vector.
            };

            /**
             * BuildType describes whether a vertex or fragment shader is available as source code, binary object or binary program.
             */
            enum BuildType {
                ShaderSource = 0,       ///< Shader object is provided as source code.
                ShaderBinary = 1,       ///< Shader object is provided as precompiled binary object.
                ShaderProgramBinary = 2 ///< Shader object represents complete shader program as binary. As a convention this is stored inside the vertex shader member.
            };

            /** 
             * Object type describes what kind of shader object is queried or passed to the RenderDevice.
             */
            enum ObjectType {
                VertexShader = 0,   ///< The queried or passed object represents a vertex shader.
                FragmentShader = 1  ///< The queried or passed object represents a fragment shader.
            };

            /**
             * If a shader is using more than a single uniform, UniformCacheHandle is the fastest way to access uniform locations.
             */
            struct UniformCacheHandle
            {
                UniformCacheHandle() : m_uniformCacheHandle(-1) {}
                bool IsValid() const { return m_uniformCacheHandle >= 0; }
            private:
                friend class Shader;
                UniformCacheHandle(Int uniformCacheHandle) : m_uniformCacheHandle(uniformCacheHandle) {}
                Int m_uniformCacheHandle;
            };

            /**
             *  Sets this Shader program as active in the Render Device.
             *  @return Whether the shader was activated (true), or not(false).
             */
            bool Activate() const;

            /**
             *  Binds the attributes from VertexBuffer object to the shader attribute names.
             *  @param vertexBuffer Vertex buffer whose attributes to bind to the shader attribute names.
             *  @param divisor
             *  @return Whether binding of vertex buffer object attributes to shader attributes succeeded(true) or not (false).
             */
            bool BindAttributes(const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer, UInt divisor = 0);

            /**
             *  Sets the vertex shader. Depending on the configuration, the vertex shader can be source-text
             *  or device dependent binary. Call this function before calling Shader::CreateProgram.
             *  @param vertexShader       The vertex shader.
             *  @param disposerFn         The function which defines the dispose method for the vertex shader.
             *                            The method Disposer::Dispose will be called on destruction.
             *                            If the Disposer is set to null the vertex shader will not be disposed.
             *  @return                   False if the vertex shader is already uploaded, otherwise it is set and returns true.
             *                            Altering the shader after upload is not allowed.
             */
            bool SetVertexShader(const void* vertexShader, DisposerFn disposerFn);

            /**
             * Sets the vertex ShaderResource handle.
             *
             * @param vertexShaderResourceHandle resource handle for accessing vertex shader.
             * @return False if the vertex shader is already uploaded, otherwise it is set and returns true.
             *   Altering the shader after upload is not allowed.
             */
            bool SetVertexShaderResourceHandle(const ResourceDataHandle& vertexShaderResourceHandle) {
                return SetShaderResourceHandle(m_vertexShaderResourceHandle, vertexShaderResourceHandle);
            }

            /**
             * @return The vertex ShaderResource handle of this shader program.
             * Create a ShaderResource for accessing the shader data.
             */
            const ResourceDataHandle& GetVertexShaderResourceHandle() const {
                return m_vertexShaderResourceHandle;
            }

            /**
             *  Sets the fragment shader. Depending on the configuration, the vertex shader can be source-text
             *  or device dependent binary. Call this function before calling Shader::CreateProgram.
             *  @param fragmentShader     The fragment shader.
             *  @param disposerFn         The function which defines the dispose method for the fragment shader.
             *                            The method Disposer::Dispose will be called on destruction.
             *                            If the Disposer is set to null the fragment shader will not be disposed.
             *  @return                   False if the fragment shader is already uploaded, otherwise it is set and returns true.
             *                            Altering the shader after upload is not allowed.
             */
            bool SetFragmentShader(const void* fragmentShader, DisposerFn disposerFn);

            /**
             * Sets the fragment ShaderResource handle.
             *
             * @param fragmentShaderResourceHandle resource handle for accessing fragment shader.
             * @return False if the fragment shader is already uploaded, otherwise it is set and returns true.
             *   Altering the shader after upload is not allowed.
             */
            bool SetFragmentShaderResourceHandle(const ResourceDataHandle& fragmentShaderResourceHandle) {
                return SetShaderResourceHandle(m_fragmentShaderResourceHandle, fragmentShaderResourceHandle);
            }

            /**
             * @return The fragment ShaderResource handle of this shader program.
             * Create a ShaderResource for accessing the shader data.
             */
            const ResourceDataHandle& GetFragmentShaderResourceHandle() const {
                return m_fragmentShaderResourceHandle;
            }


            /**
             *  Get the size of the given uniform type.
             *  @param uniformType  The type of the uniform to return the size for.
             *  @return The size of the given uniform type.
             */
            static SizeType GetSize(UniformType uniformType);

            /**
             *  Sets an arbitrary uniform in the current shader program, specified by name.
             *  @param   data                 The data to bind to the uniform.
             *  @param   uniformName          The uniform name. If not available, then the uniform is not set.
             *  @param   type                 The data type of the uniform.
             *  @param   count                The data count regarding to the type.
             *  @return true if uniform was set successfully, false otherwise.
            */
            bool SetUniform(const void* data, const Char* uniformName, UniformType type, UInt count = 1);

            /**
             *  Sets an arbitrary uniform in the current shader program, specified by uniformLocation.
             *  @param   data                 The data to bind to the uniform.
             *  @param   uniformLocation      The uniform location. If uniformLocation is -1 the uniform data passed in is silently ignored.
             *  @param   type                 The data type of the uniform.
             *  @param   count                The data count regarding to the type.
             *  @param   instanceIndex        The index of the instance array to set this uniform for. Index -1 sets the uniform for immediate use.
             *  @return true if uniform was set successfully, false otherwise.
             */
            bool SetUniform(const void* data, Int uniformLocation, UniformType type, UInt count = 1, Int instanceIndex = -1);

            /**
             *  Returns uniform location for given uniform name.
             *  @param uniformName specifies the name of the uniform.
             *  @param location of the given uniform name. If uniform name is not available then the value "-1" is returned.
             *  @return true if uniform location was found, false otherwise.
             */
            bool GetUniformLocation(const Char* uniformName, Int& location) const;

            /**
             *  Returns uniform location for given uniform semantic with a worst-case complexity of O(n), where n is the number of uniforms of the shader.
             *  @param uniformSemantic specifies the semantic of the uniform.
             *  @param location of the given uniform semantic. If uniform semantic is not available then the value "-1" is returned.
             *  @return true if uniform location was found, false otherwise.
             */
            bool GetUniformLocation(ShaderParamNames::UniformSemantic uniformSemantic, Int& location) const;

            /**
             *  Returns the uniform location for the uniform specified by the given uniform cache handle and index.
             *  @param uniformCacheHandle    The UniformCacheHandle that was acquired via RegisterUniformCacheHandle.
             *  @param accessorUniformIndex  The index used to identify the uniform that was given to RegisterUniformCacheHandle.
             *  @param location              Reference to where the uniform location for the given uniform cache handle and index will be returned in.
             *                               If the uniform is not available, a value of "-1" is returned.
             *  @return True, if uniform location was found, False otherwise.
             */
            bool GetUniformLocation(UniformCacheHandle uniformCacheHandle, UInt accessorUniformIndex, Int& location) const
            {
                FEATSTD_DEBUG_ASSERT(uniformCacheHandle.IsValid());
                FEATSTD_DEBUG_ASSERT(uniformCacheHandle.m_uniformCacheHandle < static_cast<Int>(m_uniformAccessorIdCache.Size()));
                FEATSTD_DEBUG_ASSERT(m_uniformAccessorIdCache[uniformCacheHandle.m_uniformCacheHandle].m_uniformLocations != 0);
                FEATSTD_DEBUG_ASSERT(m_uniformAccessorIdCache[uniformCacheHandle.m_uniformCacheHandle].m_uniformLocationsCount > accessorUniformIndex);
                location = m_uniformAccessorIdCache[uniformCacheHandle.m_uniformCacheHandle].m_uniformLocations[accessorUniformIndex];
                return (-1 != location);
            }

            /**
             *  Returns the active uniform count in a shader program and stores it in the out-param count.
             *  @param count Active uniform count of the shader program, out parameter.
             *  @return Whether an active uniform count could be retrieved (true) or not(false).
             */
            bool GetActiveUniformCount(Int& count) const;

            /**
             *  Returns the length of the longest active uniform variable name including null termination character.
             *  If no active uniform variables exist, 0 is returned.
             *  @param maxLength Returns the length of longest uniform variable name.
             *  @return Success if no error occurred.
             */
            bool GetActiveUniformMaxLength(Int& maxLength) const;

            /**
             *  Returns a description (name, name length, size, type) of an active uniform given by an index.
             *  @param index              Specifies the index of the uniform variable to be queried.
             *  @param nameBufferSize     The maximum number of characters the RenderDevice is allowed to write in the character buffer indicated by name including the null terminator.
             *  @param name               Returns a null terminated string containing the name of the uniform variable. A uniform array reports the uniform name only.
             *                            Structures and array of structures are reduced to its fundamental components containing "." and "[]" operators.
             *                            Thus, structures, an array of structures, or a subcomponent of a vector or matrix are no valid names.
             *  @param nameLength         Returns the actual number of characters written by RenderDevice in name string excluding the null terminator.
             *  @param size               Returns the size of the uniform variable. Uniform variables other than arrays have size 1. Arrays of structures are
             *                            reduced to its fundamental components including "." and "[]" operators, which have size 1, if they are no arrays.
             *  @param type               Returns the type of the uniform variable
             *  @return                   Whether the uniform info could be retrieved (true) or not(false).
             */
            bool GetActiveUniformInfo(Int index, Int nameBufferSize, Char* name /*out*/,  Int& nameLength /*out*/, Int& size /*out*/, UniformType& type /*out*/) const;

            /**
             *  Register the given accessor with its array of uniforms for direct uniform location lookups.
             *  @param accessorId             A unique value to identify the accessor of the uniform cache.
             *  @param uniformSemantics       The array of uniform semantics to be registered in this shader for the given accessorId.
             *  @param uniformSemanticsCount  The number of uniform semantics given in uniformSemantics, i.e. the size of the uniformSemantics array.
             *  @param uniformCacheHandle
             *  @return True, if the UniformCacheHandle could be registered in this shader for the given accessorId, False otherwise.
             */
            bool RegisterUniformCacheAccessor(const void* const accessorId, const ShaderParamNames::UniformSemantic uniformSemantics[],
                UInt uniformSemanticsCount, UniformCacheHandle& uniformCacheHandle) const;

            /**
             *  Register the given accessor with its list of uniform nodes for direct uniform location lookups.
             *  @param accessorId        A unique value to identify the accessor of the uniform cache.
             *  @param uniformNode       The first node of a linked list of UniformNodes, all to be registered in this shader for the given accessorId.
             *  @param uniformNodeCount  The number of uniform nodes given of the linked list starting with uniformNode.
             *  @param uniformCacheHandle
             *  @return True, if the UniformCacheHandle could be registered in this shader for the given accessorId, False otherwise.
             */
            bool RegisterUniformCacheAccessor(const void* const accessorId, const Candera::Internal::UniformNode* uniformNode,
                UInt uniformNodeCount, UniformCacheHandle& uniformCacheHandle) const;

            /**
             *  Deregisters the uniform cache handle of the given accessorId from all shaders.
             *  @param accessorId  A unique value to identify the accessor of the uniform cache.
             */
            static void DeregisterUniformCacheAccessor(const void* const accessorId);

            /**
             *  Deregisters the uniform cache accessor identified by the uniform cache handle from this shader.
             *  @param uniformCacheHandle  The UniformCacheHandle to identify the accessor to deregister from this shader.
             *  @return True, if the UniformCacheHandle was valid, False otherwise.
             */
            bool DeregisterUniformCacheAccessor(UniformCacheHandle& uniformCacheHandle) const;

            /**
             *  Get the uniform cache handle for the given accessorId.
             *  @param accessorId  A unique value to identify the accessor of the uniform cache.
             *  @return A UniformCacheHandle for the given accessorId. If the uniform cache handle is not valid, the accessorId was not registered beforehand.
             */
            UniformCacheHandle GetUniformCacheHandle(const void* const accessorId) const;

            /**
             *  Return the count of registered uniforms for the given uniform cache handle.
             *  @param uniformCacheHandle  The uniform cache handle to return the count of registered uniforms for.
             *  @return The count of registered uniforms for the given uniform cache handle.
             */
            UInt GetUniformCacheHandleLocationsCount(UniformCacheHandle uniformCacheHandle) const;

            /**
             *  Returns attribute location for given attribute semantic.
             *  @param attributeSemantic specifies the semantic of the attribute.
             *  @return Attribute location, -1 if attribute doesn't exist in current shader program.
             */
            Int GetAttributeLocation(ShaderParamNames::AttributeSemantic attributeSemantic) const;

            /**
             *  Sets an arbitrary constant attribute in the current shader program, specified by name.
             *  The term constant refers to being one value for all vertices processed by the current shader.
             *  This method exists to provide the possibility to set values of arbitrary constant attributes.
             *  However here the attribute location is not cached and has to be queried directly from the shader,
             *  use attributes with known semantics instead.
             *  Note: Constant vertex attributes overrule vertex arrays if they are present,
             *        if a constant vertex attribute is activated, it will be applied.
             *        This only works inside an OpenGL ES 2.0 or OpenGL 3.x Core or OpenGL 4.x Core context.
             *        Incompatibilities to OpenGL 2.0, OpenGL 3.x Compatibility profile or OpenGL 4.x Compatibility profile reside.
             *  @param   data         The data to bind to the attribute.
             *  @param   name         The attributeName name if different from the default attribute name.
             *  @param   type         The data type of the attribute.
             *  @return true if attribute was set successfully, false otherwise.
             */
            bool SetConstantVertexAttribute(const void* data, const Char* name, Shader::ConstantAttributeType type) const;

            /**
             *  Sets an arbitrary constant attribute in the current shader program, specified by attribute semantic.
             *  The term constant refers to being one value for all vertices processed by the current shader.
             *  Attribute semantic locations cache is considered.
             *  Note: Constant vertex attributes overrule vertex arrays if they are present,
             *        if a constant vertex attribute is activated, it will be applied.
             *        This only works inside an OpenGL ES 2.0 or OpenGL 3.x Core or OpenGL 4.x Core context.
             *        Incompatibilities to OpenGL 2.0, OpenGL 3.x Compatibility profile or OpenGL 4.x Compatibility profile reside.
             *  @param   data                 The data to bind to the attribute.
             *  @param   attributeSemantic    The attribute semantic. If attributeSemantic is not present in attribute cache, the data passed in is silently ignored.
             *  @param   type                 The data type of the attribute.
             *  @return true if attribute was set successfully, false otherwise.
             */
            bool SetConstantVertexAttribute(const void* data, ShaderParamNames::AttributeSemantic attributeSemantic, ConstantAttributeType type) const;

            /**
             *  Sets an arbitrary constant attribute in the current shader program, specified by it's location.
             *  The term constant refers to being one value for all vertices processed by the current shader.
             *  Note: Constant vertex attribute only have an effect if a vertex array
             *  with the same semantic doesn't exist or is deactivated manually, arrays in the vertex buffer are preferred.
             *  @param   data                 The data to bind to the attribute.
             *  @param   attributeLocation    The attributes location in the shader program.
             *  @param   type                 The data type of the attribute.
             *  @return true if attribute was set successfully, false otherwise.
             */
            bool SetConstantVertexAttribute(const void* data, Int attributeLocation, ConstantAttributeType type) const;

            /**
             *  Retrieves handle to shader program.
             *  @return the handle to shader program associated with the active pool.
             */
            Handle GetProgramHandle() const;

            /**
             *  Retrieves handle to vertex shader.
             *  @return the handle to vertex shader associated with the active pool.
             */
            Handle GetVertexShaderHandle() const;

            /**
             *  Retrieves handle to fragment shader.
             *  @return the handle to fragment shader associated with the active pool.
             */
            Handle GetFragmentShaderHandle() const;

            /**
             * Queries whether the stored vertex or fragment shader is available as source code, pre built binary object or as prebuilt program.
             * @param objectType Whether to query the stored vertex or fragment shader.
             * @param shaderBuildType Out parameter that returns the query result.
             * @return Boolean whether the query was successful or not (e.g. fail if stored pointer is 0).
             */
            bool GetBuildType(ObjectType objectType, BuildType& shaderBuildType) const;
            
            /**
            * Stores a globally unique id for this shader   
            *  @param guid to store. 
            **/
            void SetGuid( FeatStd::Internal::Guid const & guid) { m_guid = guid; }

            /**
            * Retrieves the globally unique id for this shader
            *  @return const reference to the Guid.
            **/
            const FeatStd::Internal::Guid& GetGuid() const { return m_guid; }

            /**
             *  Updates the internal cached attribute locations with regards to the attribute names defined in function ShaderParamNames::SetAttributeName.
             */
            void UpdateAttributeCache();

            /**
             *  Updates the internal cached uniform locations with regards to the uniform names defined in function ShaderParamNames::SetUniformName.
             */
            void UpdateUniformCache();

            /**
             * Sets the maximum number of instances this shader can support. The default value is 1. If the shader if suitable for instancing, the
             * value should be set to the size of the uniform arrays which are indexed using the instance ID in the shader (e.g. gl_InstanceID in OpenGL).
             * @param maximumInstanceCount  the maximum number of instances that this shader can handle.
             */
            void SetMaximumInstanceCount(UInt maximumInstanceCount) { m_maxInstanceCount = maximumInstanceCount; }

            /**
             * Gets the maximum number of instances this shader can support.
             * @return The maximum number of instances this shader can be used with.
             */
            UInt GetMaximumInstanceCount() const { return m_maxInstanceCount; }

            /**
             * Returns the Id of this vertex buffer instance.
             * @return The Id of this vertex buffer instance.
             */
            Int GetInstanceId() const { return m_instanceId; }

            /**
             * Return the current binding point available for binding.
             * @return  The current binding point available for binding.
             */
            UInt GetCurrentBindingPoint() const { return m_currentBindingPoint; }

            /**
             * Increment the current binding point available for binding. This should be done
             * whenever a current binding point was used to bind a buffer to the shader.
             */
            void IncrementCurrentBindingPoint() const { ++m_currentBindingPoint; }

            /**
             * Set the transform varyings for the shader.
             * @param transformVaryings      An array of 'tansformVaryingsCount' zero-terminated strings specifying
             *                               the names of the varying variables to use for transform feedback.
             * @param tansformVaryingsCount  The size of the transform varyings array.
             * @return                       False if the shader has already been uploaded.
             */
            bool SetTransformVaryings(const Char** transformVaryings, SizeType tansformVaryingsCount);

            /**
             * Get the transform varyings from the shader
             * @return  An array of GetTransformVaryingsCount() zero-terminated strings specifying the names of
             *          the varying variables to use for transform feedback, or 0.
             */
            const Char** GetTransformVaryings() const { return m_transformVaryings; }

            /**
             * Get the size of the transform varyings array.
             */
            SizeType GetTransformVaryingsCount() const { return m_transformVaryingsCount; }

            CANDERA_DEPRECATED_3_0_2("Retrieve shader code using Shader::ShaderResource(GetVertexShaderResourceHandle()).GetData()",
                const void* GetVertexShader() const);
            CANDERA_DEPRECATED_3_0_2("Retrieve shader code using Shader::ShaderResource(GetFragmentShaderResourceHandle()).GetData()",
                const void* GetFragmentShader() const);

            FEATSTD_RTTI_DECLARATION();

        protected:
            /**
             *  overridden from DeviceObject
             *  @return Whether compiling, uploading to VRAM and linking of the shader program succeeded (true) or not(false).
             */
            virtual bool UploadInternal(LoadingHint loadingHint) override;
            /**
             *  overridden from DeviceObject
             *  @return Whether deleting of the shader program from VRAM succeeded(true) or not(false).
             */
            virtual bool UnloadInternal(LoadingHint loadingHint) override;

            // override DeviceObject::DisposeInternal
            virtual void DisposeInternal() override;

        private:
            friend class RenderDevice;

            Candera::Internal::InstanceId<Shader> m_instanceId;

            // Handles
            Handle m_programMemoryHandle[CANDERA_MAX_CONTEXT_COUNT];
            Handle m_vertexShaderMemoryHandle[CANDERA_MAX_CONTEXT_COUNT];
            Handle m_fragmentShaderMemoryHandle[CANDERA_MAX_CONTEXT_COUNT];

            ResourceDataHandle m_vertexShaderResourceHandle;
            ResourceDataHandle m_fragmentShaderResourceHandle;

            FeatStd::Internal::Guid m_guid;
            // The maximum number of instances this shader can support.
            UInt m_maxInstanceCount;

            // Currently available binding point.
            mutable UInt m_currentBindingPoint;

            const Char** m_transformVaryings;
            SizeType m_transformVaryingsCount;

            // Auto Uniform Cache
            struct AutoUniformCacheEntry {
                ShaderParamNames::UniformSemantic m_uniformSemantic;  // uniform semantic that points to corresponding uniform string.
                Int m_uniformLocation;   // Uniform location.
                AutoUniformCacheEntry() : m_uniformSemantic(ShaderParamNames::UniformSemanticCount), m_uniformLocation(-1) {};
                AutoUniformCacheEntry(ShaderParamNames::UniformSemantic uniformSemantic, Int uniformLocation) : m_uniformSemantic(uniformSemantic), m_uniformLocation(uniformLocation) {}
                AutoUniformCacheEntry(const AutoUniformCacheEntry& uniformCacheEntry) : m_uniformSemantic(uniformCacheEntry.m_uniformSemantic), m_uniformLocation(uniformCacheEntry.m_uniformLocation) {}
                ~AutoUniformCacheEntry() {}
            };

            typedef FeatStd::Internal::SingleLinkedList<const AutoUniformCacheEntry> AutoUniformCache;
            AutoUniformCache m_autoUniformCache;

            // Custom Uniform Cache (i.e. all uniforms that are not auto uniforms. This also excludes individual uniform
            // elements of a uniform array)
            struct CustomUniformCacheEntry {
                CustomUniformCacheEntry(const Char* uniformName, UInt32 uniformNameHash, Int uniformLocation) :
                    m_uniformName(uniformName), m_uniformNameHash(uniformNameHash), m_uniformLocation(uniformLocation) {}
                const Char* m_uniformName;
                UInt32 m_uniformNameHash;
                Int m_uniformLocation;
            private:
                CustomUniformCacheEntry() : m_uniformName(0), m_uniformNameHash(0), m_uniformLocation(-1) {}
            };

            typedef FeatStd::Internal::SingleLinkedList<const CustomUniformCacheEntry> CustomUniformCache;
            CustomUniformCache m_customUniformCache;
            void ClearCustomUniformCache();


            // Uniform cache for accessors
            struct UniformAccessorIdCacheEntry {
                UniformAccessorIdCacheEntry(const void* accessorId, Int* uniformLocations, UInt uniformLocationsCount)
                    : m_accessorId(accessorId), m_uniformLocations(uniformLocations), m_uniformLocationsCount(uniformLocationsCount) {}
                UniformAccessorIdCacheEntry() : m_accessorId(0), m_uniformLocations(0), m_uniformLocationsCount(0) {}
                const void* m_accessorId;
                Int* m_uniformLocations;
                UInt m_uniformLocationsCount;
            };

            mutable Candera::Internal::Vector<UniformAccessorIdCacheEntry, FeatStd::Internal::LinearIncreasePolicy<1> > m_uniformAccessorIdCache;
            bool VerifyCacheHandleAccessor(const void* const accessorId, Int& uniformAccessorCacheEntryIndex) const;
            void ClearUniformAccessorIdCache();

            // Attribute index table
            struct AttributeCacheEntry {
                ShaderParamNames::AttributeSemantic m_attributeSemantic;  // attribute semantic that points to corresponding attribute string.
                Int m_attributeLocation;   // Attribute location.
                AttributeCacheEntry() : m_attributeSemantic(ShaderParamNames::AttributeSemanticCount), m_attributeLocation(-1) {};
                AttributeCacheEntry(ShaderParamNames::AttributeSemantic attributeSemantic, Int attributeLocation) : m_attributeSemantic(attributeSemantic), m_attributeLocation(attributeLocation) {}
                AttributeCacheEntry(const AttributeCacheEntry& attributeCacheEntry) : m_attributeSemantic(attributeCacheEntry.m_attributeSemantic), m_attributeLocation(attributeCacheEntry.m_attributeLocation) {}
                ~AttributeCacheEntry() {}
            };

            typedef FeatStd::Internal::SingleLinkedList<const AttributeCacheEntry> AttributeCache;
            AttributeCache  m_attributeCache;

            // private because only ::Create() should be used to create an instance of this class
            Shader();
            Shader(const Shader& rhs);
            Shader& operator=(const Shader& shader);

            void RegisterInstance() const;
            void DeregisterInstance() const;

            bool SetShaderResourceHandle(ResourceDataHandle& handle, const ResourceDataHandle& source) const;

            void SetShaderObjectMemoryHandle(ObjectType type, Handle handle);
            void SetShaderProgramMemoryHandle(Handle handle);

            void AddUniformLocationsToCache();
            void AddUniformBlockIndicesToCache();

            // Make this class manageable by MemoryManagement::SharedPointer
            CANDERA_SHARED_POINTER_DECLARATION();
    };
    /** @} */ // end of Core3D
} // namespace Candera

#endif    // CANDERA_SHADER_H

