//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SCENE_H)
#define CANDERA_SCENE_H

#include <Candera/Engine3D/Core/Light.h>
#include <Candera/Engine3D/Core/Group.h>
#include <Candera/Engine3D/RenderOrder/AbstractRenderOrder.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/System/Rtti/Rtti.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

// Forward declarations.
class Canvas;
class SceneListener;

/**
 * @brief   The class Scene represents a top level scene graph node. Multiple Scenes are allowed.
 *          A Scene can not be part of any other Scene. To render a Scene, at least one Camera must be part of this
 *          Scene and the Camera must have rendering enabled. A Scene can have more than one Camera.
 *          Note: Lights added to a Scene apply for that Scene only and do not affect 3D objects in any other Scene.
 */
class Scene: public Group
{
    friend class Canvas;
    friend class Light;
    friend class Renderer;

    FEATSTD_TYPEDEF_BASE(Group);

    public:

        /**
         * Creates a Scene object.
         * @return A pointer to the generated Scene object.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::Group::Create, CANDERA_LINT_REASON_EXPLICITHIDING)
        static Scene* Create();

        /**
         * Destructor.
         */
        virtual ~Scene() override;

        /**
         * Creates a clone of the scene.
         * The Clone render order is set to 0. Note that attaching the same
         * RenderOrder to two different scenes may result in undefined behavior.
         * Light list and listener list are empty as well.
         * @return A clone of the scene.
         */
        virtual Scene* Clone() const override;

        /**
         * Set AbstractRenderOrder object for this Scene.
         * Scene takes the ownership of the AbstractRenderOrder and calls AbstractRenderOrder::Dispose() on destruction of the Scene.
         * @param renderOrder Render order to be set.
         */
        CANDERA_DEPRECATED_3_1_1(
            "Render Order is now part of Camera. Scene's Render Order is no longer used.",
        void SetRenderOrder(AbstractRenderOrder* renderOrder));

        /**
         * Retrieve the RenderOrder object used by this Scene.
         * @return A Pointer to the render order used by this Scene.
         */
        CANDERA_DEPRECATED_3_1_1(
            "Render Order is now part of Camera. Scene's Render Order is no longer used.",
        AbstractRenderOrder* GetRenderOrder());

        CANDERA_DEPRECATED_3_1_1(
            "Render Order is now part of Camera. Scene's Render Order is no longer used.",
        const AbstractRenderOrder* GetRenderOrder() const);

        /**
         * Retrieves the number of lights of this scene.
         * @return The number of lights;
         */
        SizeType GetNumberOfLights() const;

        /**
         * Retrieves the first light in the light list of this scene.
         * @return The first light in the light list or 0 if non exist.
         */
        const Light* GetFirstLight();

        /**
         * Retrieves the next light in the light list of this scene.
         * @return The next light in the light list or 0 if non exist.
         */
        const Light* GetNextLight();

        /** Adds a listener to this Scene object.
         * @param listener: The lifetime of the listener is managed by the calling code. Scene does not take ownership.
         * @return true if successful, false otherwise. Null pointers are not accepted as valid SceneListener objects.
         */
        bool AddSceneListener(SceneListener* listener);

        /** Removes a listener from this Scene object.
         * @param listener specifies the SceneListener object to remove from this Scene.
         * @return true if successful, false otherwise. Null pointers are not accepted as valid SceneListener objects.
         */
        bool RemoveSceneListener(SceneListener* listener);

        FEATSTD_RTTI_DECLARATION();

        /**
        * Overrides IsRenderPrerequisiteFulfilled from Node. A Scene object itself does not fulfill render prerequisites.
        * @return Always false because a scene object itself is not rendered.
        */
        virtual bool IsRenderPrerequisiteFulfilled() const { return false; }

#if defined(CANDERA_LAYOUT_ENABLED)
        /**
        *  \brief  Indicates that the scene layout is invalid.
        *
        *  To trigger a scene to be laid out only once per Update/Render loop, all children of the scene (also widgets) can invalidate the scene layout by calling
        *  Scene::SetLayoutInvalid(). Before rendering is triggered by the application, Scene::ValidataLayout() can be called to trigger the layout if needed.
        */
        void SetLayoutInvalid() { SetLayoutValid(false); }

        /**
        *  Triggers a layout update in case the scene layout is invalid. See Scene::SetLayoutInvalid().
        */
        void ValidateLayout();
#endif

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Scene();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Scene);

        /**
        * Disposes the instance of this class.
        */
        virtual void DisposeSelf() override;

        /**
         * Add a Light to the light list of Scene.
         * @param light Light to be added to this scene's light list.
         * @return true if light has been added successfully, false otherwise.
         */
        bool AddLight(Light* light);

        /**
         * Remove Light from light list of Scene.
         * @param light Light to be removed from this scene's light list.
         * @return true if light has been removed successfully, false otherwise.
         */
        bool RemoveLight(Light* light);

        /**
         * Set this Scene's LightList as active lights in RenderDevice
         */
        void Activate();

        /**
          * Informs all SceneListeners that this Scene object has been activated.
          */
        void NotifyListenersOnSceneActivated();

    private:

        /**
         * Notification that the given canvas was added to this scene.
         * @param canvas  The canvas that was added to this scene.
         */
        void OnCanvasAdded(Canvas* canvas);

        /**
         * Notification that the given canvas was removed from this scene.
         * @param canvas  The canvas that was removed from this scene.
         */
        void OnCanvasRemoved(Canvas const* canvas);

        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
        RenderDevice::LightList m_lights;       // Light list associated to this Scene.
        Light* m_lightIterator;                 // Used to iterate over the light list.

#ifdef CANDERA_DEPRECATED_3_1_1
        AbstractRenderOrder* m_renderOrder;     // RenderOrder object associated to this Scene.
#endif

        typedef Internal::Vector<Canvas*> CanvasContainer;
        CanvasContainer m_canvases;

        /**
         * SceneListenerContainer holds all SceneListeners registered, see Scene::AddSceneListener.
         */
        typedef Internal::SingleLinkedList<SceneListener*> SceneListenerContainer;
        SceneListenerContainer m_sceneListeners;
#if defined(CANDERA_LAYOUT_ENABLED)
#ifdef FEATSTD_THREADSAFETY_ENABLED
        mutable FeatStd::Internal::CriticalSection m_lock;
#endif
        bool m_isLayoutValid:1;

        bool GetLayoutValid() const
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            return m_isLayoutValid;
        }

        void SetLayoutValid(bool val)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            m_isLayoutValid = val;
        }
#endif

        CdaDynamicProperties(Candera::Scene, Candera::Group);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_SCENE_H
