//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_RENDERERSTATISTICSOVERLAY_H
#define CANDERA_RENDERERSTATISTICSOVERLAY_H

#include <Candera/Engine3D/Core/Appearance.h>

namespace Candera {
/** @addtogroup Core3D
*  @{
*/

class Camera;
class Group;
namespace Internal { class TextMesh; }
class RenderTarget3D;

/**
 *  @brief  RendererStatisticsOverlay renders statistics data from the Renderer to the given renderTarget using
 *          the provided shader. The shader should support vertex colors to render colored text correctly.
 *          The appearance of the overlay can be retrieved and changed (e.g. blending) to adapt how it appears
 *          on screen.
 *          NOTE: Make sure to call Render(renderTarget) before swapping the renderTarget. One way to do this
 *          is using a RenderTarget::BeforeSwapEvent callback:
 *
 * \code
 *      // Event listener that renders the overlay.
 *      class RenderTargetEventListener : public FeatStd::EventListener {
 *          public:
 *          RenderTargetEventListener(Candera::RendererStatisticsOverlay* renderStatisticsOverlay)
 *              : m_rendererStatisticsOverlay(renderStatisticsOverlay) {}
 *          virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& event)
 *          {
 *              const Candera::RenderTarget::BeforeSwapEvent* beforeSwapEvent = Candera::Dynamic_Cast<const Candera::RenderTarget::BeforeSwapEvent*>(&event);
 *              if (beforeSwapEvent == 0) {
 *              return FeatStd::EventResult::Proceed;
 *              }
 *              Candera::RenderTarget* renderTarget = beforeSwapEvent->GetRenderTarget();
 *              if ((0 != m_rendererStatisticsOverlay) && (0 != renderTarget)) {
 *                  Candera::GraphicDeviceUnit* gdu = renderTarget->GetGraphicDeviceUnit();
 *                  if (0 != gdu) {
 *                      m_rendererStatisticsOverlay->Render(gdu->ToRenderTarget3D());
 *                  }
 *              }
 *          return FeatStd::EventResult::Proceed;
 *          }
 *          private:
 *          Candera::RendererStatisticsOverlay* m_rendererStatisticsOverlay;
 *      };
 *
 *      // Create the event listener
 *      FeatStd::EventListener* listener = CANDERA_NEW(RenderTargetEventListener)(rendererStatisticsOverlay);
 *      // Attach it to the renderTarget that the overlay should be rendered to.
 * \endcode
 */
class RendererStatisticsOverlay
{

public:
    /**
     *  Create a new instance of the renderer statistics overlay.
     *  Note: the shader to render the overlay should support vertex colors
     *        otherwise colored text will not show up correctly.
     *  @param shader  The shader to use for rendering the overlay.
     */
    static RendererStatisticsOverlay* Create(const SharedPointer<Shader>& shader);

    /**
     *  Destructor
     */
    virtual ~RendererStatisticsOverlay();

    /**
     *  Render the overlay to the given render target.
     *  @param renderTarget  The render target to render the overlay to.
     */
    void Render(RenderTarget3D* renderTarget);

    /**
     *  Get the appearance used to render the overlay.
     *  @return The appearance used to render the overlay.
     */
    Appearance::SharedPointer GetAppearance() const { return m_appearance; }

    /**
     *  Set the color of the statistics text.
     *  @param color  The color of the statistics text.
     */
    void SetTextColor(const Color& color) { m_textColor = color; }

    /**
     *  Set the color of the statistics values.
     *  @param color  The color of the statistics values.
     */
    void SetValueColor(const Color& color) { m_valueColor = color; }

protected:
    /**
     *  Update the text using renderer statistics data.
     */
    virtual void UpdateText();

private:
    RendererStatisticsOverlay(const SharedPointer<Shader>& shader);
    Appearance::SharedPointer m_appearance;
    Internal::TextMesh* m_textMesh;
    Group* m_dummyNode;
    Camera* m_camera;
    Color m_textColor;
    Color m_valueColor;
};
/** @} */ // end of Core3D
} // namespace Candera

#endif
