//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RENDERER_H)
#define CANDERA_RENDERER_H

#include <Candera/Environment.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/EngineBase/Common/ListenerContainer.h>
#include <Candera/Engine3D/Core/Renderable.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

// Forward declarations.
class AbstractRenderOrder;
class Camera;
class ClearMode;
class CameraRenderStrategy;
class PixelBuffer;
class RendererListener;
class RenderMode;
class RenderTarget3D;
class ExternalTextureImage;

/**
 * @brief   The class Renderer represents the unique Candera render interface to the application.
 *          Each render call an base of Candera shall be triggered via the Renderer interface.
 */
class Renderer
{
    friend class Camera;
    friend class CameraRenderStrategy;
    friend class ClearMode;
    friend class VertexBuffer;
    friend class Renderer2D3D;

    public:

        /**
         * Set global default RenderMode being used whenever an Appearance has RenderMode set to null.
         * @param renderMode Global default RenderMode to be set.
         */
        static void SetDefaultRenderMode(MemoryManagement::SharedPointer<RenderMode> renderMode);

        /**
         * Retrieve global default RenderMode being used whenever an Appearance has RenderMode set to null.
         * @return The global default RenderMode being used whenever an Appearance has RenderMode set to null.
         */
        static MemoryManagement::SharedPointer<RenderMode> GetDefaultRenderMode();

        /**
         * Renders all cameras
         * Each camera shall fulfill following requirements, in order to render associated scene:
         *  * is child of a scene graph, see interface Scene::AddChild.
         *  * has rendering enabled, see interface Camera::SetRenderingEnabled.
         *  * has an associated viewport defined, see interface Camera::SetAssociatedViewport
         * @return True if rendering of every camera succeeded, false if rendering of any camera failed.
         */
        static bool RenderAllCameras();

        /**
         * Renders the associated scene captured by the camera
         * @param camera:           The camera that captures the screen. If null, no rendering is processed.
         * @param resetStatistics   Reset the accumulated draw statistics data of the renderer.
         * @param dirtyArea         Dirty area in normalized screen coordinates [0..1]. The default values
         *                          of (0,0,1,1) denote the full screen area.
         * @param flushTexturePool  Flush the texture pool that caches unloaded textures for reuse.
         *
         * The camera shall fulfill following requirements, in order to render associated scene:
         *  * is child of a scene graph, see interface Scene::AddChild.
         *  * has rendering enabled, see interface Camera::SetRenderingEnabled.
         *  * has an associated viewport defined, see interface Camera::SetAssociatedViewport
         * @return True if rendering of camera succeeded, false otherwise.
         */
        static bool RenderCamera(Camera* camera, bool resetStatistics = true,
            const Rectangle& dirtyArea = Rectangle(0.0F, 0.0F, 1.0F, 1.0F), bool flushTexturePool = true);

        /**
         * Determine whether the renderer is currently rendering a camera.
         * @return True if RenderCamera is currently called.
         */
        static bool IsRenderingCamera() { return s_isRenderingCamera; }

        /**
         * Retrieves the number of cameras associated with this renderer.
         *
         * @return The number of cameras associated with this renderer.
         */
        static UInt32 GetCameraCount();

        /**
         * Retrieves the camera with the given index. Use GetCameraCount() to
         * obtain the upper bound of valid camera indices.
         *
         * @param index: The index of the sought camera. The index must be in the
         *               range [0..GetCameraCount()[
         * @return The camera at the specified index.
         */
        static Camera* GetCamera(UInt32 index);

         /** Adds a listener. Event/timing of the calls: before/after a node is rendered.
         * @param listener The lifetime of the listener is managed by the calling code. Renderer does not take ownership.
         * @return true if successful, false otherwise
         */
        static bool AddRendererListener(RendererListener* listener);

        /** Removes a listener.
         * @param listener specifies the RendererListener object to remove from this Renderer.
         * @return true if successful, false otherwise
         */
        static bool RemoveRendererListener(RendererListener* listener);

        /**
         * Set a light for rendering at the given index.
         * @param light  The light to set for rendering.
         * @param index  The index in the range [0..CANDERA_MAX_LIGHTS_COUNT-1] to set the light to.
         */
        static void SetLight(const Light* light, SizeType index);

        /**
         * Set uniforms of the active shader when the location of the uniform is known. The uniform data type
         * is passed as template argument.
         * @param location Location of uniform to be set. Can be retrieved from GetUniformLocation.
         * @param data     Data to be set as value of uniform.
         * @param count    Size of data to be set, shall be retrieved from GetActiveUniform.
         *                 Default value is 1, single values (e.g. one Float, one Matrix,...) count always as 1.
         * @return         True if uniform exists and value is set, false otherwise.
         */
        template <Shader::UniformType T>
        static bool SetUniformInRenderDevice(Int location, const void* data, UInt count = 1);

        /**
         * Set uniforms of the shader given.
         * @param programHandle Handle to program whose uniform will be set.
         * @param name          Name of uniform to be set.
         * @param data          Data to be set as value of uniform.
         * @param type          Uniform data type.
         * @param count         Size of data to be set, shall be retrieved from GetActiveUniform in case of arrays.
         *                      Default value is 1, single values (e.g. one Float, one Matrix,...) count always as 1.
         * @return              True if uniform exists and value is set, false otherwise.
         */
        static bool SetUniformInRenderDevice(Handle programHandle, const Char* name, const void* data, Shader::UniformType type, UInt count = 1);

        /**
         * Submit a uniform of the active shader for the given instance to the Renderer.
         * The uniform data type is passed as template argument.
         *
         * @param location      Location of uniform to be set.
         * @param data          Data to be set as value of uniform.
         * @param instanceIndex If this value is < 0, the uniform will be set immediately by the RenderDevice.
         *                      A value >= 0 refers to the index of the instance the uniform will be set for by
         *                      the Renderer before issuing an instanced draw call.
         * @return              True if uniform exists and value is set, false otherwise.
         */
        template <Shader::UniformType T>
        static bool SetUniform(Int location, const void* data, Int instanceIndex);

        /**
         * Submit a uniform of the active shader for the given instance to the Renderer.
         * If the uniform data type can be determined at compile time, the optimized template
         * version of SetUniform should be used.
         * @param location      Location of uniform to be set.
         * @param data          Data to be set as value of uniform.
         * @param type          Uniform data type.
         * @param count         Size of data to be set. If it is > 1, it is a uniform array and will be set
         *                      for instanceIndex 0 only (array of arrays is not supported in ESSL).
         * @param instanceIndex If this value is < 0, the uniform will be set immediately by the RenderDevice.
         *                      A value >= 0 refers to the index of the instance the uniform will be set for by
         *                      the Renderer before issuing an instanced draw call.
         * @return              True if uniform exists and value is set, False otherwise.
         */
        static bool SetUniform(Int location, const void* data, Shader::UniformType type, UInt count, Int instanceIndex);

        /**
         * Return the active uniform block count of the shader program.
         * @param shader  The shader from which to get the active uniform block count for.
         * @param count   The active uniform block count of the shader, out parameter.
         * @return        True if active uniform block count was successfully retrieved.
         */
        static bool GetActiveUniformBlockCount(const Shader& shader, /*out*/ Int& count);

        /**
         * Return the length of the longest active uniform block name of the shader including
         * the null termination character. If no active uniform block exists, 0 is returned.
         * @param shader     The shader to get the maximum block name length from.
         * @param maxLength  A reference to the variable to return the length of longest active
         *                   uniform block name in.
         * @return           True if the operation was successful.
         */
        static bool GetActiveUniformBlockMaxNameLength(const Shader& shader, /*out*/ Int& maxLength);

        /**
         * Return the active uniform block name of the shader corresponding to the given index.
         * @param shader             The shader to get the active uniform block name from.
         * @param uniformBlockIndex  The uniform block index to get the active uniform block name for.
         * @param nameBufferSize     The maximum number of characters (including null terminator) the Renderer
         *                           is allowed to write to uniformBlockName.
         * @param nameLength         A reference to the variable to store the actual number of characters written
         *                           (excluding null terminator) by the Renderer.
         * @param uniformBlockName   A pointer to a buffer of size nameBufferSize to store the name of the active
         *                           uniform block specified by the given uniform block index in.
         * @return  True, if the operation was successful.
         */
        static bool GetActiveUniformBlockName(const Shader& shader, UInt uniformBlockIndex, Int nameBufferSize,
            Int& nameLength, Char* uniformBlockName);

        /**
         * Return the active uniform block size of the shader corresponding to the given index.
         * @param shader             The shader to get the active uniform block size from.
         * @param uniformBlockIndex  The uniform block index to get the active uniform block size for.
         * @param blockSize          A reference to the variable to store the size of the active uniform block.
         * @return  True, if the operation was successful.
         */
        static bool GetActiveUniformBlockSize(const Shader& shader, UInt uniformBlockIndex, /*out*/ Int& blockSize);

        /**
         *  Starts the given query.
         *  @param query  Defines the new render device query to start.
         *  @return True, if the query could be activated in the render device, False otherwise.
         */
        static bool BeginQuery(const Query& query);

        /**
         * Issues a draw call, and updates renderer statistics accordingly
         * @param vertexBuffer   The vertex buffer to be rendered.
         * @param instanceCount  If this value is bigger than one, an instanced draw call will be
         *                       issued, otherwise a regular draw call.
         * @return False if an error occurred, True otherwise.
         */
        static bool Draw(const VertexBuffer& vertexBuffer, UInt instanceCount = 1);

        /**
         *  Structure to keep track of renderer draw and objects statistics.
         */
        struct Statistics {
            // Draw based statistics
            UInt DrawCalls;                         ///< number of draw calls (including instanced) submitted to the RenderDevice
            UInt InstancedDrawCalls;                ///< number of instanced draw calls submitted to the RenderDevice
            UInt Instances;                         ///< number of instances submitted with instanced draw calls
            UInt Vertices;                          ///< number of vertices submitted with draw calls
            UInt Indices;                           ///< number of indices submitted with draw calls
            UInt UniformCalls;                      ///< number of SetUniform/glUniform* calls
            UInt Uniforms;                          ///< number of uniforms set
            UInt Queries;                           ///< number of submitted queries
            UInt DrawCallsVisible;                  ///< number of draw calls deemed visible from occlusion query
            UInt DrawCallsInvisible;                ///< number of draw calls deemed invisible from occlusion query
            UInt ColorBufferClears;                 ///< number of color buffer clears
            UInt DepthBufferClears;                 ///< number of depth buffer clears
            UInt StencilBufferClears;               ///< number of stencil buffer clears
            Float AverageDirtyAreaFactor;           ///< average dirty area factor (dirty area / renderTarget area) in the range of (0..1].
            UInt CulledNodesUsingDirtyAreaScissor;  ///< number of nodes culled due to their bounding box being outside the dirty area.
            UInt UniformBufferUpdates;              ///< number of uniform buffer objects that were updated
            UInt PixelBufferMappings;               ///< number of times a pixel buffer object was mapped.

            // Objects statistics
            UInt UploadedTextures;                  ///< number of currently uploaded textures.
            UInt UploadedTexturesInReusePool;       ///< number of currently uploaded textures in the texture pool for reuse.
            UInt TotalTextureUploads;               ///< total number of texture uploads.
            UInt TotalReusedTextureUploads;         ///< total number of reused texture uploads.
            UInt SamplerObjects;                    ///< number of instantiated sampler objects.
            UInt UniformBufferObjects;              ///< number of instantiated uniform buffer objects.
            UInt PixelBufferObjects;                ///< number of instantiated pixel buffer objects.
            UInt UploadedExternalTextureImages;       ///< Number of currently uploaded direct texture images.
            UInt TotalExternalTextureImageUploads;    ///< Total number of direct texture images uploads.

            Statistics()
                :
                DrawCalls(0),
                InstancedDrawCalls(0),
                Instances(0),
                Vertices(0),
                Indices(0),
                UniformCalls(0),
                Uniforms(0),
                Queries(0),
                DrawCallsVisible(0),
                DrawCallsInvisible(0),
                ColorBufferClears(0),
                DepthBufferClears(0),
                StencilBufferClears(0),
                AverageDirtyAreaFactor(0.0F),
                CulledNodesUsingDirtyAreaScissor(0),
                UniformBufferUpdates(0),
                PixelBufferMappings(0),
                UploadedTextures(0),
                UploadedTexturesInReusePool(0),
                TotalTextureUploads(0),
                TotalReusedTextureUploads(0),
                SamplerObjects(0),
                UniformBufferObjects(0),
                PixelBufferObjects(0),
                m_dirtyAreaCounter(0) {}

            /**
             *  Reset the draw statistics for this frame.
             */
            void Reset() {
                DrawCalls = 0;
                InstancedDrawCalls = 0;
                Instances = 0;
                Vertices = 0;
                Indices = 0;
                UniformCalls = 0;
                Uniforms = 0;
                Queries = 0;
                DrawCallsVisible = 0;
                DrawCallsInvisible = 0;
                ColorBufferClears = 0;
                DepthBufferClears = 0;
                StencilBufferClears = 0;
                AverageDirtyAreaFactor = 0.0F;
                CulledNodesUsingDirtyAreaScissor = 0;
                UniformBufferUpdates = 0;
                PixelBufferMappings = 0;
                m_dirtyAreaCounter = 0;
            }

        private:
            friend class Renderer;
            UInt m_dirtyAreaCounter;
        };

        /**
         * Get render statistics (number of draw calls submitted, number of vertices processed, etc.) since the last
         * RenderCamera() or RenderAllCameras() call.
         * @return  A reference to the Renderer::Statistics that were collected since the last RenderCamera()
         *          or RenderAllCameras() call.
         */
        static const Statistics& GetStatistics() { return s_statistics; }

        /**
         * Upload a Texture given to VRAM.
         * Handle to texture memory is generated during upload and will be set to the given texture.
         * Postcondition on success: Texture is active.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture to be activated and associated to textureImage.
         * @param  loadingHint  Indicates how uploading should be handled.
         * @return              True if uploading and activating texture succeeded. False otherwise, also fails if
         *                      bitmap of texture image is not valid.
         */
        static bool UploadBitmapTextureImage(BitmapTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint);

        /**
         * Upload a CubeMap texture given to VRAM.
         * Handle to texture memory is generated during upload and will be set to the given texture.
         * Postcondition on success: Texture is active.
         * If mipmapping is enabled, custom bitmap chains will only be uploaded if every bitmap has one such chain attached.
         * Otherwise the mipmaps are generated.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture to be activated and associated to textureImage.
         * @param  loadingHint  Indicates how uploading should be handled.
         * @return              True if uploading all six cube map faces and activating the texture succeeded. False otherwise, also fails if
         *                      any bitmap of any face is not valid.
         */
        static bool UploadCubeMapTextureImage(CubeMapTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint);

        /**
         * Unload a Texture from graphics device memory (VRAM).
         * @param  textureImage  Texture image to unload from VRAM.
         * @param  loadingHint   Indicates how unloading should be handled.
         * @return               True if texture image was unloaded.
         */
        static bool UnloadBitmapTextureImage(BitmapTextureImage& textureImage, DeviceObject::LoadingHint loadingHint);

        /**
         * Flush the texture pool and thereby destroy any textures that have been previously unloaded,
         * but cached for reuse.
         * @param contextIndex  The index of the context resource pool to flush its associated textures.
         * @return              True if the operation was successful. False, if the destruction of a texture failed.
         */
        static bool FlushTexturePool(SizeType contextIndex);

        /**
         * Flush the texture pools from all contexts and thereby destroy any textures that have been
         * previously unloaded, but cached for reuse.
         * @return  True if the operation was successful. False, if the destruction of a texture failed.
         */
        static bool FlushTexturePools();

        /**
          * Unload a CubeMap Texture from graphics device memory (VRAM).
          * @param  textureImage  Texture image to unload from VRAM.
          * @param  loadingHint   Indicates how unloading should be handled.
          * @return               True if texture image was unloaded.
          */
        static bool UnloadCubeMapTextureImage(CubeMapTextureImage& textureImage, DeviceObject::LoadingHint loadingHint);

        /**
         * Upload VertexBuffer to graphics device memory (VRAM).
         * @param  vertexBuffer  VertexBuffer object containing vertex geometry and index buffer to upload.
         * @param  loadingHint   Indicates how uploading should be handled.
         * @return               True if vertex geometry and index buffer in vertex buffer are valid
         *                       and could be uploaded to VRAM.
         */
        static bool UploadVertexBuffer(VertexBuffer& vertexBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Unload VertexBuffer to graphics device memory (VRAM).
         * @param  vertexBuffer  VertexBuffer object containing memory handles to VRAM memory to delete.
         * @param  loadingHint   Indicates how unloading should be handled.
         * @return               True if VertexBuffer data are deleted from VRAM.
         */
        static bool UnloadVertexBuffer(VertexBuffer& vertexBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Upload Shader to graphics device memory (VRAM).
         * @param  shader       Shader object containing vertex and fragment shader or shader program to upload.
         * @param  loadingHint  Indicates how uploading should be handled.
         * @return              True if shader objects are valid and could be uploaded to VRAM.
         */
        static bool UploadShader(Shader& shader, DeviceObject::LoadingHint loadingHint);

        /**
         * Delete shader in graphic device memory (VRAM).
         * @param  vertexShaderHandle    Handle to vertex shader to be deleted.
         * @param  fragmentShaderHandle  Handle to fragment shader to be deleted.
         * @param  programHandle         Handle to shader program to be deleted.
         * @param  shaderName
         * @param  loadingHint           Indicates how unloading should be handled.
         * @return                       True if shaders were deleted.
         */
        static bool DeleteShader(Handle vertexShaderHandle, Handle fragmentShaderHandle, Handle programHandle, const Char* shaderName, DeviceObject::LoadingHint loadingHint);

        /**
         * Activate the shader for rendering.
         * @param shader  The shader to activate.
         * @param program
         * @return        True if the shader was successfully activated.
         */
        static bool ActivateShader(const Shader& shader, Handle program);

        /**
         * Upload RenderBuffer to graphics device memory (VRAM).
         * The handle to the RenderBuffer in VRAM is set to the specified RenderBuffer object
         * @param  renderBuffer  Specifies the RenderBuffer object to upload.
         * @param  loadingHint   Indicates how uploading should be handled.
         * @return               True if uploading render buffer succeeded.
         */
        static bool UploadRenderBuffer(RenderBuffer& renderBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Unload RenderBuffer from graphics device memory (VRAM).
         * @param  renderBuffer  Specifies the RenderBuffer object to unload.
         * @param  loadingHint   Indicates how unloading should be handled.
         * @return               True if renderBuffer was deleted in VRAM, false if renderBuffer doesn't have any video memory handle.
         */
        static bool UnloadRenderBuffer(RenderBuffer& renderBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Upload a Texture given to VRAM.
         * Handle of texture is generated during upload and will be set
         * to the given texture.
         * Postcondition on success: Texture is active.
         * @param  textureImage The texture image to upload to VRAM.
         * @param  unit         Texture unit to which the texture is bound.
         * @param  loadingHint  Indicates how uploading should be handled.
         * @return              True if uploading and activating texture
         *                      succeeded. False otherwise.
         */
        static bool UploadDirectTextureImage(DirectTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint);

        /**
         * Unload a Texture from graphics device memory (VRAM).
         * @param  textureImage  Texture image to unload from VRAM.
         * @param  loadingHint   Indicates how unloading should be handled.
         * @return                  True if texture image was unloaded.
         */
        static bool UnloadDirectTextureImage(DirectTextureImage& textureImage, DeviceObject::LoadingHint loadingHint);

        /**
        * Prepare(map) an external image for the use as texture.
        * Handle of texture is generated during upload and will be set
        * to the given texture.
        * Postcondition on success: Texture is active.
        * @param  textureImage The texture image to prepare.
        * @param  unit         Texture unit to which the texture is bound.
        * @param  loadingHint  Indicates how uploading should be handled.
        * @return              True if uploading and activating texture
        *                      succeeded. False otherwise.
        */
        static bool UploadExternalTextureImage(ExternalTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint);

        /**
        * Unmap an external image from a texture.
        * @param  textureImage  Texture image to unmap.
        * @param  loadingHint   Indicates how unloading should be handled.
        * @return                  True if texture image was unloaded.
        */
        static bool UnloadExternalTextureImage(ExternalTextureImage& textureImage, DeviceObject::LoadingHint loadingHint);


        /**
         * Upload a uniform buffer to VRAM.
         * @param  uniformBuffer  The uniform buffer to upload.
         * @param  loadingHint    Indicates how uploading should be handled.
         * @return                True if the data was successfully uploaded to VRAM.
         */
        static bool UploadUniformBuffer(UniformBuffer& uniformBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Update the data of a uniform buffer in VRAM.
         * @param  uniformBuffer  The uniform buffer to update.
         * @return                True if the data was successfully updated in VRAM.
         */
        static bool UpdateUniformBuffer(UniformBuffer& uniformBuffer);

        /**
         * Unload a device buffer from VRAM.
         * @param  uniformBuffer  The uniform buffer to unload.
         * @param  loadingHint    Indicates how unloading should be handled.
         * @return                True if the buffer was successfully unloaded from VRAM.
         */
        static bool UnloadUniformBuffer(UniformBuffer& uniformBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Bind a uniform buffer to the uniform block of a shader.
         * @param  uniformBuffer      The uniform buffer to bind to the shader.
         * @param  shader             The shader to bind the uniform buffer to.
         * @param  uniformBlockIndex  The uniform block index from the shader to bind the uniform buffer to.
         * @return                    True if the uniform buffer was successfully bound to the uniform block
         *                            of the shader.
         */
        static bool BindUniformBufferToShader(const UniformBuffer& uniformBuffer, const Shader& shader, Int uniformBlockIndex);

        /**
         * Return if uniform buffers are supported.
         * @return  True if uniform buffers are supported.
         */
        static bool IsUniformBufferSupported();

        /**
         * Upload a pixel buffer to the GPU.
         * @param  pixelBuffer  The pixel buffer to upload.
         * @param  loadingHint  Indicates how uploading should be handled.
         * @return              True if the pixel buffer was successfully uploaded.
         */
        static bool UploadPixelBuffer(PixelBuffer& pixelBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Unload a pixel buffer from the GPU.
         * @param  pixelBuffer  The pixel buffer to unload.
         * @param  loadingHint  Indicates how unloading should be handled.
         * @return              True if the pixel buffer was successfully unloaded.
         */
        static bool UnloadPixelBuffer(PixelBuffer& pixelBuffer, DeviceObject::LoadingHint loadingHint);

        /**
         * Bind a pixel buffer.
         * @param pixelBuffer  The pixel buffer to bind.
         * @return             True if the pixel buffer was successfully bound.
         */
        static bool BindPixelBuffer(const PixelBuffer& pixelBuffer);

        /**
         * Unbind a pixel buffer.
         * @param pixelBuffer  The pixel buffer to unbind.
         * @return             True if the pixel buffer was successfully unbound.
         */
        static bool UnbindPixelBuffer(const PixelBuffer& pixelBuffer);

        /**
         * Map the currently bound pixel buffer.
         * @param pixelBuffer  The pixel buffer to map.
         * @return  The address of the mapped pixel buffer.
         */
        static void* MapPixelBuffer(const PixelBuffer& pixelBuffer);

        /**
         * Unmap the currently bound pixel buffer.
         * @param pixelBuffer  The pixel buffer to unmap.
         * @return             True if the pixel buffer was successfully unmapped.
         */
        static bool UnmapPixelBuffer(const PixelBuffer& pixelBuffer);

        /**
         * Return if pixel buffers are supported.
         * @return  True if pixel buffers are supported.
         */
        static bool IsPixelBufferSupported();

        /**
         * Bind a vertex buffer as a transform feedback  buffer.
         * @param vertexBuffer  The vertex buffer to bind as a transform feedback buffer.
         * @return              True if the vertex buffer was successfully bound as a transform feedback buffer.
         */
        static bool BindTransformFeedbackBuffer(const VertexBuffer& vertexBuffer);

        /**
         * Unbind a transform feedback buffer.
         * @param vertexBuffer  The transform feedback buffer to unbind.
         * @return             True if the transform feedback buffer was successfully unbound.
         */
        static bool UnbindTransformFeedbackBuffer(const VertexBuffer& vertexBuffer);

        /**
         * Return if transform feedback buffers are supported.
         * @return  True if transform feedback buffers are supported.
         */
        static bool IsTransformFeedbackBufferSupported();

        /**
         * Begin the transform feedback operation.
         * @param vertexBuffer  The vertex buffer determining the primitive type used in the operation.
         * @return              True if starting the transform feedback operation was successful.
         */
        static bool BeginTransformFeedback(const VertexBuffer& vertexBuffer);

        /**
         * End the transform feedback operation.
         * @return  True if ending the transform feedback operation was successful.
         */
        static bool EndTransformFeedback();

        /**
         * Map the currently bound transform feedback buffer.
         * @param vertexBuffer  The transform feedback buffer (i.e. a vertex buffer that was bound for
         *                      transform feedback) to map.
         * @return              The address of the mapped transform feedback buffer.
         */
        static void* MapTransformFeedbackBuffer(const VertexBuffer& vertexBuffer);

        /**
         * Unmap the currently bound transform feedback buffer.
         * @param vertexBuffer The transform feedback buffer to unmap.
         * @return             True if the transform feedback buffer was successfully unmapped.
         */
        static bool UnmapTransformFeedbackBuffer(const VertexBuffer& vertexBuffer);

        /**
         *  Activates the given Camera and its ClearMode states in RenderDevice.
         *  @param camera  The Camera to deactivate.
         *  @return        False, if the Camera or the Camera's RenderTarget is 0, or if any activation in the RenderDevice failed.
         *                 True, otherwise.
         */
        static bool ActivateCamera(const Camera* camera);

        /**
         *  Deactivates the given Camera.
         *  @param camera  The Camera to deactivate.
         *  @return        False, if the Camera or the Camera's RenderTarget is 0. True, otherwise.
         */
        static bool DeactivateCamera(const Camera* camera);

    private:
        static const Shader* s_activeShader;

        static Statistics s_statistics;

        static Rectangle s_dirtyArea;

        static bool s_isRenderingCamera;

        typedef Candera::Internal::ListenerContainer<RendererListener> RendererListenerContainer;
        static RendererListenerContainer s_rendererListeners;

        class OnNodePreRenderEvent;
        class OnNodePostRenderEvent;
        class RendererCameraListener;
        class OnCameraPostRenderEvent;
        class OnCameraPreActivateEvent;
        class OnCameraPostActivateEvent;
        class OnCameraPreClearEvent;
        class OnCameraPostClearEvent;


        static void OnCameraRenderTargetChanged(const Camera* camera, RenderTarget3D* previousRenderTarget);
        static void OnCameraRenderStrategyChanged(const Camera* camera, const CameraRenderStrategy* previousCameraRenderStrategy);

        static bool RenderCameraInternal(Camera* camera, const Rectangle& dirtyArea, bool flushTexturePool);

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Renderer::Renderer, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        Renderer();
        virtual ~Renderer() = 0; // Provoke abstract (static) class

        /**
         * Add Camera to camera list of Renderer.
         * @param camera Camera to be added.
         * @return False if camera is 0, otherwise true (in this case the camera is added).
         */
        static bool AddCamera(Camera* camera);
        /**
         * Remove Camera from camera list of Renderer.
         * @param Camera to be removed from the renderer's camera list.
         * @return False if camera is 0 or not in list, true otherwise (then it's removed).
         */
        static bool RemoveCamera(const Camera* camera);
         /**
          * Called before a node is rendered. Notifies the RendererListeners.
          * @param node Notifies renderer listeners that node will be rendered next.
          */
        static void NotifyRendererListenersOnNodePreRender(Node *node);
        /**
         * Called after a node is rendered. Notifies the RendererListeners.
         * @param node Notifies renderer listeners that node was rendered last.
         */
        static void NotifyRendererListenersOnNodePostRender(Node *node);

        /**
         * Render all nodes of the renderOrder.
         * @param renderOrder            Contains the nodes to be rendered.
         * @param useImmediateRendering  If True, nodes will immediately render themselves and notify
         *                               renderer listeners via Pre/Post callbacks.
         *                               If False, the renderer will try to accumulate nodes and issue
         *                               instanced draw calls instead of individual draw calls whenever
         *                               possible. In the instanced draw call case, renderer listeners
         *                               will not be notified.
         * @return False if an error occurred, True otherwise.
         */
        static bool Render(AbstractRenderOrder* renderOrder, bool useImmediateRendering = true);

        /**
         * Determine the maximum instance count for the given node.
         * @param node  The node to determine the maximum instance count for.
         * @return The maximum number of instances this node can be used with.
         */
        static UInt DetermineMaximumInstanceCount(const Node* node);

        /**
         * Checks if the given renderable can be batched together with the given node in a single draw call.
         * @param renderableA  The renderable node that the second node will be compared to.
         *                     ATTENTION: this renderable must have a determined maximum instance count
         *                                bigger than one, meaning DetermineMaximumInstanceCount() must be
         *                                called first to verify it before using a renderable with this function.
         * @param nodeB        The node to be compared against the first argument.
         * @result True, if the two arguments can be batched. False, otherwise.
         */
        static bool IsBatchable(const Renderable& renderableA, const Node* nodeB);

        /**
         * Activates the appearance, vertex buffer, and binds vertex attributes to the shader of the appearance.
         * @param renderable  The renderable node to activate.
         * @return False if an error occurred, True otherwise.
         */
        static bool Activate(Renderable& renderable);

        /**
         * Activate the shader parameters of the given renderable for the given instance.
         * @param renderable     The renderable node to set the shader parameters from.
         * @param instanceIndex  The index of the instance to set the shader parameters for.
         * @return False if an error occurred, True otherwise.
         */
        static bool ActivateShaderParams(Renderable& renderable, Int instanceIndex);

        /**
         *  Set the given Camera as the active Camera used for subsequent render calls.
         *  @param camera  Camera to be set active for subsequent render calls.
         */
        static void SetActiveCamera(const Camera* camera) { RenderDevice::SetActiveCamera(camera); }

        /**
         *  Retrieve camera currently active.
         *  @return  The currently active camera.
         */
        static const Camera* GetActiveCamera() { return RenderDevice::GetActiveCamera(); }

        /**
         *  Activates the ClearMode in the RenderDevice.
         *  @param clearMode  The ClearMode to activate in the RenderDevice.
         *  @param camera     Camera containing needed settings.
         *  @return           True, if the operation was successful. False, otherwise.
         */
        static bool ActivateClearMode(const Camera& camera);

        /**
         *  Activates the ClearMode in the RenderDevice.
         *  @param clearMode  The ClearMode to activate in the RenderDevice.
         *  @return           True, if the operation was successful. False, otherwise.
         */
        static bool ActivateClearMode(const ClearMode* clearMode);

        /**
         *  Set normalized scissor rectangle in render device.
         *  @param rectangle     Is the normalized scissor rectangle, where x:0.0, y:0.0 is left, upper and x:1.0, y:1.0 is right, bottom corner of render target.
         *                       Thus, a rectangle with left:0.5, top:0.0, width: 0.5, height 0.5 addresses the right upper quarter of the render target.
         *  @param renderTarget  Defines the render target the normalized scissor rectangle is related to.
         *  @return              True if setting scissor rectangle succeeded.
         */
        static bool ActivateScissor(const Rectangle& rectangle, const RenderTarget3D* renderTarget);

        /**
         * Functions to manage bins for accumulating the uniforms for instances by basic types (Float, Int)
         * and their size (e.g. FloatMat3 has a type of Float and size of 9)
         */
        static bool ResetUniformBins(SizeType expectedSize);
        static bool SubmitUniformBins();

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        struct UniformBinElement {
            Int uniformLocation;
            UInt indexOfNextElementWithSameLocation : 31;
            UInt startsElementsSequence : 1;
            ElementType elements[ElementCount];
        };

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        static Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& GetUniformBin();

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        static bool ResetUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin, SizeType expectedSize);

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        static bool AutoCorrectUniformBinSize(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin);

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        static bool ResizeUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin, SizeType size);

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        static bool AddToUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin, Int location, const void* data);

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        static SizeType& GetUniformBinSize();

        template <Shader::UniformType T, typename ElementType, Int ElementCount>
        static bool SubmitUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin);

        /**
         * Functions to manage the bin used for batching uniforms with the same uniform location
         */
        static bool ResetBatchedBin(SizeType expectedSize);

        static SizeType s_expectedBatchSize;

       /**
        *  Is invoked after a camera is completely rendered.
        *  @param camera The camera that was rendered.
        */
        static void NotifyListenersOnCameraPostRender(Camera* camera);

       /**
        *  Called before a camera is activated.
        *  @param camera The camera that will be activated.
        */
        static void NotifyListenersOnCameraPreActivate(const Camera* camera);

       /**
        *  Is invoked after a camera is activated.
        *  @param camera The camera that was activated.
        */
        static void NotifyListenersOnCameraPostActivate(const Camera* camera);

       /**
        *  Is invoked before clearing the framebuffer.
        *  @param camera The camera that will be activated.
        */
        static void NotifyListenersOnCameraPreClear(const Camera* camera);

       /**
        *  Is invoked after a clearing the framebuffer.
        *  @param camera The camera that was activated.
        */
        static void NotifyListenersOnCameraPostClear(const Camera* camera);

        /**
         *  Called by ContextProvider shortly before context is destroyed. Performs cleanup by unloading
         *  internal objects and releasing resources.
         *  @param contextResourcePool The ContextResourcePool to be destroyed.
         */
        static void OnContextUnload(const ContextResourcePool& contextResourcePool);
        friend class ContextProvider3D;

        struct TexturePoolEntry {
            UInt64 m_propertyKey;
            Handle m_handle;
            UInt m_size;    // used for VideoMemoryStatistic
        };

        static Candera::Internal::Vector<TexturePoolEntry> s_texturePool[CANDERA_MAX_CONTEXT_COUNT];
};

template<> bool Renderer::SetUniform<Shader::Float>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::FloatVec2>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::FloatVec3>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::FloatVec4>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::Integer>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::IntegerVec2>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::IntegerVec3>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::IntegerVec4>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::Bool>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::BoolVec2>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::BoolVec3>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::BoolVec4>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::FloatMat2>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::FloatMat3>(Int location, const void* data, Int instanceIndex);
template<> bool Renderer::SetUniform<Shader::FloatMat4>(Int location, const void* data, Int instanceIndex);

template <Shader::UniformType T>
inline bool Renderer::SetUniformInRenderDevice(Int location, const void* data, UInt count)
{
    ++s_statistics.UniformCalls;
    s_statistics.Uniforms += count;
    return RenderDevice::SetUniform<T>(location, data, count);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
inline bool Renderer::SetUniformInRenderDevice(Handle programHandle, const Char* name, const void* data, Shader::UniformType type, UInt count)
{
    ++s_statistics.UniformCalls;
    s_statistics.Uniforms += count;
    return RenderDevice::SetUniform(programHandle, name, data, type, count);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_RENDERER_H
