//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Renderer.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/CameraListener.h>
#include <Candera/Engine3D/Core/CameraRenderStrategy.h>
#ifdef CANDERA_2D_OVER_3D_ENABLED
#include <Candera/EngineBase/Common/GlyphAtlas.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/SurfaceAllocator2DOver3D.h>
#endif
#include <Candera/Engine3D/Core/GlyphAtlas3D.h>
#include <Candera/Engine3D/Core/PixelBuffer.h>
#include <Candera/Engine3D/Core/Query.h>
#include <Candera/Engine3D/Core/RenderBuffer.h>
#include <Candera/Engine3D/Core/RendererListener.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Diagnostics/VideoMemoryStatistic.h>
#include <Candera/System/Monitor/MonitorPublicIF.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/System/Container/Vector.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#include <CanderaPlatform/Device/Common/Base/DirectTextureImage.h>
#include <CanderaPlatform/Device/Common/Base/ExternalTextureImage.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {

#define CANDERA_DISABLE_TEXTURE_POOL

    FEATSTD_LINT_CURRENT_SCOPE(1576, "Violates MISRA C++ 2008 Required Rule 14-7-3: specialization of function templates moved to seperate files on purpose")

    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    const Shader* Renderer::s_activeShader = 0;
    Renderer::Statistics Renderer::s_statistics;
    Rectangle Renderer::s_dirtyArea = Rectangle(0.0F, 0.0F, 1.0F, 1.0F);
    bool Renderer::s_isRenderingCamera = false;
    SizeType Renderer::s_expectedBatchSize = 0;
    Candera::Internal::Vector<Renderer::TexturePoolEntry> Renderer::s_texturePool[CANDERA_MAX_CONTEXT_COUNT];

    void Renderer::OnCameraRenderTargetChanged(const Camera* camera, RenderTarget3D* previousRenderTarget)
    {
        if ((camera->GetCameraRenderStrategy() != 0) && (!camera->GetCameraRenderStrategy()->IsRenderPassCompleted())) {
            FEATSTD_LOG_WARN("RenderTarget3D of Camera changed while the render pass is not completed. Incomplete frame might be rendered.");
            if ((previousRenderTarget != 0) && (previousRenderTarget->m_activeCameraCount != 0)) {
                --previousRenderTarget->m_activeCameraCount;
                if (previousRenderTarget->m_activeCameraCount == 0) {
                    previousRenderTarget->m_currentRenderPass = RenderTarget3D::BeforeFirstRenderPass;
                }
            }
            camera->GetCameraRenderStrategy()->SetRenderPassCompleted(true);
        }
    }

    void Renderer::OnCameraRenderStrategyChanged(const Camera* camera, const CameraRenderStrategy* previousCameraRenderStrategy)
    {
        RenderTarget3D* renderTarget = camera->GetRenderTarget();
        bool isCameraActive = (renderTarget != 0) &&
            (renderTarget->m_activeCameraCount != 0) &&
            (!previousCameraRenderStrategy->IsRenderPassCompleted());

        if (isCameraActive) {
            FEATSTD_LOG_WARN("RenderStrategy of Camera changed while the render pass is not completed. Incomplete frame might be rendered.");
            --renderTarget->m_activeCameraCount;
            if (renderTarget->m_activeCameraCount == 0) {
                renderTarget->m_currentRenderPass = RenderTarget3D::BeforeFirstRenderPass;
            }
        }

        if (camera->GetCameraRenderStrategy() != 0) {
            camera->GetCameraRenderStrategy()->SetRenderPassCompleted(true);
        }
    }

    class Renderer::RendererCameraListener: public CameraListener {
    public:
        static RendererCameraListener* GetInstance()
        {
            static RendererCameraListener instance;
            return &instance;
        }
        virtual void OnPreRender(Camera* /*camera*/) override {}
        virtual void OnPostRender(Camera* /*camera*/) override {}

        virtual void OnRenderTargetChanged(Camera* camera, RenderTarget3D* previousRenderTarget) override
        {
            Renderer::OnCameraRenderTargetChanged(camera, previousRenderTarget);
        }

        virtual void OnCameraRenderStrategyChanged(Camera* camera, CameraRenderStrategy* previousCameraRenderStrategy) override
        {
            Renderer::OnCameraRenderStrategyChanged(camera, previousCameraRenderStrategy);
        }
    };

Renderer::RendererListenerContainer Renderer::s_rendererListeners;

/**
 * CameraLessBySequenceNumber is a function object to compare Cameras by sequence number.
 */
class CameraLessBySequenceNumber
{
public:
    bool operator()(const Camera* a, const Camera* b) const
    {
        return a->GetSequenceNumber() < b->GetSequenceNumber();
    }
};

//Private function to access default RenderMode
static SharedPointer<RenderMode>& DefaultRenderMode()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(SharedPointer<RenderMode>, defaultRenderMode, RenderMode::Create());
    return defaultRenderMode;
}

static SharedPointer<RenderMode>& s_forceInitDefaultRenderMode = DefaultRenderMode();

void Renderer::SetDefaultRenderMode(SharedPointer<RenderMode> renderMode)
{
    DefaultRenderMode() = renderMode;
}

SharedPointer<RenderMode> Renderer::GetDefaultRenderMode()
{
    return DefaultRenderMode();
}

/**
 *  CameraContainer holds all Camera objects that have been added to a Scene object.
 *  If a Camera is added to a Scene, then it adds itself to the CameraList.
 *  The class Renderer sequentially renders all Camera objects in the CameraContainer.
 */
typedef Candera::Internal::Vector<Camera*> CameraContainer;

CameraContainer& GetCameraContainer()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(CameraContainer, s_container);
    return s_container;
}

FeatStd::Internal::CriticalSection* GetCameraContainerCriticalSection()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_containerCriticalSection);
    return &s_containerCriticalSection;
}

static CameraContainer& s_forceInitCameraContainer = GetCameraContainer();

bool Renderer::AddCamera(Camera* camera)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    //Lock the CameraContainer CriticalSection to avoid starting a render pass while adding the camera to the list.
    FeatStd::Internal::CriticalSectionLocker lock(GetCameraContainerCriticalSection());
#endif

    CameraContainer& cameras = GetCameraContainer();
    bool result = true;
    if (camera != 0) {
        result = cameras.Add(camera);
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1502, objSort, CameraLessBySequenceNumber only defines the comparison operator for sorting the list)
        CameraLessBySequenceNumber objSort;
        cameras.Sort(objSort);
        static_cast<void>(camera->AddCameraListener(RendererCameraListener::GetInstance()));
    }
    else {
        result = false;
    }

    return result;
}

bool Renderer::RemoveCamera(const Camera* camera)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    //Lock the CameraContainer CriticalSection to avoid starting a render pass while removing the camera from the list.
    FeatStd::Internal::CriticalSectionLocker lock(GetCameraContainerCriticalSection());
#endif

    CameraContainer& cameras = GetCameraContainer();
    bool result = false;
    if (camera != 0) {
        Int i = 0;
        for (CameraContainer::Iterator it = cameras.Begin(); it != cameras.End(); ++it) {
            if ( *it == camera ) {
                static_cast<void>((*it)->RemoveCameraListener(RendererCameraListener::GetInstance()));
                // i is a valid index.
                static_cast<void>(cameras.Remove(i));
                if (0 == cameras.Size()) {
                    cameras.Free();
                }
                result = true;
                break;
            }

            ++i;
        }
    }
    return result;
}

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::RenderAllCameras()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    //Lock the CameraContainer CriticalSection to avoid any change in the Camera list while iterating over it and rendering.
    FeatStd::Internal::CriticalSectionLocker lock(GetCameraContainerCriticalSection());
#endif

#ifdef FEATSTD_MONITOR_ENABLED
    Candera::Monitor::RTMonitor::HandlePaused(3);
    CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::Frame_3D));
#endif

    s_statistics.Reset();

    CameraContainer& cameras = GetCameraContainer();
    bool result = true;
    const SizeType cameraListSize = cameras.Size();

    //Before the first render pass, count number of cameras that will render on
    // each RenderTarget.
    for (SizeType i = 0; i < cameraListSize; i++) {
        RenderTarget3D* renderTarget = cameras[i]->GetRenderTarget();
        if ((renderTarget != 0) &&
            (renderTarget->m_currentRenderPass == RenderTarget3D::BeforeFirstRenderPass)) {
            ++renderTarget->m_activeCameraCount;
        }
    }

    //Render each camera in camera list.
    for (SizeType i = 0; i < cameraListSize; i++) {
        Camera* camera = cameras[i];
        RenderTarget3D* renderTarget = camera->GetRenderTarget();
        CameraRenderStrategy* cameraRenderStrategy = camera->GetCameraRenderStrategy();

        //Skip camera render if:
        bool skipCamera =
            // - it has no RenderTarget to render on OR
            (renderTarget == 0) ||
            //  - this is not the first render pass on the RenderTarget AND
            ((renderTarget->m_currentRenderPass >= RenderTarget3D::AfterFirstRenderPass) &&
            //   - it does not support multi pass rendering OR
             ((cameraRenderStrategy == 0) ||
             //   - it supports multi pass rendering, but the rendering is complete.
              cameraRenderStrategy->IsRenderPassCompleted()));
        if (!skipCamera) {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, renderTarget, CANDERA_LINT_REASON_OPERANDNOTNULL);
            //If nothing else has been rendered on the RenderTarget:
            if (renderTarget->m_currentRenderPass == RenderTarget3D::BeforeFirstRenderPass) {
                //Clear RenderTarget if
                bool clearRenderTarget =
                    //automatic clearing of RenderTarget is enabled AND
                    renderTarget->IsAutoClearEnabled() &&
                    //a ClearMode is set on the RenderTarget.
                    (renderTarget->GetSharedClearMode() != 0) &&
                    (renderTarget->GetSharedClearMode()->GetClearMode() != 0);
                if (clearRenderTarget) {
                    renderTarget->Sync();
                    renderTarget->SafeBeginDraw(renderTarget);
                    //Disable scissoring for RenderTarget clearing.
                    result = RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 0);
                    if (!Renderer::ActivateClearMode(renderTarget->GetSharedClearMode()->GetClearMode())) {
                        FEATSTD_LOG_WARN("ClearMode activation failed.");
                    }
                    renderTarget->SafeEndDraw();

                }
                //Mark the first pass on the RenderTarget as started, no further clearing needed.
                renderTarget->m_currentRenderPass = RenderTarget3D::DuringFirstRenderPass;
            }
            //Do the actual rendering.
            result = RenderCamera(camera, /* resetStatistics */ false, Rectangle(0.0F, 0.0F, 1.0F, 1.0F),
                /* flushTexturePool */ false) && result;
            //If render pass is completed:
            if ((cameraRenderStrategy == 0) || cameraRenderStrategy->IsRenderPassCompleted()) {
                //decrease number of cameras that need to further render on the RenderTarget.
                --renderTarget->m_activeCameraCount;
            }
        }

        //RenderTarget should swap if no more cameras need to render.
        bool shouldSwap = (renderTarget != 0) &&
            (renderTarget->m_activeCameraCount == 0) &&
            (renderTarget->m_currentRenderPass != RenderTarget3D::AfterLastRenderPass);

        if (shouldSwap) {
            //if RenderTarget should be swapped automatically:
            if (renderTarget->IsAutoSwapEnabled()) {
                //swap the RenderTarget.
                renderTarget->SwapBuffers();
            }
            //Mark the RenderTarget as fully rendered.
            renderTarget->m_currentRenderPass = RenderTarget3D::AfterLastRenderPass;
        }
    }

    //After the full render pass, mark the RenderTargets that are not fully rendered.
    for (SizeType i = 0; i < cameraListSize; i++) {
        RenderTarget3D* renderTarget = cameras[i]->GetRenderTarget();
        if (renderTarget != 0) {
            if (renderTarget->m_currentRenderPass == RenderTarget3D::DuringFirstRenderPass) {
                renderTarget->m_currentRenderPass = RenderTarget3D::AfterFirstRenderPass;
            }
            if (renderTarget->m_currentRenderPass == RenderTarget3D::AfterLastRenderPass) {
                renderTarget->m_currentRenderPass = RenderTarget3D::BeforeFirstRenderPass;
            }
        }
    }

    result = FlushTexturePools() && result;
    return result;
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

bool Renderer::RenderCamera(Camera* camera, bool resetStatistics, const Rectangle& dirtyArea, bool flushTexturePool)
{
    if (resetStatistics) {
        s_statistics.Reset();
    }

    s_isRenderingCamera = true;
    bool result = RenderCameraInternal(camera, dirtyArea, flushTexturePool);
    s_isRenderingCamera = false;
    return result;
}

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::RenderCameraInternal(Camera* camera, const Rectangle& dirtyArea, bool flushTexturePool)
{
    MONITOR_HANDLE_RUNNING();

    // Render only if Camera has set rendering enabled.
    if ((camera != 0) && camera->IsEffectiveRenderingEnabled()) {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderCamera, camera->GetName())); // don't record, if not rendered

        CameraRenderStrategy* const cameraRenderStrategy = camera->GetCameraRenderStrategy();
        const bool isCameraRenderStrategyAvailable = cameraRenderStrategy != 0;

        Scene* const scene = camera->GetScene();

        if (scene == 0) {
            FEATSTD_LOG_ERROR("Render camera failed, invalid scene.");
            return false;
        }

        RenderTarget3D* renderTarget = camera->GetRenderTarget();
        if (renderTarget == 0) {
            FEATSTD_LOG_ERROR("Render camera failed, invalid RenderTarget.");
            return false;
        }

        s_dirtyArea = dirtyArea;
        if (camera->IsScissoringEnabled()) {
            s_dirtyArea.Intersect(camera->GetScissorRectangle());
        }

        Float dirtyAreaFactor = (s_dirtyArea.GetWidth() * s_dirtyArea.GetHeight());
        s_statistics.AverageDirtyAreaFactor *= static_cast<Float>(s_statistics.m_dirtyAreaCounter);
        s_statistics.AverageDirtyAreaFactor += dirtyAreaFactor;
        if (0 != ++s_statistics.m_dirtyAreaCounter) {
            s_statistics.AverageDirtyAreaFactor /= static_cast<Float>(s_statistics.m_dirtyAreaCounter);
        }
        else {
            s_statistics.AverageDirtyAreaFactor = 0.0F; // Overflow happened, reset the value.
        }

        // Activate current camera.
        camera->SetClearingEnabled(isCameraRenderStrategyAvailable ? cameraRenderStrategy->IsRenderPassCompleted() : true);
        if (!camera->Activate()) {
            FEATSTD_LOG_ERROR("Render camera failed, camera->Activate failed");
            s_dirtyArea.Set(0.0F, 0.0F, 1.0F, 1.0F);
            return false;
        }

        // Activate current scene.
        scene->Activate();

        // Scene & camera are activated at this point. Notify CameraListeners that renderer is about to render.
        camera->NotifyListenersOnPreRender();

        // Prepare and sort render order according to active camera.
        camera->PrepareRenderOrder(s_dirtyArea);

        // Reset dirty area for next pass of RenderCameraInternal
        s_dirtyArea.Set(0.0F, 0.0F, 1.0F, 1.0F);

        if (!isCameraRenderStrategyAvailable) {
            if (!Render(camera->GetRenderOrder(), /*useImmediateRendering*/ false)) {
                return false;
            }
        }
        else {
            if (!cameraRenderStrategy->Render(camera->GetRenderOrder())) {
                return false;
            }
        }

        // Must call RenderTarget::EndDraw before swap buffers.
        static_cast<void>(camera->Deactivate());

        bool isRenderPassCompleted = isCameraRenderStrategyAvailable ? cameraRenderStrategy->IsRenderPassCompleted() : true;

        bool isSwapRequired = (isRenderPassCompleted &&
            (!renderTarget->IsAutoSwapEnabled()) &&
            (camera->IsSwapEnabled()));

        if (isSwapRequired) {
            renderTarget->SwapBuffers();
        }

        camera->NotifyListenersOnPostRender();
        NotifyListenersOnCameraPostRender(camera);
    }

    bool success = true;
    if (flushTexturePool) {
        success = FlushTexturePool(static_cast<SizeType>(ContextResourcePool::GetActive().GetIndex()))/* && success*/;
    }

    return success;
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()


#ifdef CGIDEVICE_OPENGLES_30
struct LightsUniformBuffer : public UniformBuffer
{
    virtual const void* GetData() const override { return &m_data; }
    virtual SizeType GetSize() const override {
        // Assert that Light::Data has the same size as the light uniform block in the shaders.
        FEATSTD_COMPILETIME_ASSERT(((
            sizeof(Int32)   /* type */ +
            sizeof(Int32)   /* enabled */ +
            sizeof(Float)   /* spotCosCutoff */ +
            sizeof(Float)   /* spotExponent */ +
            sizeof(Float)   /* range */ +
            sizeof(Float)*3 /* padding */ +
            sizeof(Vector3) /* position */ +
            sizeof(Float)   /* padding */ +
            sizeof(Vector3) /* direction */ +
            sizeof(Float)   /* padding */ +
            sizeof(Vector3) /* halfplane */ +
            sizeof(Float)   /* padding */ +
            sizeof(Vector4) /* attenuation */ +
            sizeof(Vector4) /* ambient */ +
            sizeof(Vector4) /* diffuse */ +
            sizeof(Vector4) /* specular */ +
            sizeof(Vector3) /* cameraLookAtVector */ +
            sizeof(Float)   /* padding */) * CANDERA_MAX_LIGHTS_COUNT)
            == sizeof(m_data));

        // Assert that the layout of Data matches with the layout of the uniform
        // block (ShaderParamNames::LightsBlock) in the shader.
        FEATSTD_DEBUG_ASSERT(
            (FEATSTD_OFFSET_OF(Light::UniformBlock, type) == 0) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, enabled) == 4) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, spotCosCutoff) == 8) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, spotExponent) == 12) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, range) == 16) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, position) == 32) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, direction) == 48) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, halfplane) == 64) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, attenuation) == 80) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, ambient) == 96) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, diffuse) == 112) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, specular) == 128) &&
            (FEATSTD_OFFSET_OF(Light::UniformBlock, cameraLookAtVector) == 144));

        return sizeof(m_data);
    }

    Light::UniformBlock m_data[CANDERA_MAX_LIGHTS_COUNT];
};

static LightsUniformBuffer s_lightsUniformBuffer;
static const Light* s_lightsBlock[CANDERA_MAX_LIGHTS_COUNT] = { 0 };

static bool ActivateLightsUniformBuffer(const Shader& shader)
{
    Int uniformBlockIndex = 0;
    bool wasSuccessful = true;
    bool hasLightsBlock = shader.GetUniformLocation(ShaderParamNames::LightsBlock, uniformBlockIndex);

    if (hasLightsBlock) {
        bool isUploaded = s_lightsUniformBuffer.IsUploaded();
        if (!isUploaded) {
            isUploaded = s_lightsUniformBuffer.Upload(DeviceObject::NoHint);
            wasSuccessful = isUploaded;
        }

        if (isUploaded) {
            wasSuccessful = Renderer::BindUniformBufferToShader(s_lightsUniformBuffer, shader, uniformBlockIndex);
            if (wasSuccessful && s_lightsUniformBuffer.IsDirty()) {
                wasSuccessful = Renderer::UpdateUniformBuffer(s_lightsUniformBuffer);
            }
        }
    }

    return wasSuccessful;
}

#endif

bool Renderer::Draw(const VertexBuffer& vertexBuffer, UInt instanceCount)
{
    const VertexGeometry* geometry = vertexBuffer.GetVertexGeometry();
    if (geometry == 0) {
        return false;
    }

#ifdef CGIDEVICE_OPENGLES_30
     bool wasSuccessful = ActivateLightsUniformBuffer(*s_activeShader);
#endif

    SizeType elementStart = vertexBuffer.GetElementStart();
    SizeType elementCount = vertexBuffer.GetElementCount();
    s_statistics.DrawCalls++;

    UInt vertexCount = static_cast<UInt>(elementCount);
    UInt indexCount = geometry->GetIndexCount();
    if (geometry->GetBufferType() != VertexGeometry::ArrayBuffer) {
        vertexCount = geometry->GetVertexCount();
        indexCount = static_cast<UInt>(elementCount);
    }

    if (1 >= instanceCount) {
        s_statistics.Vertices += vertexCount;
        s_statistics.Indices += indexCount;
#ifdef CGIDEVICE_OPENGLES_30
        return (RenderDevice::DrawVertexBuffer(vertexBuffer, elementStart, elementCount) && wasSuccessful);
#else
        return RenderDevice::DrawVertexBuffer(vertexBuffer, elementStart, elementCount);
#endif
    }

    s_statistics.InstancedDrawCalls++;
    s_statistics.Instances += instanceCount;
    s_statistics.Vertices += (vertexCount * instanceCount);
    s_statistics.Indices += (indexCount * instanceCount);
#ifdef CGIDEVICE_OPENGLES_30
    return (RenderDevice::DrawInstancedVertexBuffer(vertexBuffer, FeatStd::Internal::NumericConversion<UInt>(
        instanceCount), elementStart, elementCount) && wasSuccessful);
#else
    return RenderDevice::DrawInstancedVertexBuffer(vertexBuffer, FeatStd::Internal::NumericConversion<UInt>(
        instanceCount), elementStart, elementCount);
#endif
}

bool Renderer::Render(AbstractRenderOrder* renderOrder, bool useImmediateRendering)
{
    const bool isInstancingSupported = RenderDevice::IsInstancingSupported();
    renderOrder->IterationBegin();

    if (useImmediateRendering || (!isInstancingSupported)) {
        // Loop RenderOrder to render all Nodes.
        for (/* Node from where rendering starts */; renderOrder->IterationHasMoreNodes(); renderOrder->IterationMoveToNextNode()) {
            Node* const currentNode = renderOrder->IterationGetNode();
            if (currentNode == 0) {
                FEATSTD_LOG_ERROR("Render camera failed, currentNode == 0.");
                return false;
            }

            NotifyRendererListenersOnNodePreRender(currentNode);
            {
                CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderCameraRenderOrderLoop, currentNode->GetName()));
                currentNode->Render();
            }
            NotifyRendererListenersOnNodePostRender(currentNode);
        }
    }
    else {
        while (renderOrder->IterationHasMoreNodes()) {
            Node* nodeA = renderOrder->IterationGetNode();
            renderOrder->IterationMoveToNextNode();

            const UInt maximumInstanceCount = DetermineMaximumInstanceCount(nodeA);
            if ((1 >= maximumInstanceCount) || (!renderOrder->IterationHasMoreNodes())) {
                NotifyRendererListenersOnNodePreRender(nodeA);
                {
                    CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderCameraRenderOrderLoop, nodeA->GetName()));
                    nodeA->Render();
                }
                NotifyRendererListenersOnNodePostRender(nodeA);
            }
            else {
                MONITOR_INTERNAL_BATCHED_DRAW_BEGIN(nodeA);
                if (!ResetUniformBins(maximumInstanceCount)) {
                    MONITOR_INTERNAL_BATCHED_DRAW_END();
                    return false;
                }

                FEATSTD_DEBUG_ASSERT(0 != Dynamic_Cast<Renderable*>(nodeA)); // was verified by GetMaximumInstanceCount()
                Renderable& renderableA = *(static_cast<Renderable*>(nodeA));
                if (!Activate(renderableA)) {
                    MONITOR_INTERNAL_BATCHED_DRAW_END();
                    continue;
                }

                if (!ActivateShaderParams(renderableA, 0)) {
                    MONITOR_INTERNAL_BATCHED_DRAW_END();
                    continue;
                }

                UInt instanceCount = 1;
                while ((instanceCount < maximumInstanceCount) && renderOrder->IterationHasMoreNodes()) {
                    Node* const nodeB = renderOrder->IterationGetNode();

                    if (!IsBatchable(renderableA, nodeB)) {
                        break;
                    }

                    FEATSTD_DEBUG_ASSERT(0 != Dynamic_Cast<Renderable*>(nodeB)); // was verified by IsBatchable()
                    Renderable& renderableB = *(static_cast<Renderable*>(nodeB));
                    if (ActivateShaderParams(renderableB, instanceCount)) {
                        MONITOR_INTERNAL_BATCHED_DRAW(nodeB);
                        instanceCount++;
                    }

                    renderOrder->IterationMoveToNextNode();
                }

                if (!SubmitUniformBins()) {
                    MONITOR_INTERNAL_BATCHED_DRAW_END();
                    return false;
                }

                {
                    CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderCameraRenderOrderLoop, "Batched Drawcall"));
                    if (!Draw(*(renderableA.m_vertexBuffer), instanceCount)) {
                        FEATSTD_LOG_ERROR("Renderer failed drawing instanced sequence of %d Nodes, starting with:\"%s\".",
                            instanceCount, (nodeA->GetName() == 0) ? "" : nodeA->GetName());
                    }
                }
                MONITOR_INTERNAL_BATCHED_DRAW_END();
            }
        }
    }

    return true;
}

bool Renderer::IsBatchable(const Renderable& renderableA, const Node* nodeB)
{
    const Renderable* const renderableB = Dynamic_Cast<const Renderable*>(nodeB);
    if ((0 == renderableB) || (!renderableB->m_isBatchable)) {
        return false;
    }

    const VertexBuffer* const vertexBufferA = renderableA.m_vertexBuffer.GetPointerToSharedInstance();
    const VertexBuffer* const vertexBufferB = renderableB->m_vertexBuffer.GetPointerToSharedInstance();

    if (vertexBufferA != vertexBufferB) {
        // nodes need to share the same vertex buffer in order to be batch-able
        return false;
    }

    const Appearance* const appearanceB = nodeB->m_appearance.GetPointerToSharedInstance();
    if (0 == appearanceB) {
        return false;
    }

    if (0 != appearanceB->GetNextPass().GetPointerToSharedInstance()) {
        // multi-pass appearances cannot be batched
        return false;
    }

    const Appearance* appearanceA = renderableA.m_appearance.GetPointerToSharedInstance();
    FEATSTD_DEBUG_ASSERT(0 != appearanceA); // was verified by IsBatchable()
    if ((appearanceA == appearanceB))
    {
        // when the appearance is the same, we can skip all the rest of the compares.
        return true;
    }

    AbstractShaderParamSetter* shaderParamSetter = appearanceB->m_shaderParamSetter.GetPointerToSharedInstance();
    if ((0 == shaderParamSetter) || (!shaderParamSetter->IsInstancingSupported()))
    {
        // without a setter or a setter that supports instancing, the uniforms for instancing cannot be initialized
        return false;
    }

    const Shader* const shaderB = appearanceB->m_shader.GetPointerToSharedInstance();
    if (shaderB != appearanceA->m_shader.GetPointerToSharedInstance()) {
        // different shaders cannot be used for instancing
        return false;
    }

    if (appearanceA->m_renderMode.GetPointerToSharedInstance() != appearanceB->m_renderMode.GetPointerToSharedInstance()) {
        // potentially all members of the 2 renderModes could be compared for equality, but due to
        // the cost of that, we classify only nodes with shared renderModes to be batch-able
        return false;
    }

    for (UInt32 i = 0; i < Appearance::TextureUnitCount; ++i)
    {
        const Texture* const textureA = appearanceA->m_textureUnits[i].GetPointerToSharedInstance();
        const Texture* const textureB = appearanceB->m_textureUnits[i].GetPointerToSharedInstance();
        if (textureA != textureB) {
            return false;
        }
    }

    return true;
}

UInt Renderer::DetermineMaximumInstanceCount(const Node* node)
{
    const Renderable* const renderable = Dynamic_Cast<const Renderable*>(node);
    if ((0 == renderable) || (!renderable->m_isBatchable)) {
        return 1;
    }

    const Appearance* const appearance = node->m_appearance.GetPointerToSharedInstance();
    if (0 == appearance) {
        return 1;
    }

    const Shader* const shader = appearance->m_shader.GetPointerToSharedInstance();
    if (0 == shader) {
        return 1;
    }

    const UInt maximumBatchCount = shader->GetMaximumInstanceCount();
    if (1 >= maximumBatchCount) {
        // shader does not support instancing
        return 1;
    }

    AbstractShaderParamSetter* shaderParamSetter = appearance->m_shaderParamSetter.GetPointerToSharedInstance();
    if ((0 == shaderParamSetter) || (!shaderParamSetter->IsInstancingSupported()))
    {
        // without a setter or a setter that supports instancing, the uniforms for instancing cannot be initialized
        return 1;
    }

    if (0 != appearance->GetNextPass().GetPointerToSharedInstance()) {
        // multi-pass appearances cannot be batched
        return 1;
    }

    const VertexBuffer* const vertexBuffer = renderable->m_vertexBuffer.GetPointerToSharedInstance();
    if (0 == vertexBuffer) {
        return 1;
    }

    return maximumBatchCount;
}

bool Renderer::Activate(Renderable& renderable)
{
    // null checks have already been performed, therefore asserts suffice here
    Appearance* appearance = renderable.m_appearance.GetPointerToSharedInstance();
    FEATSTD_DEBUG_ASSERT(0 != appearance);

    Shader* shader = appearance->m_shader.GetPointerToSharedInstance();
    FEATSTD_DEBUG_ASSERT(0 != shader);

    if (!appearance->Activate(&renderable)) {
        FEATSTD_LOG_ERROR("Activation of Appearance failed for Node:\"%s\".", (renderable.GetName() == 0) ? "" : renderable.GetName());
        return false;
    }

    VertexBuffer* vertexBuffer = renderable.m_vertexBuffer.GetPointerToSharedInstance();
    FEATSTD_DEBUG_ASSERT(0 != vertexBuffer);

    if (!vertexBuffer->Activate()) {
        FEATSTD_LOG_ERROR("Activation of VertexBuffer failed for Node:\"%s\".", (renderable.GetName() == 0) ? "" : renderable.GetName());
        return false;
    }

    if (!shader->BindAttributes(VertexBuffer::SharedPointer(vertexBuffer))) {
        FEATSTD_LOG_ERROR("Shader::BindAttributes failed for Node:\"%s\".", (renderable.GetName() == 0) ? "" : renderable.GetName());
        return false;
    }

    return true;
}

bool Renderer::ActivateShaderParams(Renderable& renderable, Int instanceIndex)
{
    Appearance* appearance = renderable.m_appearance.GetPointerToSharedInstance();
    FEATSTD_DEBUG_ASSERT(0 != appearance); // was verified by IsBatchable()

    AbstractShaderParamSetter* shaderParamSetter = appearance->m_shaderParamSetter.GetPointerToSharedInstance();
    FEATSTD_DEBUG_ASSERT(0 != shaderParamSetter); // was verified by IsBatchable()

    if (!shaderParamSetter->Activate(renderable, SharedPointer<Appearance>(appearance), instanceIndex)) {
        FEATSTD_LOG_ERROR("Activation of ShaderParamSetter failed for Node:\"%s\".", (renderable.GetName() == 0) ? "" : renderable.GetName());
        return false;
    }

    return true;
}

UInt32 Renderer::GetCameraCount() { return FeatStd::Internal::NumericConversion<UInt32>(GetCameraContainer().Size()); }
Camera* Renderer::GetCamera(UInt32 index) { return GetCameraContainer()[FeatStd::Internal::NumericConversion<Int>(index)]; }

bool Renderer::AddRendererListener(RendererListener* listener)
{
    bool isSuccessful = false;
    if (listener != 0) {
        isSuccessful = s_rendererListeners.Add(listener);
    }
    return isSuccessful;
}

bool Renderer::RemoveRendererListener(RendererListener* listener)
{
    bool isSuccessful = false;
    if (listener != 0) {
        isSuccessful = s_rendererListeners.Remove(listener);
        if (isSuccessful && (0 == s_rendererListeners.Size())) {
            s_rendererListeners.Free();
        }
    }
    return isSuccessful;
}

/**
 * @brief Class implementing listener notification before rendering.
 */
class Renderer::OnNodePreRenderEvent : public Renderer::RendererListenerContainer::Event
{
public:
    OnNodePreRenderEvent(Node* node) : m_node(node) {}
    virtual void Notify(RendererListener* listener) override
    {
        listener->OnNodePreRender(m_node);
    }
private:
    Node* m_node;
};

/**
 * @brief Class implementing listener notification after rendering.
 */
class Renderer::OnNodePostRenderEvent : public Renderer::RendererListenerContainer::Event
{
public:
    OnNodePostRenderEvent(Node* node) : m_node(node) {}
    virtual void Notify(RendererListener* listener) override
    {
        listener->OnNodePostRender(m_node);
    }
private:
    Node* m_node;
};

/**
*  @brief Class implementing listener notification before camera activation.
*/
class Renderer::OnCameraPreActivateEvent : public Renderer::RendererListenerContainer::Event
{
public:
    OnCameraPreActivateEvent(const Camera *camera) :
        m_camera(camera) {}
    virtual void Notify(RendererListener* listener) {
        listener->OnCameraPreActivate(m_camera);
    }
private:
    const Camera* m_camera;
};

/**
*  @brief Class implementing listener notification after camera activation.
*/
class Renderer::OnCameraPostActivateEvent : public Renderer::RendererListenerContainer::Event
{
public:
    OnCameraPostActivateEvent(const Camera *camera) :
        m_camera(camera) {}
    virtual void Notify(RendererListener* listener) {
        listener->OnCameraPostActivate(m_camera);
    }
private:
    const Camera* m_camera;
};

/**
*  @brief Class implementing listener notification before the camera issues a clear call.
*/
class Renderer::OnCameraPreClearEvent : public Renderer::RendererListenerContainer::Event
{
public:
    OnCameraPreClearEvent(const Camera *camera) :
        m_camera(camera) {}
    virtual void Notify(RendererListener* listener) {
        listener->OnCameraPreClear(m_camera);
    }
private:
    const Camera* m_camera;
};

/**
*  @brief Class implementing listener notification after the camera issued the clear call.
*/
class Renderer::OnCameraPostClearEvent : public Renderer::RendererListenerContainer::Event
{
public:
    OnCameraPostClearEvent(const Camera *camera) :
        m_camera(camera) {}
    virtual void Notify(RendererListener* listener) {
        listener->OnCameraPostClear(m_camera);
    }
private:
    const Camera* m_camera;
};


void Renderer::NotifyRendererListenersOnNodePreRender(Node *node)
{
    OnNodePreRenderEvent event(node);
    s_rendererListeners.Iterate(event);
}

void Renderer::NotifyRendererListenersOnNodePostRender(Node *node)
{
    OnNodePostRenderEvent event(node);
    s_rendererListeners.Iterate(event);
}

void Renderer::SetLight(const Light* light, SizeType index)
{
#ifdef CGIDEVICE_OPENGLES_30
    if (index >= CANDERA_MAX_LIGHTS_COUNT) {
        FEATSTD_DEBUG_FAIL();
        return;
    }

    if (0 == light) {
        s_lightsBlock[index] = 0;
        if (s_lightsUniformBuffer.m_data[index].enabled != 0) {
            s_lightsUniformBuffer.m_data[index].enabled = 0;
            s_lightsUniformBuffer.SetDirty();
        }
    }
    else {
        if ((s_lightsBlock[index] != light) || (light->m_isDataDirty)) {
            s_lightsBlock[index] = light;
            MemoryPlatform::Copy(&(s_lightsUniformBuffer.m_data[index]), &(light->m_uniformBlock), sizeof(Light::UniformBlock));
            light->m_isDataDirty = false;
            s_lightsUniformBuffer.SetDirty();
        }
    }
#else
    FEATSTD_UNUSED2(light, index);
#endif
}

bool Renderer::ResetUniformBins(SizeType expectedSize)
{
    bool wasSuccessful = ResetBatchedBin(expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::Float, Float, 1>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::FloatVec2, Float, 2>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::FloatVec3, Float, 3>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::FloatVec4, Float, 4>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::Integer, Int, 1>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::IntegerVec2, Int, 2>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::IntegerVec3, Int, 3>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::IntegerVec4, Int, 4>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::Bool, Int, 1>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::BoolVec2, Int, 2>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::BoolVec3, Int, 3>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::BoolVec4, Int, 4>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::FloatMat2, Float, 4>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::FloatMat3, Float, 9>(), expectedSize);
    wasSuccessful = wasSuccessful && ResetUniformBin(GetUniformBin<Shader::FloatMat4, Float, 16>(), expectedSize);
    return wasSuccessful;
}

bool Renderer::SubmitUniformBins()
{
    bool wasSuccessful = SubmitUniformBin<Shader::Float, Float, 1>(GetUniformBin<Shader::Float, Float, 1>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::FloatVec2, Float, 2>(GetUniformBin<Shader::FloatVec2, Float, 2>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::FloatVec3, Float, 3>(GetUniformBin<Shader::FloatVec3, Float, 3>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::FloatVec4, Float, 4>(GetUniformBin<Shader::FloatVec4, Float, 4>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::Integer, Int, 1>(GetUniformBin<Shader::Integer, Int, 1>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::IntegerVec2, Int, 2>(GetUniformBin<Shader::IntegerVec2, Int, 2>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::IntegerVec3, Int, 3>(GetUniformBin<Shader::IntegerVec3, Int, 3>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::IntegerVec4, Int, 4>(GetUniformBin<Shader::IntegerVec4, Int, 4>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::Bool, Int, 1>(GetUniformBin<Shader::Bool, Int, 1>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::BoolVec2, Int, 2>(GetUniformBin<Shader::BoolVec2, Int, 2>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::BoolVec3, Int, 3>(GetUniformBin<Shader::BoolVec3, Int, 3>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::BoolVec4, Int, 4>(GetUniformBin<Shader::BoolVec4, Int, 4>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::FloatMat2, Float, 4>(GetUniformBin<Shader::FloatMat2, Float, 4>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::FloatMat3, Float, 9>(GetUniformBin<Shader::FloatMat3, Float, 9>());
    wasSuccessful = wasSuccessful && SubmitUniformBin<Shader::FloatMat4, Float, 16>(GetUniformBin<Shader::FloatMat4, Float, 16>());
    return wasSuccessful;
}

typedef Candera::Internal::Vector<Float> BatchedBin;
BatchedBin& GetBatchedBin()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(BatchedBin, s_batchedBin);
    return s_batchedBin;
}

static BatchedBin& s_forceInitBatchedBin = GetBatchedBin();

/**
*  @brief Class implementing listener notification after the camera has been rendered.
*/
class Renderer::OnCameraPostRenderEvent : public Renderer::RendererListenerContainer::Event
{
public:
    OnCameraPostRenderEvent(Camera *camera) :
      m_camera(camera) {}
      virtual void Notify(RendererListener* listener) {
          listener->OnCameraPostRender(m_camera);
      }
private:
    Camera* m_camera;
};


bool Renderer::ResetBatchedBin(SizeType expectedSize)
{
    expectedSize *= (sizeof(Matrix4) / sizeof(Float)); // Matrix4 is the largest Shader::UniformType

    BatchedBin& bin = GetBatchedBin();
    if (bin.GetCapacity() < expectedSize) {
        if (!bin.Reserve(expectedSize)) {
            return false;
        }

        // add default elements, to avoid per frame copy constructors using Vector<>::Add() later
        const SizeType oldSize = bin.Size();
        for (SizeType i = oldSize; i < bin.GetCapacity(); ++i) {
            static_cast<void>(bin.Add(0.0F));
        }
    }

    return true;
}

template <Shader::UniformType T, typename ElementType, Int ElementCount>
SizeType& Renderer::GetUniformBinSize()
{
    static SizeType s_binSize;
    return s_binSize;
}

template <Shader::UniformType T, typename ElementType, Int ElementCount>
bool Renderer::ResetUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin, SizeType expectedSize)
{
    if (!ResizeUniformBin<T, ElementType, ElementCount>(bin, expectedSize)) {
        return false;
    }

    GetUniformBinSize<T, ElementType, ElementCount>() = 0;
    s_expectedBatchSize = expectedSize;

    return AutoCorrectUniformBinSize<T, ElementType, ElementCount>(bin);
}

template <Shader::UniformType T, typename ElementType, Int ElementCount>
bool Renderer::AutoCorrectUniformBinSize(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin)
{
    SizeType binSize = GetUniformBinSize<T, ElementType, ElementCount>();
    return ResizeUniformBin<T, ElementType, ElementCount>(bin, binSize + s_expectedBatchSize);
}

template <Shader::UniformType T, typename ElementType, Int ElementCount>
bool Renderer::ResizeUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin, SizeType size)
{
    if (bin.GetCapacity() < size) {
        if (!bin.Reserve(size)) {
            return false;
        }

        // add default elements, to avoid per frame copy constructors using Vector<>::Add() later
        const SizeType oldSize = bin.Size();
        for (SizeType i = oldSize; i < bin.GetCapacity(); ++i) {
            bin.Add(UniformBinElement<T, ElementType, ElementCount>());
        }
    }

    return true;
}

template <Shader::UniformType T, typename ElementType, Int ElementCount>
Candera::Internal::Vector<Renderer::UniformBinElement<T, ElementType, ElementCount> >& Renderer::GetUniformBin()
{
    static Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> > s_UniformBin;
    return s_UniformBin;
}

void Renderer::NotifyListenersOnCameraPostRender(Camera *camera)
{
    OnCameraPostRenderEvent event(camera);
    s_rendererListeners.Iterate(event);
}

void Renderer::NotifyListenersOnCameraPreActivate(const Camera *camera)
{
    OnCameraPreActivateEvent event(camera);
    s_rendererListeners.Iterate(event);
}

void Renderer::NotifyListenersOnCameraPostActivate(const Camera *camera)
{
    OnCameraPostActivateEvent event(camera);
    s_rendererListeners.Iterate(event);
}

void Renderer::NotifyListenersOnCameraPreClear(const Camera *camera)
{
    OnCameraPreClearEvent event(camera);
    s_rendererListeners.Iterate(event);
}

void Renderer::NotifyListenersOnCameraPostClear(const Camera *camera)
{
    OnCameraPostClearEvent event(camera);
    s_rendererListeners.Iterate(event);
}



template <Shader::UniformType T, typename ElementType, Int ElementCount>
bool Renderer::AddToUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin, Int location, const void* data)
{
    SizeType& binSize = GetUniformBinSize<T, ElementType, ElementCount>();
    if (binSize >= bin.Size()) {
        if (!AutoCorrectUniformBinSize<T, ElementType, ElementCount>(bin)) {
            return false;
        }
    }

    bool foundPreviousElement = false;
    for (Int i = FeatStd::Internal::NumericConversion<Int>(binSize) - 1; i >= 0; --i) {
        if (bin[i].uniformLocation == location) {
            bin[i].indexOfNextElementWithSameLocation = FeatStd::Internal::NumericConversion<UInt16>(binSize);
            foundPreviousElement = true;
            break;
        }
    }

    bin[binSize].uniformLocation = location;
    bin[binSize].indexOfNextElementWithSameLocation = FeatStd::Internal::NumericConversion<UInt16>(bin.GetCapacity());
    bin[binSize].startsElementsSequence = foundPreviousElement ? 0 : 1;

    const ElementType* elementTypeData = reinterpret_cast<const ElementType*>(data);
    for (SizeType i = 0; i < ElementCount; ++i) {
        bin[binSize].elements[i] = elementTypeData[i];
    }

    binSize++;
    return true;
}

template<>
bool Renderer::SetUniform<Shader::Float>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::Float>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::Float, Float, 1>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::FloatVec2>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::FloatVec2>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::FloatVec2, Float, 2>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::FloatVec3>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::FloatVec3>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::FloatVec3, Float, 3>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::FloatVec4>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::FloatVec4>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::FloatVec4, Float, 4>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::Integer>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::Integer>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::Integer, Int, 1>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::IntegerVec2>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::IntegerVec2>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::IntegerVec2, Int, 2>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::IntegerVec3>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::IntegerVec3>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::IntegerVec3, Int, 3>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::IntegerVec4>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::IntegerVec4>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::IntegerVec4, Int, 4>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::Bool>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::Bool>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::Bool, Int, 1>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::BoolVec2>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::BoolVec2>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::BoolVec2, Int, 2>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::BoolVec3>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::BoolVec3>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::BoolVec3, Int, 3>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::BoolVec4>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::BoolVec4>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::BoolVec4, Int, 4>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::FloatMat2>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::FloatMat2>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::FloatMat2, Float, 4>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::FloatMat3>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::FloatMat3>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::FloatMat3, Float, 9>(), location, data);
}

template<>
bool Renderer::SetUniform<Shader::FloatMat4>(Int location, const void* data, Int instanceIndex)
{
    if (0 > instanceIndex) {
        return SetUniformInRenderDevice<Shader::FloatMat4>(location, data);
    }

    return AddToUniformBin(GetUniformBin<Shader::FloatMat4, Float, 16>(), location, data);
}

bool Renderer::SetUniform(Int location, const void* data, Shader::UniformType type, UInt count, Int instanceIndex)
{
    if (0 > instanceIndex) {
        s_statistics.UniformCalls++;
        s_statistics.Uniforms += count;
        // immediate mode, no batching/instancing
        return RenderDevice::SetUniform(location, data, type, count);
    }

    if (0 == instanceIndex) {
        // Samplers are shared amongst instances and only need to be set once.
        if (Shader::Sampler2D == type) {
            return SetUniformInRenderDevice<Shader::Sampler2D>(location, data, count);
        }
        else {
            if (Shader::SamplerCube == type) {
                return SetUniformInRenderDevice<Shader::SamplerCube>(location, data, count);
            }
        }

        if (1 < count) {
            // This is a shared uniform array, it is not instance-able since
            // array of arrays are not supported in OpenGL ES.
            s_statistics.UniformCalls++;
            s_statistics.Uniforms += count;
            return RenderDevice::SetUniform(location, data, type, count);
        }
    }

    if (1 < count) {
        // This is a shared uniform array, which has already been set with instanceIndex 0.
        // It is not instance-able since array of arrays are not supported in OpenGL ES.
        return false;
    }

    bool ret = false;
    switch (type) {
    case Shader::Float: ret = SetUniform<Shader::Float>(location, data, instanceIndex); break;
    case Shader::FloatVec2: ret = SetUniform<Shader::FloatVec2>(location, data, instanceIndex); break;
    case Shader::FloatVec3: ret = SetUniform<Shader::FloatVec3>(location, data, instanceIndex); break;
    case Shader::FloatVec4: ret = SetUniform<Shader::FloatVec4>(location, data, instanceIndex); break;

    case Shader::Integer: ret = SetUniform<Shader::Integer>(location, data, instanceIndex); break;
    case Shader::IntegerVec2: ret = SetUniform<Shader::IntegerVec2>(location, data, instanceIndex); break;
    case Shader::IntegerVec3: ret = SetUniform<Shader::IntegerVec3>(location, data, instanceIndex); break;
    case Shader::IntegerVec4: ret = SetUniform<Shader::IntegerVec4>(location, data, instanceIndex); break;

    case Shader::Bool: ret = SetUniform<Shader::Bool>(location, data, instanceIndex); break;
    case Shader::BoolVec2: ret = SetUniform<Shader::BoolVec2>(location, data, instanceIndex); break;
    case Shader::BoolVec3: ret = SetUniform<Shader::BoolVec3>(location, data, instanceIndex); break;
    case Shader::BoolVec4: ret = SetUniform<Shader::BoolVec4>(location, data, instanceIndex); break;

    case Shader::FloatMat2: ret = SetUniform<Shader::FloatMat2>(location, data, instanceIndex); break;
    case Shader::FloatMat3: ret = SetUniform<Shader::FloatMat3>(location, data, instanceIndex); break;
    case Shader::FloatMat4: ret = SetUniform<Shader::FloatMat4>(location, data, instanceIndex); break;

    default: break;
    }

    return ret;
}

template <Shader::UniformType T, typename ElementType, Int ElementCount>
bool Renderer::SubmitUniformBin(Candera::Internal::Vector<UniformBinElement<T, ElementType, ElementCount> >& bin)
{
    // To save runtime memory consumption, the same temporary batched bin array is used for Floats and Integers.
    // For this reason, the size has to be the same.
    FEATSTD_COMPILETIME_ASSERT(sizeof(Float) == sizeof(Int32));
    BatchedBin& batchedBin = GetBatchedBin();

    const Int binSize = FeatStd::Internal::NumericConversion<Int>(GetUniformBinSize<T, ElementType, ElementCount>());

    Int startIndexForNextIteration = 0;
    while (startIndexForNextIteration < binSize) {
        Int offset = 0;
        UInt instanceCount = 0;
        for (Int i = startIndexForNextIteration; i < binSize; i = bin[i].indexOfNextElementWithSameLocation) {
            MemoryPlatform::Copy(&(batchedBin[offset]), &(bin[i].elements[0]), sizeof(ElementType)*ElementCount);
            offset += ElementCount;
            instanceCount++;
        }

        if (!SetUniformInRenderDevice<T>(bin[startIndexForNextIteration].uniformLocation, &(batchedBin[0]), instanceCount)) {
            return false;
        }

        startIndexForNextIteration++;
        while (!bin[startIndexForNextIteration].startsElementsSequence && startIndexForNextIteration < binSize) {
            startIndexForNextIteration++;
        }
    }

    return true;
}

bool Renderer::BeginQuery(const Query& query)
{
    if (RenderDevice::BeginQuery(query)) {
        s_statistics.Queries++;
        return true;
    }

    return false;
}

bool Renderer::ActivateCamera(const Camera* camera)
{
    if (0 == camera) {
        return false;
    }

    RenderTarget3D* renderTarget = camera->GetRenderTarget();
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::ActivateCamera, camera->GetName()));
        Renderer::NotifyListenersOnCameraPreActivate(camera);
        const_cast<Camera*>(camera)->NotifyListenersOnPreActivate();

        if ((renderTarget != 0) && (renderTarget->Activate())) {
            renderTarget->Sync();
            renderTarget->SafeBeginDraw(renderTarget);
        }
        else {
            return false;
        }
        const_cast<Camera*>(camera)->NotifyListenersOnPostActivate();
        Renderer::NotifyListenersOnCameraPostActivate(camera);
    }

    bool success = true;
    RenderDevice::SetActiveCamera(camera);
    RenderDevice::ActivateViewport(*camera);

    if (camera->IsClearingEnabled()) {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::ClearCamera, camera->GetName()));
        Renderer::NotifyListenersOnCameraPreClear(camera);
        const_cast<Camera*>(camera)->NotifyListenersOnPreClear();

        success = ActivateClearMode(*camera);

        const_cast<Camera*>(camera)->NotifyListenersOnPostClear();
        Renderer::NotifyListenersOnCameraPostClear(camera);
    }
    else {
        if (camera->IsScissoringEnabled()) {
            success = ActivateScissor(camera->GetScissorRectangle(), renderTarget);
        }
        else {
            const bool useDirtyArea = (s_dirtyArea.GetLeft() != 0.0F) || (s_dirtyArea.GetTop() != 0.0F) ||
                (s_dirtyArea.GetWidth() != 1.0F) || (s_dirtyArea.GetHeight() != 1.0F);
            if (useDirtyArea) {
                success = ActivateScissor(s_dirtyArea, renderTarget);
            }
            else {
                success = RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 0);
            }
        }
    }

    if (!success) {
        static_cast<void>(DeactivateCamera(camera));
    }

    return success;
}

bool Renderer::DeactivateCamera(const Camera* camera)
{
    if (0 == camera) {
        return false;
    }

    RenderDevice::SetActiveCamera(0);

    if (0 == camera->GetRenderTarget()) {
        return false;
    }

    camera->GetRenderTarget()->SafeEndDraw();
    return true;
}

bool Renderer::ActivateClearMode(const Camera& camera)
{
    const RenderTarget3D* renderTarget = camera.GetRenderTarget();
    if (renderTarget == 0) {
        return false;
    }

    bool isScissoringEnabledForClearingOnly = false;
    bool success = true;

    // If scissoring is enabled activate scissor rectangle.
    if (camera.IsScissoringEnabled()) {
        success = ActivateScissor(camera.GetScissorRectangle(), renderTarget);
    }
    else {
        const bool useDirtyArea = (s_dirtyArea.GetLeft() != 0.0F) || (s_dirtyArea.GetTop() != 0.0F) ||
            (s_dirtyArea.GetWidth() != 1.0F) || (s_dirtyArea.GetHeight() != 1.0F);
        if (useDirtyArea) {
            success = ActivateScissor(s_dirtyArea, renderTarget);
        }
        else {
            // If scissoring is disabled, clearing is enabled and the viewport
            // does not contain the entire RenderTarget, temporarily activate the
            // viewport as the scissor rectangle during clearing.
            isScissoringEnabledForClearingOnly = (camera.GetViewport().GetTop() > Float(0)) ||
                (camera.GetViewport().GetLeft() > Float(0)) ||
                ((camera.GetViewport().GetTop() + camera.GetViewport().GetHeight()) < Float(1)) ||
                ((camera.GetViewport().GetLeft() + camera.GetViewport().GetWidth()) < Float(1));

            if (isScissoringEnabledForClearingOnly) {
                success = ActivateScissor(camera.GetViewport(), renderTarget);
            }
            else {
                success = RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 0);
            }
        }
    }

    success = success && ActivateClearMode(&(camera.GetClearMode()));

    //Disable temporary scissor rectangle.
    if (isScissoringEnabledForClearingOnly) {
        success = success && RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 0);
    }

    return success;
}

bool Renderer::ActivateClearMode(const ClearMode* clearMode)
{
    if (0 == clearMode) {
        return false;
    }

    MONITOR_INTERNAL_CLEAR_3D_BEGIN();

    bool success = true;
    bool isEffectiveColorClearEnabled = false;
    if (clearMode->IsSkyBoxEnabled() && (clearMode->GetSkyBox() != 0)) {
        success = clearMode->GetSkyBox()->Activate();
    }
    else if (clearMode->IsColorClearEnabled()) {
        success = RenderDevice::ActivateColorWriteEnabled(
            clearMode->IsColorWriteRedEnabled(),
            clearMode->IsColorWriteGreenEnabled(),
            clearMode->IsColorWriteBlueEnabled(),
            clearMode->IsColorWriteAlphaEnabled());
        success = success && RenderDevice::ActivateClearColor(clearMode->GetClearColor());
        isEffectiveColorClearEnabled = true;
    }
    else {
        //do nothing
    }

    if (clearMode->IsDepthClearEnabled()) {
        success = success && RenderDevice::ActivateRenderState(RenderDevice::DepthWriteEnable, 1U);
        success = success && RenderDevice::ActivateDepthClearValue(clearMode->GetClearDepth());
    }

    if (clearMode->IsStencilClearEnabled()) {
        success = success && RenderDevice::ActivateStencilWriteMask(clearMode->GetStencilWriteMask());
        success = success && RenderDevice::ActivateStencilClearValue(clearMode->GetStencilClearValue());
    }

    success = success && RenderDevice::ClearFrameBuffer(isEffectiveColorClearEnabled, clearMode->IsDepthClearEnabled(),
        clearMode->IsStencilClearEnabled());

    if (success) {
        if (isEffectiveColorClearEnabled) {
            ++s_statistics.ColorBufferClears;
        }

        if (clearMode->IsDepthClearEnabled()) {
            ++s_statistics.DepthBufferClears;
        }

        if (clearMode->IsStencilClearEnabled()) {
            ++s_statistics.StencilBufferClears;
        }
    }

    MONITOR_INTERNAL_CLEAR_3D_END();
    return success;
}

bool Renderer::ActivateScissor(const Rectangle& rectangle, const RenderTarget3D* renderTarget)
{
    bool success = RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 1);

    if ((s_dirtyArea.GetLeft() != 0.0F) || (s_dirtyArea.GetTop() != 0.0F) ||
        (s_dirtyArea.GetWidth() != 1.0F) || (s_dirtyArea.GetHeight() != 1.0F)) {
        return (success && RenderDevice::ActivateScissorRectangle(s_dirtyArea, renderTarget));
    }

    return (success && RenderDevice::ActivateScissorRectangle(rectangle, renderTarget));
}

UInt64 GetPropertyKey(const BitmapTextureImage& bitmapTextureImage)
{
    const Bitmap* bitmap = bitmapTextureImage.GetBitmap().GetPointerToSharedInstance();
    FEATSTD_DEBUG_ASSERT(0 != bitmap);

    // PropertyKey layout:
    // f = pixel format, a = pack alignment, m = mipmapping enabled, w = width, h = height
    // ffffffff|ffffffff|ffffffff|fffaaaam|wwwwwwww|wwwwwwww|hhhhhhhh|hhhhhhhh
    const UInt packAlignmentBits = 4;
    const UInt mipmappingEnabledBits = 1;

    FEATSTD_DEBUG_ASSERT(bitmap->GetPackAlignment() < (1 << packAlignmentBits));
    FEATSTD_DEBUG_ASSERT(bitmap->GetPixelFormat() < (1 << (32 - packAlignmentBits - mipmappingEnabledBits)));
    UInt64 propertyKey = (static_cast<UInt64>(bitmap->GetPixelFormat()) << (packAlignmentBits + mipmappingEnabledBits))
        | (static_cast<UInt64>(bitmap->GetPackAlignment()) << (mipmappingEnabledBits))
        | (bitmapTextureImage.IsMipMappingEnabled() ? 1U : 0U);

    const UInt64 shiftBy32Bits = 0x100000000; // actual shifting by 32 bits is undefined behavior in C/C++.
    propertyKey = (static_cast<UInt64>(propertyKey)* shiftBy32Bits)
        | (static_cast<UInt64>(bitmap->GetWidth()) << 16)
        | static_cast<UInt64>(bitmap->GetHeight());
    return propertyKey;
}

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UploadBitmapTextureImage(BitmapTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading BitmapTextureImage (name: '%s', unit: %i) while rendering.", textureImage.GetName(), unit);
    }

#ifdef CANDERA_DISABLE_TEXTURE_POOL
    if (RenderDevice::UploadBitmapTextureImageInternal(textureImage, unit)) {
        ++s_statistics.UploadedTextures;
        ++s_statistics.TotalTextureUploads;
        return true;
    }

    return false;
#else
    bool success = false;
    Bitmap* bitmap = textureImage.GetBitmap().GetPointerToSharedInstance();
    if (0 != bitmap) {
        Int activeContextIndex = ContextResourcePool::GetActive().GetIndex();
        Candera::Internal::Vector<TexturePoolEntry>& texturePool = s_texturePool[activeContextIndex];
        const UInt64 propertyKey = GetPropertyKey(textureImage);
        for (Int i = 0; i < texturePool.Size(); ++i) {
            if (propertyKey == texturePool[i].m_propertyKey) {
                Handle handle = texturePool[i].m_handle;
                static_cast<bool>(texturePool.Remove(i));
                --s_statistics.UploadedTexturesInReusePool;
                ++s_statistics.TotalReusedTextureUploads;
                success = RenderDevice::ReuseTexture(handle, textureImage, unit);
                break;
            }
        }
    }

    if (!success) {
        success = RenderDevice::UploadBitmapTextureImageInternal(textureImage, unit);
    }

    if (success) {
        ++s_statistics.UploadedTextures;
        ++s_statistics.TotalTextureUploads;
    }

    return success;
#endif
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UploadCubeMapTextureImage(CubeMapTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading CubeMapTextureImage (name: '%s', unit: %i) while rendering.", textureImage.GetName(), unit);
    }
    return RenderDevice::UploadCubeMapTextureImage(textureImage, unit);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UnloadBitmapTextureImage(BitmapTextureImage& textureImage, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading BitmapTextureImage (name: '%s') while rendering.", textureImage.GetName());
    }

#ifdef CANDERA_DISABLE_TEXTURE_POOL
    if (RenderDevice::UnloadBitmapTextureImageInternal(textureImage)) {
        --s_statistics.UploadedTextures;
        return true;
    }

    return false;
#else
    Bitmap* bitmap = textureImage.GetBitmap().GetPointerToSharedInstance();
    bool success = false;
    if (0 != bitmap) {
        TexturePoolEntry texturePoolEntry;
        texturePoolEntry.m_propertyKey = GetPropertyKey(textureImage);
        texturePoolEntry.m_handle = textureImage.GetVideoMemoryHandle();
        texturePoolEntry.m_size = textureImage.GetSize();
        ++s_statistics.UploadedTexturesInReusePool;
        success = s_texturePool[ContextResourcePool::GetActive().GetIndex()].Add(texturePoolEntry);

        RenderDevice::SetTextureStateCache(textureImage, 0);
        textureImage.SetVideoMemoryHandle(0);

        // Unload the textureImage from the VideoMemoryStatistic, because it stored a raw pointer to it
        // that can be invalid after unloading.
        Diagnostics::VideoMemoryStatistic::OnTextureImageUnloaded(textureImage);

        // The memory is still in use though, so we have to add it again to the statistics.
        Diagnostics::VideoMemoryStatistic::AddTextureMemoryUsed(texturePoolEntry.m_size);
    }
    else {
        success = RenderDevice::UnloadBitmapTextureImageInternal(textureImage);
    }

    if (success) {
        --s_statistics.UploadedTextures;
    }

    return success;
#endif
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

bool Renderer::FlushTexturePool(SizeType contextIndex)
{
    const SizeType texturePoolSize = sizeof(s_texturePool) / sizeof(Candera::Internal::Vector<TexturePoolEntry>);
    if (texturePoolSize <= contextIndex) {
        return false;
    }

    bool success = true;
    for (Int j = 0; j < static_cast<Int>(s_texturePool[contextIndex].Size()); ++j) {
        if (RenderDevice::DestroyTexture(s_texturePool[contextIndex][j].m_handle)) {
            Diagnostics::VideoMemoryStatistic::SubtractTextureMemoryUsed(s_texturePool[contextIndex][j].m_size);
            --s_statistics.UploadedTexturesInReusePool;
        }
        else {
            success = false;
        }
    }

    s_texturePool[contextIndex].Clear();
    return success;
}

bool Renderer::FlushTexturePools()
{
    const Int activeContextIndex = ContextResourcePool::GetActive().GetIndex();

    // Flush currently active texture pool.
    bool success = FlushTexturePool(static_cast<SizeType>(activeContextIndex));

    for (Int i = 0; i < CANDERA_MAX_CONTEXT_COUNT; ++i) {
        if (s_texturePool[i].Size() > 0) {
            // Active context has already been flushed to avoid unnecessary context switch.
            FEATSTD_DEBUG_ASSERT(i != activeContextIndex);

            ContextResourcePool* context = ContextResourcePool::GetContextResourcePool(i);
            if (context->Activate()) {
                success = FlushTexturePool(static_cast<SizeType>(activeContextIndex)) && success;
            }
            else {
                success = false;
            }
        }
    }

    // restore active context
    if (ContextResourcePool::GetActive().GetIndex() != activeContextIndex) {
        success = (ContextResourcePool::GetContextResourcePool(activeContextIndex)->Activate()) && success;
    }

    return success;
}

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UnloadCubeMapTextureImage(CubeMapTextureImage& textureImage, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading CubeMapTextureImage (name: '%s') while rendering.", textureImage.GetName());
    }
    return RenderDevice::UnloadCubeMapTextureImage(textureImage);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UploadVertexBuffer(VertexBuffer& vertexBuffer, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading VertexBuffer (name: '%s') while rendering.", vertexBuffer.GetName());
    }
    return RenderDevice::UploadVertexBuffer(vertexBuffer);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UnloadVertexBuffer(VertexBuffer& vertexBuffer, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading VertexBuffer(name: '%s') while rendering.", vertexBuffer.GetName());
    }
    return RenderDevice::UnloadVertexBuffer(vertexBuffer);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UploadShader(Shader& shader, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading Shader (name: '%s') while rendering.", shader.GetName());
    }
    return RenderDevice::UploadShader(shader);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::DeleteShader(Handle vertexShaderHandle, Handle fragmentShaderHandle, Handle programHandle, const Char* shaderName, DeviceObject::LoadingHint loadingHint)
{
    FEATSTD_UNUSED(shaderName); // if logging is disabled, variable/parameter is not used
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading Shader (name: '%s') while rendering.", shaderName);
    }
    return RenderDevice::DeleteShader(vertexShaderHandle, fragmentShaderHandle, programHandle);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::ActivateShader(const Shader& shader, Handle program)
{
    s_activeShader = &shader;
    return RenderDevice::ActivateShader(program);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()


FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UploadRenderBuffer(RenderBuffer& renderBuffer, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading RenderBuffer (name: '%s') while rendering.", renderBuffer.GetName());
    }
    return RenderDevice::UploadRenderBuffer(renderBuffer);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UnloadRenderBuffer(RenderBuffer& renderBuffer, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading RenderBuffer (name: '%s') while rendering.", renderBuffer.GetName());
    }
    return RenderDevice::UnloadRenderBuffer(renderBuffer);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UploadDirectTextureImage(DirectTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading DirectTextureImage (name: '%s', unit: %i) while rendering.", textureImage.GetName(), unit);
    }
    return RenderDevice::UploadDirectTextureImage(textureImage, unit);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UnloadDirectTextureImage(DirectTextureImage& textureImage, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading DirectTextureImage (name: '%s') while rendering.", textureImage.GetName());
    }
    return RenderDevice::UnloadDirectTextureImage(textureImage);
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()


// external texture
FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UploadExternalTextureImage(ExternalTextureImage& textureImage, UInt unit, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading ExternalTextureImage (name: '%s', unit: %i) while rendering.", textureImage.GetName(), unit);
    }

    if (RenderDevice::UploadExternalTextureImage(textureImage, unit)) {
        ++s_statistics.UploadedExternalTextureImages;
        ++s_statistics.TotalExternalTextureImageUploads;
        return true;
    }

    return false;
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
bool Renderer::UnloadExternalTextureImage(ExternalTextureImage& textureImage, DeviceObject::LoadingHint loadingHint)
{
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading ExternalTextureImage (name: '%s') while rendering.", textureImage.GetName());
    }

    if (RenderDevice::UnloadExternalTextureImage(textureImage)) {
        --s_statistics.UploadedExternalTextureImages;
        return true;
    }

    return false;
}
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
//end external texture

bool Renderer::GetActiveUniformBlockCount(const Shader& shader, Int& count)
{
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::GetActiveUniformBlockCount(shader.GetProgramHandle(), count);
#else
    FEATSTD_UNUSED(shader);
    count = 0;
    return true;
#endif
}

bool Renderer::GetActiveUniformBlockMaxNameLength(const Shader& shader, Int& maxLength)
{
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::GetActiveUniformBlockMaxNameLength(shader.GetProgramHandle(), maxLength);
#else
    FEATSTD_UNUSED(shader);
    maxLength = 0;
    return true;
#endif
}

bool Renderer::GetActiveUniformBlockName(const Shader& shader, UInt uniformBlockIndex, Int nameBufferSize,
    Int& nameLength, Char* uniformBlockName)
{
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::GetActiveUniformBlockName(shader.GetProgramHandle(), uniformBlockIndex, nameBufferSize,
        nameLength, uniformBlockName);
#else
    FEATSTD_UNUSED4(shader, uniformBlockIndex, nameBufferSize, uniformBlockName);
    nameLength = 0;
    return true;
#endif
}


bool Renderer::GetActiveUniformBlockSize(const Shader& shader, UInt uniformBlockIndex, Int& blockSize)
{
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::GetActiveUniformBlockSize(shader.GetProgramHandle(), uniformBlockIndex, blockSize);
#else
    FEATSTD_UNUSED2(shader, uniformBlockIndex);
    blockSize = 0;
    return true;
#endif
}

bool Renderer::UploadUniformBuffer(UniformBuffer& uniformBuffer, DeviceObject::LoadingHint loadingHint)
{
#ifdef CGIDEVICE_OPENGLES_30
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading UniformBlock (name: '%s') while rendering.", uniformBuffer.GetName());
    }

    Handle deviceHandle = 0;
    if (RenderDevice::UploadUniformBuffer(uniformBuffer.GetData(), uniformBuffer.GetSize(), deviceHandle)) {
        uniformBuffer.m_deviceHandle[ContextResourcePool::GetActive().GetIndex()] = deviceHandle;
        uniformBuffer.m_isDirty = false;
        ++s_statistics.UniformBufferObjects;
        return true;
    }
#else
    FEATSTD_UNUSED2(uniformBuffer, loadingHint);
#endif
    return false;
}

bool Renderer::UpdateUniformBuffer(UniformBuffer& uniformBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    if (RenderDevice::UpdateUniformBuffer(uniformBuffer.m_deviceHandle[ContextResourcePool::GetActive().GetIndex()],
        uniformBuffer.GetData(), uniformBuffer.GetSize())) {
        uniformBuffer.m_isDirty = false;
        ++s_statistics.UniformBufferUpdates;
        return true;
    }
#else
    FEATSTD_UNUSED(uniformBuffer);
#endif
    return false;
}

bool Renderer::UnloadUniformBuffer(UniformBuffer& uniformBuffer, DeviceObject::LoadingHint loadingHint)
{
#ifdef CGIDEVICE_OPENGLES_30
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading UniformBlock (name: '%s') while rendering.", uniformBuffer.GetName());
    }

    Int activeContextIndex = ContextResourcePool::GetActive().GetIndex();
    if (RenderDevice::UnloadBuffer(uniformBuffer.m_deviceHandle[activeContextIndex])) {
        uniformBuffer.m_deviceHandle[activeContextIndex] = Handle();
        uniformBuffer.SetDirty();
        --s_statistics.UniformBufferObjects;
        return true;
    }
#else
    FEATSTD_UNUSED2(uniformBuffer, loadingHint);
#endif
    return false;
}

bool Renderer::BindUniformBufferToShader(const UniformBuffer& uniformBuffer, const Shader& shader, Int uniformBlockIndex)
{
#ifdef CGIDEVICE_OPENGLES_30
    if (RenderDevice::BindUniformBufferToBlock(uniformBuffer.m_deviceHandle[ContextResourcePool::GetActive().GetIndex()],
        shader.GetProgramHandle(), static_cast<UInt>(uniformBlockIndex), shader.GetCurrentBindingPoint())) {
        shader.IncrementCurrentBindingPoint();
        return true;
    }
#else
    FEATSTD_UNUSED3(uniformBuffer, shader, uniformBlockIndex);
#endif
    return false;
}

bool Renderer::IsUniformBufferSupported()
{
#ifdef CGIDEVICE_OPENGLES_30
    return true;
#else
    return false;
#endif
}

bool Renderer::UploadPixelBuffer(PixelBuffer& pixelBuffer, DeviceObject::LoadingHint loadingHint)
{
#ifdef CGIDEVICE_OPENGLES_30
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Uploading PixelBuffer (name: '%s') while rendering.", pixelBuffer.GetName());
    }

    Handle deviceHandle = 0;
    if (RenderDevice::UploadPixelBuffer(0, pixelBuffer.GetSize(), pixelBuffer.IsPixelPack(), deviceHandle)) {
        pixelBuffer.m_deviceHandle[ContextResourcePool::GetActive().GetIndex()] = deviceHandle;
        ++s_statistics.PixelBufferObjects;
        return true;
    }
#else
    FEATSTD_UNUSED2(pixelBuffer, loadingHint);
#endif
    return false;
}

bool Renderer::UnloadPixelBuffer(PixelBuffer& pixelBuffer, DeviceObject::LoadingHint loadingHint)
{
#ifdef CGIDEVICE_OPENGLES_30
    if ((DeviceObject::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
        FEATSTD_LOG_WARN("Unloading PixelBuffer (name: '%s') while rendering.", pixelBuffer.GetName());
    }

    Int activeContextIndex = ContextResourcePool::GetActive().GetIndex();
    if (RenderDevice::UnloadBuffer(pixelBuffer.m_deviceHandle[activeContextIndex])) {
        pixelBuffer.m_deviceHandle[activeContextIndex] = Handle();
        --s_statistics.PixelBufferObjects;
        return true;
    }
#else
    FEATSTD_UNUSED2(pixelBuffer, loadingHint);
#endif
    return false;
}

bool Renderer::BindPixelBuffer(const PixelBuffer& pixelBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    Int activeContextIndex = ContextResourcePool::GetActive().GetIndex();
    return RenderDevice::BindPixelBuffer(pixelBuffer.m_deviceHandle[activeContextIndex], pixelBuffer.IsPixelPack());
#else
    FEATSTD_UNUSED(pixelBuffer);
    return false;
#endif
}

bool Renderer::UnbindPixelBuffer(const PixelBuffer& pixelBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::UnbindPixelBuffer(pixelBuffer.IsPixelPack());
#else
    FEATSTD_UNUSED(pixelBuffer);
    return false;
#endif
}

void* Renderer::MapPixelBuffer(const PixelBuffer& pixelBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    ++s_statistics.PixelBufferMappings;
    return RenderDevice::MapPixelBuffer(0, pixelBuffer.GetSize(), pixelBuffer.IsPixelPack());
#else
    FEATSTD_UNUSED(pixelBuffer);
    return 0;
#endif
}

bool Renderer::UnmapPixelBuffer(const PixelBuffer& pixelBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::UnmapPixelBuffer(pixelBuffer.IsPixelPack());
#else
    FEATSTD_UNUSED(pixelBuffer);
    return false;
#endif
}

bool Renderer::IsPixelBufferSupported()
{
#ifdef CGIDEVICE_OPENGLES_30
    return true;
#else
    return false;
#endif
}

bool Renderer::BindTransformFeedbackBuffer(const VertexBuffer& vertexBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    if (Handle() != vertexBuffer.GetIndexBufferMemoryHandle()) {
        // Transform feedback buffers must not have an index buffer.
        return false;
    }

    return RenderDevice::BindTransformFeedbackBuffer(vertexBuffer.GetVertexArrayMemoryHandle(), 0);
#else
    FEATSTD_UNUSED(vertexBuffer);
    return false;
#endif
}

bool Renderer::UnbindTransformFeedbackBuffer(const VertexBuffer& vertexBuffer)
{
    FEATSTD_UNUSED(vertexBuffer);
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::UnbindTransformFeedbackBuffer(0);
#else
    return false;
#endif
}

bool Renderer::IsTransformFeedbackBufferSupported()
{
#ifdef CGIDEVICE_OPENGLES_30
    return true;
#else
    return false;
#endif
}

bool Renderer::BeginTransformFeedback(const VertexBuffer& vertexBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    RenderDevice::TransformFeedbackPrimitiveType primitiveType = RenderDevice::TransformFeedbackPrimitiveTypeCount;
    switch (vertexBuffer.GetPrimitiveType()) {
    case VertexBuffer::Points: primitiveType = RenderDevice::Points; break;
    case VertexBuffer::Lines: primitiveType = RenderDevice::Lines; break;
    case VertexBuffer::Triangles: primitiveType = RenderDevice::Triangles; break;
    default:
        return false;
    }

    return RenderDevice::BeginTransformFeedback(primitiveType);
#else
    FEATSTD_UNUSED(vertexBuffer);
    return false;
#endif
}

bool Renderer::EndTransformFeedback()
{
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::EndTransformFeedback();
#else
    return false;
#endif
}

void* Renderer::MapTransformFeedbackBuffer(const VertexBuffer& vertexBuffer)
{
#ifdef CGIDEVICE_OPENGLES_30
    SizeType length = static_cast<SizeType>(vertexBuffer.GetSize());
    return RenderDevice::MapTransformFeedbackBuffer(0, length);
#else
    FEATSTD_UNUSED(vertexBuffer);
    return 0;
#endif
}

bool Renderer::UnmapTransformFeedbackBuffer(const VertexBuffer& vertexBuffer)
{
    FEATSTD_UNUSED(vertexBuffer);
#ifdef CGIDEVICE_OPENGLES_30
    return RenderDevice::UnmapTransformFeedbackBuffer();
#else
    return false;
#endif
}

void Renderer::OnContextUnload(const ContextResourcePool& contextResourcePool)
{
    ContextResourcePool::ContextProviderIterator contextProviderIterator = contextResourcePool.GetContextProviderIterator();
    SizeType contextProviderCount = 0;
    while (contextProviderIterator.IsValid()) {
        contextProviderIterator++;
        contextProviderCount++;
    }

    if (contextProviderCount <= 1) {
        GlyphAtlas3D::GetInstance().Clear();
#ifdef CANDERA_2D_OVER_3D_ENABLED
        GlyphAtlas& glyphAtlas2D = GlyphAtlas::GetInstance();
        bool isUsedInThisContext = false;
        for (SizeType i = 0; i < glyphAtlas2D.GetBitmapImageCount(); ++i) {
            const BitmapImage2D::SharedPointer& bitmapImage2D = glyphAtlas2D.GetBitmapImage(i);
            SharedPointer<TextureImage> textureImage = Candera::Internal::SurfaceAllocator2DOver3D::GetTextureImage(bitmapImage2D->Get2DSurfaceHandle());
            FEATSTD_DEBUG_ASSERT(textureImage != 0);
            BitmapTextureImage* bitmapTextureImage = Dynamic_Cast<BitmapTextureImage*>(textureImage.GetPointerToSharedInstance());
            if ((0 != bitmapTextureImage) && (bitmapTextureImage->IsUploaded())) {
                isUsedInThisContext = true;
                break;
            }
        }

        if (isUsedInThisContext) {
            glyphAtlas2D.Clear();
        }
#endif
    }

#ifdef CGIDEVICE_OPENGLES_30
    static_cast<void>(s_lightsUniformBuffer.Unload(DeviceObject::NoHint));
#endif
    RenderDevice::OnContextUnload(static_cast<SizeType>(contextResourcePool.GetIndex()));
}

#ifdef CANDERA_DISABLE_TEXTURE_POOL
FEATSTD_LINT_NEXT_EXPRESSION(1960, "Misra 16-0-3 #undef is necessary here")
#undef CANDERA_DISABLE_TEXTURE_POOL
#endif

} // namespace Candera
