//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RENDERMODE_H)
#define CANDERA_RENDERMODE_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/EngineBase/Common/InstanceId.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
/**  @addtogroup Core3D
 *   @{
 */

/**
 *  @brief   The class RenderMode is an Appearance component that encapsulates polygon-level and per-fragment compositing render attributes.
 *            If an object's Appearance has set RenderMode to null, then the default RenderMode is used instead. For details how to set the
 *            default render mode, see Renderer::SetDefaultRenderMode.
 *
 *            If a Camera has a RenderMode attached [camera->GetApperance()->SetRenderMode(...)], then the Camera's RenderMode overrules the
 *            DefaultRenderMode during its render pass. Thus, all Nodes rendered by the Camera that do not have their
 *            own RenderMode set, use the RenderMode applied to the Camera.
 *
 *            If a RenderMode has set an inheritance bit for a certain render attribute, then the property of the base render mode is used,
 *            this is either the Camera's RenderMode if set or the DefaultRenderMode otherwise.
 */
class RenderMode: public CanderaObject
{
    FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(RenderMode);

        /**
         *  Culling determines which side of a polygon is removed.
         */
        enum Culling
        {
            FrontFaceCulling = 0,   ///< Front face culling.
            BackFaceCulling  = 1,   ///< Back face culling.
            NoCulling       = 3     ///< Culling disabled.
        };

        /**
         *  Winding defines the front face of a polygon.
         *  A polygon side is the front-face if its screen-space vertices are in the same order as the winding specifies.
         */
        enum Winding
        {
            ClockWise  = 0,         ///< Clockwise winding.
            CounterClockWise = 1    ///< Counter-clockwise winding.
        };

        /**
         *  ComparisonFunction defines the comparison function for depth-, stencil-, or sampler state operations.
         *  A comparison function specifies how the source (new) data is compared against the destination (existing) data
         *  before passing the comparison operations (storing the data).
         */
        enum ComparisonFunction
        {
            CompareNever = 0,           ///< Never pass the comparison.
            CompareLess = 1,            ///< If source data is less than destination data, the comparison passes.
            CompareEqual = 2,           ///< If source data is equal than destination data, the comparison passes.
            CompareLessEqual = 3,       ///< If source data is less than or equal to destination data, the comparison passes.
            CompareGreater = 4,         ///< If source data is greater than destination data, the comparison passes.
            CompareNotEqual = 5,        ///< If source data is not equal to destination data, the comparison passes.
            CompareGreaterEqual = 6,    ///< If source data is greater than or equal to destination data, the comparison passes.
            CompareAlways = 7           ///< Always pass the comparison.
        };

        /**
         *  BlendFactor specifies the source and destination fragment's coefficient in blending equation.
         */
        enum BlendFactor
        {
            Zero = 0,                   ///< Zero.
            One  = 1,                   ///< One.
            SourceColor = 2,            ///< Source color.
            InverseSourceColor = 3,     ///< Inverse source color.
            SourceAlpha = 4,            ///< Source alpha.
            InverseSourceAlpha = 5,     ///< Inverse source alpha.
            DestColor = 6,              ///< Destination color.
            InverseDestColor = 7,       ///< Inverse destination color.
            DestAlpha = 8,              ///< Destination alpha.
            InverseDestAlpha = 9,       ///< Inverse destination alpha.
            ConstantColor = 10,         ///< Constant color.
            InverseConstantColor = 11,  ///< Inverse constant color.
            ConstantAlpha = 12,         ///< Constant alpha.
            InverseConstantAlpha = 13,  ///< Inverse constant alpha.
            SourceAlphaSaturate = 14    ///< Source alpha saturate.
        };

        /**
         *  BlendOperation specifies the blend operation used in blending equation.
         */
        enum BlendOperation
        {
            Add = 0,               ///< Add source and destination.
            Subtract = 1,          ///< Subtract destination from source.
            ReverseSubtract = 2,   ///< Subtract source from destination.
            Min = 3,               ///< Component-wise minimum value of source and destination.
            Max = 4                ///< Component-wise maximum value of source and destination.
        };

        /**
         *  StencilPlane specifies the face associated with the provided stencil function.
         */
        enum StencilPlane
        {
            FrontFace = 0,        ///< Front face.
            BackFace = 1,         ///< Back face.
            FrontAndBackFace = 2  ///< Front and back face.
        };

        /**
         *  StencilOperation defines the stencil-buffer operation.
         */
        enum StencilOperation
        {
            SetToZero = 0,      ///< Set the stencil-buffer entry to zero.
            Keep = 1,           ///< Do not update the entry in the stencil buffer.
            Replace = 2,        ///< Replace the stencil-buffer entry with the reference value.
            Increment = 3,      ///< Increment the stencil-buffer entry, clamping to 2^n where n is the number of bits in the stencil buffer.
            Decrement = 4,      ///< Decrement the stencil-buffer entry, clamping to zero.
            Invert = 5,         ///< Invert the bits in the stencil-buffer entry.
            IncrementWrap = 6,  ///< Increment the stencil-buffer entry, wrapping to zero if the new value exceeds the maximum value.
            DecrementWrap = 7   ///< Decrement the stencil-buffer entry, wrapping to the maximum value if the new value is less than zero.
        };

        /**
         *  CoverageOperation defines the coverage operation if coverage sampling is enabled (See OpenGLES extension GL_NV_coverage_sample).
         */
        enum CoverageOperation
        {
            CoverAllFragments = 0,  ///< Cover all fragments.
            CoverEdgeFragments = 1, ///< Cover edge fragments.
            CoverAutomatic = 2      ///< Cover automatic.
        };

        /**
         *  BlendMode combines the source and destination blend factors with the operation used in blending equation.
         *  RGB blending equation: final_RGB = source_RGB *  sourceRGBFactor (+operationRGB+) destination_RGB *  destRGBFactor.
         *  Alpha blending equation: final_alpha = source_alpha *  sourceAlphaFactor (+operationAlpha+) destination_alpha *  destAlphaFactor.
         *  Default values are: sourceRGBFactor = SourceAlpha, destRGBFactor = InverseSourceAlpha, operationRGB = Add,
         *                      sourceAlphaFactor = One, destAlphaFactor = Zero, operationAlpha = Add.
         */
        struct BlendMode {
            BlendFactor sourceRGBFactor;
            BlendFactor destRGBFactor;
            BlendOperation operationRGB;
            BlendFactor sourceAlphaFactor;
            BlendFactor destAlphaFactor;
            BlendOperation operationAlpha;

            BlendMode() : sourceRGBFactor(SourceAlpha), destRGBFactor(InverseSourceAlpha), operationRGB(Add),
                          sourceAlphaFactor(One), destAlphaFactor(Zero), operationAlpha(Add) {}
        };

        /**
         *  StencilFunction.
         */
        struct StencilFunctionData {
            ComparisonFunction compFunc;
            Int32 refValue;
            UInt32 mask;
            StencilFunctionData() : compFunc(CompareAlways), refValue(0), mask(0xFFFFFFFF) {}
            StencilFunctionData(ComparisonFunction initCompFunc, Int32 initRefValue = 0, UInt32 initMask = 0xFFFFFFFF) :
                compFunc(initCompFunc), refValue(initRefValue), mask(initMask) {}
        };

        /**
         *  StencilOperation.
         */
        struct StencilOperationData {
            StencilOperation stencilFail;
            StencilOperation depthFail;
            StencilOperation depthPass;
            StencilOperationData() : stencilFail(Keep), depthFail(Keep), depthPass(Keep) {}
            StencilOperationData(StencilOperation initStencilFail, StencilOperation initDepthFail, StencilOperation initDepthPass) :
                stencilFail(initStencilFail), depthFail(initDepthFail), depthPass(initDepthPass) {}
        };

        /**
         *  DepthBias.
         */
        struct DepthBias {
            Float scaleFactor;
            Float units;
            DepthBias() : scaleFactor(0.0F), units(0.0F) {}
            DepthBias(Float initScaleFactor, Float initUnits): scaleFactor(initScaleFactor), units(initUnits) {}
        };

        /**
         *  Creates an instance of this class.
         *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
         */
        static RenderMode::SharedPointer Create();

        /**
         *  Destructor
         */
        virtual ~RenderMode() override;

        /**
        *  Create a copy of this material.
        *  @return   A Shared pointer to the copy.
        */
        RenderMode::SharedPointer Clone() const;

        /**
         *  Set culling mode, which determines the side of polygon to remove before rasterization.
         *  Default value is: BackFaceCulling.
         *  @param mode Culling mode which determines the side of polygon to remove before rasterization.
         *              Can be FrontFaceCulling, BackFaceCulling and NoCulling.
         */
        void SetCulling(Culling mode) { m_cullMode = mode; }

        /**
         *  Retrieve the current culling mode, which determines the side of polygon to remove before rasterization.
         *  @return The current culling mode.
         */
        Culling GetCulling() const { return m_cullMode; }

        /**
         *  Set winding mode that specifies the polygon's front face in screen-space coordinates.
         *  Default value is: CounterClockWise.
         *  @param mode The winding mode to be set. Can be CounterClockWise or ClockWise.
         */
        void SetWinding(Winding mode) { m_windMode = mode; }

        /**
         *  Retrieve the current winding, specifying the polygon's front face in screen-space coordinates.
         *  @return The current winding.
         */
        Winding GetWinding() const { return m_windMode; }

        /**
         *  Enable or disable writing of the incoming fragment's color value into color buffer.
         *  @param enableRed true enables and false disables writing to color buffer.
         *  @param enableGreen true enables and false disables writing to color buffer.
         *  @param enableBlue true enables and false disables writing to color buffer.
         *  @param enableAlpha true enables and false disables writing to color buffer.
         *  Per default writing to each color component is enabled.
         */
        void SetColorWriteEnabled(bool enableRed, bool enableGreen, bool enableBlue, bool enableAlpha);

        /**
         *  Enable or disable writing of the incoming fragment's color red value into color buffer.
         *  @param enableRed true enables and false disables writing to color buffer.
         */
        void SetColorWriteRedEnabled(bool enableRed) { m_isColorWriteRedEnabled = enableRed; }

        /**
         *  Retrieves whether writing color red into the color buffer is enabled or not.
         *  @return True when writing is enabled, false otherwise
         */
        bool IsColorWriteRedEnabled() const { return m_isColorWriteRedEnabled; }

        /**
         *  Enable or disable writing of the incoming fragment's color green value into color buffer.
         *  @param enableGreen true enables and false disables writing to color buffer.
         */
        void SetColorWriteGreenEnabled(bool enableGreen) { m_isColorWriteGreenEnabled = enableGreen; }

        /**
         *  Retrieves whether writing color green into the color buffer is enabled or not.
         *  @return True when writing is enabled, false otherwise
         */
        bool IsColorWriteGreenEnabled() const { return m_isColorWriteGreenEnabled; }

        /**
         *  Enable or disable writing of the incoming fragment's color blue value into color buffer.
         *  @param enableBlue true enables and false disables writing to color buffer.
         */
        void SetColorWriteBlueEnabled(bool enableBlue) { m_isColorWriteBlueEnabled = enableBlue; }

        /**
         *  Retrieve whether writing color blue into the color buffer is enabled or not.
         *  @return True when writing is enabled, false otherwise
         */
        bool IsColorWriteBlueEnabled() const { return m_isColorWriteBlueEnabled; }

        /**
         *  Enable or disable writing of the incoming fragment's color alpha value into color buffer.
         *  @param enableAlpha true enables and false disables writing to color buffer.
         */
        void SetColorWriteAlphaEnabled(bool enableAlpha) { m_isColorWriteAlphaEnabled = enableAlpha; }

        /**
         *  Retrieve whether writing color value alpha into the color buffer is enabled or not.
         *  @return True when writing is enabled, false otherwise
         */
        bool IsColorWriteAlphaEnabled() const { return m_isColorWriteAlphaEnabled; }

        /**
         *  Enable or disable writing of the incoming fragment's depth value into depth buffer.
         *  If the depth buffer test is disabled this setting will be ignored and no writing
         *  to the depth buffer will occur.
         *  @param enable: true enables and false disables writing to depth buffer.
         *  Default value is: true.
         */
        void SetDepthWriteEnabled(bool enable) { m_isDepthWriteEnabled = enable; }

        /**
         *  Retrieve whether writing of the incoming fragment's depth value into depth buffer is enabled or not.
         *  @return True if writing is enabled, false otherwise.
         */
        bool IsDepthWriteEnabled() const { return m_isDepthWriteEnabled; }

        /**
         *  Enable or disable the test of the incoming fragment's depth value against the depth buffer.
         *  @param enable: true enables and false disables depth buffer test.
         *  Default value is: true.
         */
        void SetDepthTestEnabled(bool enable) { m_isDepthTestEnabled = enable; }

        /**
         *  Retrieve whether the test of the incoming fragment's depth value against the depth buffer is enabled or not.
         *  @return True if the test is enabled, false otherwise.
         */
        bool IsDepthTestEnabled() const { return m_isDepthTestEnabled; }

        /**
         *  Set the depth test comparison function.
         *  @param compFunc The depth test comparison function to be set.
         */
        void SetDepthComparisonFunction(ComparisonFunction compFunc) { m_depthCompFunc = compFunc; }

        /**
         *  Retrieve the depth test comparison function.
         *  @return The depth test comparison function.
         */
        ComparisonFunction GetDepthComparisonFunction() const { return m_depthCompFunc; }

        /**
         *  Enable or disable blending according to the BlendMode set.
         *  @param enable: true enables and false disables blending.
         *  Default value is: false.
         */
        void SetBlendingEnabled(bool enable) { m_isBlendingEnabled = enable; }

        /**
         *  Retrieve whether blending according to the BlendMode set is enabled or not.
         *  @return True if blending is enabled, false otherwise.
         */
        bool IsBlendingEnabled() const { return m_isBlendingEnabled; }

        /**
         *  Set BlendMode for RGBA. Selects a method of combining the incoming fragments RGBA values
         *  with the RGBA values already in the frame buffer.
         *  Blending Equation: final_color = source_color *  sourceFactor (+operation+) dest_color *  destFactor.
         *  Default values are: sourceRGBFactor(SourceAlpha), destRGBFactor(InverseSourceAlpha), operationRGB(Add),
         *                      sourceAlphaFactor(One), destAlphaFactor(Zero), operationAlpha(Add),
         *  @param sourceFactor defines the source blend factor for both RGB and alpha values.
         *  @param destFactor defines the destination blend factor for both RGB and alpha values.
         *  @param operation defines the blend operation for both RGB and alpha values.
         */
        void SetBlendMode(BlendFactor sourceFactor, BlendFactor destFactor, BlendOperation operation);

        /**
         *  Set BlendMode for RGB and Alpha separately.
         *  @param sourceRGBFactor defines the source blend factor for RGB values.
         *  @param destRGBFactor defines the destination blend factor for RGB values.
         *  @param operationRGB defines the blend operation for RGB values.
         *  @param sourceAlphaFactor defines the source blend factor for alpha value.
         *  @param destAlphaFactor defines the destination blend factor the alpha value.
         *  @param operationAlpha defines the blend operation for alpha value.
         */
        void SetBlendModeSeparate(BlendFactor sourceRGBFactor, BlendFactor destRGBFactor, BlendOperation operationRGB,
                                  BlendFactor sourceAlphaFactor, BlendFactor destAlphaFactor, BlendOperation operationAlpha);

        /**
         *  Retrieves the blend mode set for this RenderMode object.
         *  @return blend mode specified for RGBA.
         */
        const BlendMode& GetBlendMode() const { return m_blendMode; }

        /**
         *  Set the blending color to use in blending equation when BlendFactor is:
         *  ConstantColor, InverseConstantColor, ConstantAlpha, or InverseConstantAlpha.
         *  Default value is: Color(0.0F, 0.0F, 0.0F, 1.0F).
         *  @param color Blending color to be used in blending equations with BlendFactor ConstantColor.
         */
        void SetBlendColor(const Color& color) { m_blendColor = color; }

        /**
         *  Retrieve the blend color.
         *  @return The blend color.
         */
        const Color& GetBlendColor() const { return m_blendColor; }

        /**
         *  Enable or disable the stencil test against the stencil buffer.
         *  @param enable: true enables and false disables stencil buffer test.
         *  Default value is: false.
         */
        void SetStencilTestEnabled(bool enable) { m_isStencilTestEnabled = enable; }

        /**
         *  Returns if the stencil test is enabled for this render mode.
         *  @return if the stencil test is enabled.
         */
        bool IsStencilTestEnabled() const { return m_isStencilTestEnabled; }

        /**
         *  Set stencil function and reference value for stencil testing.
         *  Equation for Stencil Test: (refValue & mask) compFunc (stencil & mask).
         *  Front face stencil affects non-polygons and front-facing polygons whereas back face stencil affects back-facing polygons only.
         *  @param function specifies the comparison function for the stencil test.
         *  @param face specifies the stencil plane associated with the provided stencil function.
         *  Default value is: compFunc(Always), refValue(0), mask(0xFFFFFFFF).
         */
        void SetStencilFunction(const StencilFunctionData& function, const StencilPlane face = FrontAndBackFace);

        /**
         *  Get stencil function for the specified face. If FrontAndBackFace is passed as an argument the
         *  function for the front face is returned.
         *  Front face stencil affects non-polygons and front-facing polygons whereas back face stencil affects back-facing polygons only.
         *  @param face specifies the face for which the stencil function is requested.
         *  @return the requested stencil function.
         */
        const StencilFunctionData& GetStencilFunction(const StencilPlane face = FrontFace) const;

        /**
         *  Set stencil operations applied to the stencil buffer for three conditions:
         *  Condition 1: The fragment fails the stencil test. (Subsequent depth test and other tests will not be applied.)
         *  Condition 2: The fragment fails the depth test after the stencil test passed.
         *  Condition 3: The fragment passes both the stencil and depth test.
         *  Front face stencil affects non-polygons and front-facing polygons whereas back face stencil affects back-facing polygons only.
         *  @param operation defines the operations for the conditions mentioned above.
         *  @param face specifies whether front and / or back stencil plane is updated.
         *  Default value is: stencilFail(Keep), depthFail(Keep), depthPass(Keep).
         */
        void SetStencilOperation(const StencilOperationData& operation, const StencilPlane face = FrontAndBackFace);

        /**
         *  Get stencil operation for the specified face. If FrontAndBackFace is passed as an argument the
         *  operation for the front face is returned.
         *  Front face stencil affects non-polygons and front-facing polygons whereas back face stencil affects back-facing polygons only.
         *  @param face specifies the face for which the stencil operation is requested.
         *  @return the requested stencil operation.
         */
        const StencilOperationData& GetStencilOperation(const StencilPlane face = FrontFace) const;

        /**
         *  Set the write mask applied to values written into the stencil buffer.
         *  Front face stencil affects non-polygons and front-facing polygons whereas back face stencil affects back-facing polygons only.
         *  @param writeMask  specifies a bit mask to enable and disable writing of individual bits in the stencil planes.
         *  @param face       specifies whether front and / or back stencil write mask is updated.
         *  Default value is: 0xFFFFFFFF.
         */
        void SetStencilWriteMask(const UInt32 writeMask, const StencilPlane face = FrontAndBackFace);

        /**
         *  Get stencil write mask for the specified face. If FrontAndBackFace is passed as an argument the
         *  write mask for the front face is returned.
         *  Front face stencil affects non-polygons and front-facing polygons whereas back face stencil affects back-facing polygons only.
         *  @param face specifies the face for which the stencil write mask is requested.
         *  @return the requested stencil write mask.
         */
        UInt32 GetStencilWriteMask(const StencilPlane face = FrontFace) const;

        /**
         *  Set the depth bias that can be applied to co-planar primitives to reduce z-fighting, according to following function:
         *  Depth bias = (max *  scaleFactor) + (r *  units); where "max" is the maximum depth slope of the triangle being rendered and "r"
         *  is an implementation-defined constant that is guaranteed to produce the smallest resolvable offset.
         *  @param scaleFactor specifies a scale factor that is used to determine how much bias can be applied to primitives. The default value is 0.
         *  @param units defines a constant depth offset used to determine how much bias can be applied to primitives. The initial value is 0.
         *
         *  Note: Polygons that are coplanar can be made to appear not coplanar by adding a z-bias to each one. This is a technique commonly used to
         *  ensure that shadows, decals, or hidden-line images on coplanar surfaces are displayed properly. The depth bias is added before the
         *  depth test is performed but does not influence the original depth value written into depth buffer.
         */
        void SetDepthBias(Float scaleFactor, Float units);

        /**
         *  Get depth bias, which is the offset applied to primitive's depth values for depth tests comparisons.
         *  @return the depth bias including the scale factor and units. For details how to set the depth bias see RenderMode::SetDepthBias.
         */
        const DepthBias& GetDepthBias() const { return m_depthBias; }

        /**
         *  Enable or disable rasterizer discard.
         *  Note: This state only takes effect on ES3.0 platforms.
         *  @param enable  True enables, and false disables rasterizer discard.
         *  Default value is: false.
         */
        void SetRasterizerDiscardEnabled(bool enable) { m_isRasterizerDiscardEnabled = enable; }

        /**
         *  Retrieve whether rasterizer discard is enabled or not.
         *  @return True if rasterizer discard is enabled, false otherwise.
         */
        bool IsRasterizerDiscardEnabled() const { return m_isRasterizerDiscardEnabled; }

        /**
         *  Defines the flags used in the inheritance mask. Attributes can be marked as "inherited"; if they are, the value for them is taken not 
         *  from this RenderMode,  but from a "base" RenderMode instead.
         */
        enum InheritanceBit
        {
            CullingModeBit = 0x1U,                      ///< CullingModeBit
            WindingModeBit = 0x2U,                      ///< WindingModeBit
            DepthWriteEnabledBit = 0x4U,                ///< DepthWriteEnabledBit
            DepthTestEnabledBit = 0x8U,                 ///< DepthTestEnabledBit
            StencilTestEnabledBit = 0x10U,              ///< StencilTestEnabledBit
            DepthComparisonFunctionBit = 0x20U,         ///< DepthComparisonFunctionBit
            BlendingEnabledBit = 0x40U,                 ///< BlendingEnabledBit
            BlendModeBit = 0x80U,                       ///< BlendModeBit
            BlendColorBit = 0x100U,                     ///< BlendColorBit
            StencilFunctionBackPlaneBit = 0x200U,       ///< StencilFunctionBackPlaneBit
            StencilFunctionFrontPlaneBit = 0x400U,      ///< StencilFunctionFrontPlaneBit
            StencilOperationBackPlaneBit = 0x800U,      ///< StencilOperationBackPlaneBit
            StencilOperationFrontPlaneBit = 0x1000U,    ///< StencilOperationFrontPlaneBit
            StencilWriteMaskBackPlaneBit = 0x2000U,     ///< StencilWriteMaskBackPlaneBit
            StencilWriteMaskFrontPlaneBit = 0x4000U,    ///< StencilWriteMaskFrontPlaneBit
            ColorWriteEnabledBit  = 0x8000U,            ///< ColorWriteEnabledBit
            DepthBiasBit = 0x10000U,                    ///< DepthBiasBit
            RasterizerDiscardBit = 0x20000U             ///< RasterizerDiscardBit
        };

        /**
         *  "Inheritance concept": Settings can be marked as "inherited"; if they are, the value for them is taken not from this RenderMode,
         *  but from a "base" RenderMode instead.
         *  Default mask: 0 (nothing is inherited)
         *  @param inheritanceMask defines all inheritance bits in one UInt32 bit mask. If a bit is set to one, then the associated
         *  RenderMode setting is inherited. The inheritanceMask set completely overwrites the previous inheritanceMask.
         */
        void SetInheritanceMask(UInt32 inheritanceMask);

        /**
         *  "Inheritance concept": Settings can be marked as "inherited"; if they are, the value for them is taken not from this RenderMode,
         *  but from a "base" RenderMode instead.
         *  Default mask: 0 (nothing is inherited)
         *  @param inheritanceBitSelection: bitwise OR of InheritanceBit values with InheritanceMask.
         */
        void SetInheritanceBitsToOne(UInt32 inheritanceBitSelection);

        /**
         *  "Inheritance concept": settings can be marked as "inherited"; if they are, the value for them is taken not from this RenderMode,
         *  but from a "base" RenderMode instead.
         *  Default mask: 0 (nothing is inherited)
         *  @param inheritanceBitSelection: bitwise OR of InheritanceBit values with InheritanceMask.
         */
        void SetInheritanceBitsToZero(UInt32 inheritanceBitSelection);

        /**
         *  Retrieve the current inheritance mask.
         *  @return The current inheritance mask.
         */
        UInt32 GetInheritanceMask() const;

        /**
         *  Retrieve whether the bit is inherited or not.
         *  @param bit The bit to be examined.
         *  @return True if the bis is inherited, false otherwise
         */
        bool IsInherited(InheritanceBit bit) const;

        /**
         *  Activates this RenderMode's render attributes in RenderDevice. "Inheritance" of settings is applied.
         *  @param baseMode if a setting is marked as "inherited" (see inheritance mask functions), its value is taken from baseMode
         *  @return False if any render attribute couldn't be activated, true otherwise.
         */
        bool Activate(const RenderMode::SharedPointer& baseMode) const;

        /**  Activates this RenderMode's render attributes in RenderDevice.
         *  Any "inheritance" settings are ignored, i.e. the render state settings of this RenderMode are applied, without consulting any
         *  other RenderMode object.
         *  @return False if any render attribute couldn't be activated, true otherwise.
         */
        bool Activate() const;

        /**
         * Returns the Id of this vertex buffer instance.
         * @return The Id of this vertex buffer instance.
         */
        Int GetInstanceId() const { return m_instanceId; }

        FEATSTD_RTTI_DECLARATION();

    private:
        Internal::InstanceId<RenderMode> m_instanceId;

        //Per-fragment compositing attributes
        bool m_isColorWriteRedEnabled;       // Default value: true
        bool m_isColorWriteGreenEnabled;     // Default value: true
        bool m_isColorWriteBlueEnabled;      // Default value: true
        bool m_isColorWriteAlphaEnabled;     // Default value: true
        bool m_isDepthWriteEnabled;          // Default value: true
        bool m_isDepthTestEnabled;           // Default value: true
        bool m_isStencilTestEnabled;         // Default value: false
        bool m_isBlendingEnabled;            // Default value: false
        bool m_isRasterizerDiscardEnabled;   // Default value: false

        //Polygon-level attributes
        UInt32 m_stencilWriteMaskBackPlane; // Default value: 0xFFFFFFFF
        UInt32 m_stencilWriteMaskFrontPlane; // Default value: 0xFFFFFFFF
        UInt32 m_inheritanceMask;
        Culling m_cullMode;             // Default value: BackFaceCulling
        Winding m_windMode;             // Default value: CounterClockWise
        BlendMode m_blendMode;          // Default values for RGB: SourceAlpha, InverseSourceAlpha, Add; Default values for Alpha: One, Zero, Add.
        Color m_blendColor;                  // Default value: Color(0.0F, 0.0F, 0.0F, 1.0F)
        ComparisonFunction m_depthCompFunc;  // Default value: Less
        StencilFunctionData m_stencilFunctionBackPlane; // Default value: compFunc(Always), refValue(0), mask(0xFFFFFFFF)
        StencilFunctionData m_stencilFunctionFrontPlane; // Default value: compFunc(Always), refValue(0), mask(0xFFFFFFFF)
        StencilOperationData m_stencilOperationBackPlane; // Default value: stencilFail(Keep), depthFail(Keep), depthPass(Keep)
        StencilOperationData m_stencilOperationFrontPlane; // Default value: stencilFail(Keep), depthFail(Keep), depthPass(Keep)
        DepthBias m_depthBias;               // Default value: scaleFactor = 0.0F, units = 0.0F

        // private because only ::Create() should be used to create an instance of this class
        FEATSTD_LINT_NEXT_EXPRESSION(1704, "Instance accessible via Create.")
        RenderMode();
        RenderMode(const RenderMode& renderMode);
        RenderMode& operator=(const RenderMode& renderMode);

        bool AffectsFrontFace(const StencilPlane face) const { return ((face == FrontAndBackFace) || (face == FrontFace)); }

        bool AffectsBackFace(const StencilPlane face) const { return ((face == FrontAndBackFace) || (face == BackFace)); }

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif    // CANDERA_RENDERMODE_H

