//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CanderaReflectionCamera_h)
#define CanderaReflectionCamera_h

#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief   A ReflectionCamera represents a reflected view of an associated source camera (See function SetSourceCamera).
 *          As the ReflectionCamera is synchronized with the view and projection from the source camera, its own transformation parameters are ignored, these are:
 *          Scale, Rotation, Translation, Generic Transform, LookAt-, LookUp-, and RightVector.
 *          Other parameters like RenderTarget, Viewport, ScopeMask, ClearMode, RenderMode are decoupled from source camera, and must be set explicitly for each
 *          ReflectionCamera object.
 *          Note: To provide correct culling of reflected triangles, set winding opposed to the source camera.(see RenderMode::SetWinding).
 */

class ReflectionCamera : public Camera
{
    FEATSTD_TYPEDEF_BASE(Camera);

    public:
        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return A pointer to the created ReflectionCamera object.
         */
        static ReflectionCamera* Create();

        /**
         *  Destructor
         */
        virtual ~ReflectionCamera() override;

        /**
         *  Clones this Node only.
         *  Associated nodes like source camera are not referenced from the copy.
         *  @return  The pointer to the cloned Node if successful, null otherwise.
         */
        virtual ReflectionCamera* Clone() const override;

        /**
         *  Sets the reflection plane along a Node given is mirrored.
         *  @param plane defines the reflection plane.
         */
        void SetReflectionPlane(const Plane& plane);

        /**
         *  Retrieves reflection plane that reflects the view of the source camera.
         *  @return the normalized plane's normal vector.
         */
        const Plane& GetReflectionPlane() const { return m_plane; }

        /**
         *  Sets the source camera that includes the view matrix that is reflected.
         *  @param camera specifies the source camera node. The view and projection of this object are synchronized from
         *  the source camera, automatically. If parameter camera is null or points to this object, then the ReflectionCamera object is not rendered.
         */
        void SetSourceCamera(Camera* camera) { m_sourceCamera = (camera != this) ? camera : 0; }

        /**
         *  Retrieve the source camera that includes the view matrix that is reflected.
         *  @return The source camera that includes the view matrix that is reflected.
         */
        Camera* GetSourceCamera() { return m_sourceCamera; }
        const Camera* GetSourceCamera() const { return m_sourceCamera; }

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Deliberately non-public Constructor and Copy-Constructor, use function Create() to create an instance of this object.
        ReflectionCamera();
        ReflectionCamera(const ReflectionCamera& reflectionCamera);

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

        /**
         *  Activates this Camera and its ClearMode states in RenderDevice.
         *  @return false if activation fails (E.g. if no source camera is set).
         */
        virtual bool Activate() override;

    private:
        // Copying with assignment operator is not supported.
        ReflectionCamera& operator = (const ReflectionCamera& reflectionCamera);

        Camera* m_sourceCamera; // Associated source camera that is reflected.
        Plane m_plane;          // Plane along the Camera's view matrix is reflected.

        /**
         *  Reflects the Camera's view matrix along the reflection plane.
         */
        void ReflectView();

        /**
         *  Copies view and projection matrix of source camera associated (see function SetReflectedCamera).
         */
        void SyncViewProjectionFromSourceCamera();

        CdaDynamicProperties(Candera::ReflectionCamera, Candera::Camera);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif    // CanderaReflectionCamera_h

