//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CanderaQueryProperty_h)
#define CanderaQueryProperty_h

#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/NodeListener.h>
#include <Candera/Engine3D/Core/Query.h>
#include <Candera/System/Container/Vector.h>

namespace Candera {

    class Mesh;
    class OcclusionCullingCameraRenderStrategy;

    namespace Internal {

        class QueryProperty : public NodeListener {
        
        FEATSTD_TYPEDEF_BASE(NodeListener);

        public:

            /**
             * Create and attach a QueryProperty object to a Node.
             * @param  node                                  Defines the Node which the QueryProperty object is attached to.
             * @param  occlusionCullingCameraRenderStrategy  The OcclusionCullingCameraRenderStrategy to issue the query for.
             * @param  type                                  Query::Type to be used for the query.
             * @return true, if the QueryProperty has been attached successfully, false otherwise.
             */
            static bool Set(Node& node, const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy,
                Query::Type type = Query::QueryAnySamplesPassedConservative);

            /**
             * Remove the QueryProperty of the given node from the given occlusion culling camera render strategy.
             * @param  node                                  Defines the Node which the QueryProperty is attached to.
             * @param  occlusionCullingCameraRenderStrategy  The OcclusionCullingCameraRenderStrategy to remove the query from.
             * @return true, if the QueryProperty was removed from the occlusionCullingCameraRenderStrategy,
             *         false, if the node does not have a QueryProperty or if no occlusionCullingCameraRenderStrategy was set.
             */
            static bool Remove(Node& node, const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy);

            /**
             * Remove the QueryProperty from the given occlusion culling camera render strategy.
             * @param  node                                  Defines the Node which the QueryProperty is attached to.
             * @param  occlusionCullingCameraRenderStrategy  The OcclusionCullingCameraRenderStrategy to remove the query from.
             * @return true, if the QueryProperty was removed from the occlusionCullingCameraRenderStrategy,
             *         false, if the node does not have a QueryProperty or if no occlusionCullingCameraRenderStrategy was set.
             */
            bool Remove(const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy);

            /**
             * Retrieve a QueryProperty object that is attached to a Node.
             * @param  node  Defines the Node to retrieve the QueryProperty object from.
             * @return The pointer to QueryProperty object attached to the Node, or 0 if no QueryProperty has been attached.
             */
            static QueryProperty* Get(const Node& node);

            /**
             * Retrieve a default QueryProperty object to fulfill (unregistered) DynamicProperty interface.
             * @return Default QueryProperty object.
             */
            static const QueryProperty& GetDefault();

            /**
             * Constructs a QueryProperty object.
             */
            QueryProperty();

            /**
             * Destructs the QueryProperty object.
             */
            ~QueryProperty();

            /**
             * CopyConstructor required to fulfill (unregistered) DynamicProperty interface.
             */
            QueryProperty(const QueryProperty& queryProperty);

            /**
             * Operator == required to fulfill (unregistered) DynamicProperty interface.
             */
            inline bool operator==(const QueryProperty&) const { return false; } // always return false, no need to serve DynamicPropertyDefaultValue

            /**
             * Operator = required to fulfill (unregistered) DynamicProperty interface.
             */
            QueryProperty& operator=(const QueryProperty& other);

            /**
             * Transforms the node's bounding box to world space and draws it.
             * @param node  Node to be rendered
             */
            void RenderBoundingBox(const Node* node) const;

            /**
            * Is invoked when the transformation of a node is assumed to change.
            * @param Node The node to listen to.
            */
            virtual void OnTransformChange(Node*) override {};

            /**
            * Invoked whenever the node is removed from its parent.
            */
            virtual void OnNodeRemoved(Node* parent, Node* node) override;

            /**
             * The QueryInfo structure performs actual queries and holds information about the associated camera render strategy.
             */
            struct QueryInfo {
            public:
                QueryInfo()
                    :
                    m_occlusionCullingCameraRenderStrategy(0),
                    m_wasQuerySubmitted(false) {}

                bool Begin() const { return m_query->Begin(); }
                bool End() const { m_wasQuerySubmitted = true; return m_query->End(); }

                UInt32 GetResult() const { return m_query->GetResult(); }
                bool IsResultAvailable() const { return m_wasQuerySubmitted ? m_query->IsResultAvailable() : false; }

            private:
                friend class QueryProperty;
                Query::SharedPointer m_query;
                const OcclusionCullingCameraRenderStrategy* m_occlusionCullingCameraRenderStrategy;
                mutable bool m_wasQuerySubmitted;
            };

            /**
             * Get the QueryInfo for the given OcclusionCullingCameraRenderStrategy.
             * @param occlusionCullingCameraRenderStrategy  The OcclusionCullingCameraRenderStrategy the QueryInfo is associated with.
             * @return The QueryInfo for the given OcclusionCullingCameraRenderStrategy, or 0 if there is no associated QueryInfo.
             */
            QueryInfo* GetQueryInfo(const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy);

        private:

            friend class ClearQueryPropertyTraverser;
            void CleanupQueryInfos();

            // Create an invisible mesh that does not write Z values for rendering bounding boxes instead of the actual node.
            static Mesh* CreateBoundingBoxMesh();

            struct SingleInstance {
                SingleInstance() : m_boundingBox(0), m_defaultQueryProperty(0), m_referenceCounter(0) {}
                Mesh* m_boundingBox;
                QueryProperty* m_defaultQueryProperty;
                UInt32 m_referenceCounter;
            };

            // Singleton for the bounding box mesh and the default QueryProperty
            static SingleInstance s_Instance;

            // Vector to hold one QueryInfo per OcclusionCullingCameraRenderStrategy
            Internal::Vector<QueryInfo, FeatStd::Internal::LinearIncreasePolicy<1> > m_queryInfos;

            // Flag to determine if this QueryProperty is the default one.
            bool m_isDefault;

            /**
            * MetaInfo required for DynamicProperty Declaration.
            */
            CdaDynamicProperties(Node, DynamicProperties::DynamicPropertyHost);
            CdaDynamicPropertyUnregistered(QueryProperty, QueryProperty);
            CdaDynamicPropertyDefaultValue(GetDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicPropertiesEnd();
        };
    }
}

#endif
