//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(PROXY_TEXTURE_IMAGE_H)
#define PROXY_TEXTURE_IMAGE_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/TextureImage.h>
#include <Candera/Engine3D/Core/RendererListener.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {

    /** @addtogroup Core3D
  *  @{
  */

class ImageSource3D;

/**
 *  @brief The class ProxyTextureImage provides a TextureImage interface to an externally maintained image source, which it wraps.
 *  As an ImageSource object exposes read-only access to pixel buffer the wrapping ProxyTextureImage exposes no RenderTarget interface.
 *  Typically, the ProxyTextureImage is used to gain access to e.g FramebufferObjects or other objects that implement the ImageSource3D interface
 *  to be able to use them as common TextureImage objects like for instance BitmapTextureImage or CubeMapTextureImage.
 */
class ProxyTextureImage : public TextureImage {
    FEATSTD_TYPEDEF_BASE(TextureImage);

public:
    typedef MemoryManagement::Disposer<ImageSource3D*> ImageSourceDisposer;
    typedef ImageSourceDisposer::DisposerFunction ImageSourceDisposerFn;

    /**
     *  Creates a SharedPointer to a ProxyTextureImage which wraps any given image source.
     *  @param imageSource The ImageSource3D object.
     *  @param disposer
     *  @return A shared pointer to the created ProxyTextureImage object.
     */
    static MemoryManagement::SharedPointer<ProxyTextureImage> Create(ImageSource3D* imageSource, ImageSourceDisposerFn disposer = 0);

    /**
     *  Destructs the ProxyTextureImage object.
     */
    virtual ~ProxyTextureImage() override;

    /**
     *  Retrieves the video memory handle.
     *  @return The current video memory handle.
     */
    virtual Handle GetVideoMemoryHandle() const override;

    /**
     *  Retrieve whether mipmapping is enabled or not.
     *  @return True if mipmapping is enabled otherwise false.
     */
    virtual bool IsMipMappingEnabled() const override;

    /**
     * Activates the ProxyTextureImage object at the texture unit given.
     * @param unit defines the texture unit the ProxyTextureImage shall be activated for.
     */
    FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on virtual function matches the base-class argument it overrides. (See TextureImage.h)")
    virtual bool Activate(UInt unit = 0) override;

    /**
     * Sets synchronization enabled or not
     * @param enable defines if synchronization shall be enabled or not.
     */
    void SetSyncEnabled(bool enable) {
        m_listener.m_isSyncEnabled = enable;
    }

    /**
     * Retrieves if synchronization is enabled or not
     * @return True if synchronization is enabled, false otherwise.
     */
    bool IsSyncEnabled() const {
        return m_listener.m_isSyncEnabled;
    }

    /**
     * Returns the ImageSource3D object wrapped by this ProxyTextureImage.
     */
    virtual ImageSource3D* ToImageSource3D() override {
return m_listener.m_imageSource;
    }

    /**
     * TextureTarget of this TextureImage. Specifies whether this texture image represents a 2D texture or a cube-map texture.
     * @return TextureTargetType specifies if texture image is a 2D texture or a cube-map texture.
     */
    virtual TextureTargetType GetTextureTargetType() const override;

    FEATSTD_RTTI_DECLARATION();

protected:
    // Constructs a ProxyTextureImage object with given ImageSource
    ProxyTextureImage(ImageSource3D* imageSource, ImageSourceDisposerFn disposer);

    /**
     * Implementation for child classes where the actual upload happens.
     * @return Whether internal upload succeeded or not. In base class always true.
     */
    virtual bool UploadInternal(LoadingHint loadingHint) override {
        FEATSTD_UNUSED(loadingHint);
        return true;
    }

    /**
     * Implementation for child classes where the actual unload happens.
     * @return Whether internal unload succeeded or not. In base class always true.
     */
    virtual bool UnloadInternal(LoadingHint loadingHint) override {
        FEATSTD_UNUSED(loadingHint);
        return true;
    }

    // override DeviceObject::DisposeInternal
    virtual void DisposeInternal() override {}

private:

    class ProxyTextureRendererListener : public RendererListener
    {
    public:
        explicit ProxyTextureRendererListener(
            ImageSource3D* imageSource, 
            ImageSourceDisposerFn disposer);
        ~ProxyTextureRendererListener();

        virtual void OnNodePreRender(Node* node) override; 
        virtual void OnNodePostRender(Node* node) override;

        bool m_isSyncEnabled;
        ImageSource3D* m_imageSource;
        ImageSourceDisposerFn m_disposer;
    };

    ProxyTextureRendererListener m_listener;
};

/** @} */ // end of Core3D

}

#endif
