//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_PointSprite_H)
#define CANDERA_PointSprite_H

#include <Candera/Engine3D/Core/Renderable.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */


/**
 *  @brief PointSprite represents a GL point primitive.
 *  The size (glPointSize) of the point is calculated by default as
 *    s * SquareRoot(1.0 / (A + B*d + C*d*d))
 *  where s is the size as set by the user via SetPointSize, d is the  distance to the camera,
 *  and A, B, C are scale factors set by the user.
 *  All PointSprites share the same VertexBuffer by default. If a different VertexBuffer is required,
 *  it needs to be set manually per instance.
 */
class PointSprite : public Renderable
{
    FEATSTD_TYPEDEF_BASE(Renderable);

    public:
        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return A pointer to the created PointSprite object.
         */
        static PointSprite* Create();

        /**
         *  Destructor
         */
        virtual ~PointSprite() override;

        /**
         *  Clones this PointSprite only.
         *  Attached PointSprite resources (including VertexBuffer) are not deep-copied but referenced.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual PointSprite* Clone() const override;

        /**
         *  A Color passed to the shader, in order to eg modulate the color given by a texture.
         *  @param color The color to be passed to the shader.
         */
        CANDERA_DEPRECATED_3_1_1(
            "Please use the Diffuse color of the PointSprite's material",
            void SetColor(const Color& color));

        /**
         *  Sets the point size.
         *  @param size The size of the point.
         */
        void SetPointSize(Float size) { m_pointSize = size; }

        /**
         *  Retrieve the point size.
         *  @return The point size.
         */
        Float GetPointSize() const { return m_pointSize; }

        /**
         *  Set the point size scale A.
         *  @param scaleA The point size scale A.
         */
        void SetPointSizeScaleA(Float scaleA) { m_pointSizeScaleA = scaleA; }

        /**
         *  Retrieve the point size scale A.
         *  @return The point size scale A.
         */
        Float GetPointSizeScaleA() const { return m_pointSizeScaleA; }

        /**
         *  Set the point size scale B.
         *  @param scaleB The point size scale B.
         */
        void SetPointSizeScaleB(Float scaleB) { m_pointSizeScaleB = scaleB; }

        /**
         *  Retrieve the point size scale B.
         *  @return The point size scale B.
         */
        Float GetPointSizeScaleB() const { return m_pointSizeScaleB; }

        /**
         *  Set the point size scale C.
         *  @param scaleC The point size scale C.
         */
        void SetPointSizeScaleC(Float scaleC) { m_pointSizeScaleC = scaleC; }

        /**
         *  Retrieve the point size scale C.
         *  @return The point size scale C.
         */
        Float GetPointSizeScaleC() const { return m_pointSizeScaleC; }

        /**
         *  Retrieve the VertexBuffer object.
         *  @return The VertexBuffer object.
         */
        const VertexBuffer* GetVertexBuffer() const { return m_vertexBuffer.GetPointerToSharedInstance(); }

        /**
         *  Set the vertex buffer of this PointSprite. By default PointSprites share the same vertex buffer.
         *  Therefore if the vertex buffer of an individual PointSprite needs to be modified, a new vertex
         *  buffer needs to be set instead.
         *  @param vertexBuffer  The vertex buffer to be set.
         */
        void SetVertexBuffer(VertexBuffer::SharedPointer vertexBuffer) { m_vertexBuffer = vertexBuffer; }

        //overridden functions

        /**
         *  Upload asset data attached to this Node to video memory.
         *  Asset data is TextureImage, Shader, and VertexBuffer.
         *  @return True if vertex buffer and appearance could be uploaded, false otherwise.
         */
        virtual bool Upload() override;

        /**
         *  Unload asset data attached to this Node from video memory.
         *  Asset data is for instance TextureImage, Shader, and VertexBuffer.
         *  @return True if vertex buffer and appearance could be unloaded, false otherwise.
         */
        virtual bool Unload() override;

        FEATSTD_RTTI_DECLARATION();

    protected:

        /**
         *  Renders this PointSprite. This method is overridden from Node.
         */
        virtual void Render() override;

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

        /**
         *  Tests if this sprite intersects with a pick in absolute Viewport coordinate in pixels.
         *  @remark                   This interface is usually used for object selection in scene editors.
         *  @param camera             The view camera. (i.e. the current camera)
         *  @param x                  The x value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param y                  The y value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param distance [out]     The distance from the nearest point off the sprite to the near plane of the camera.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this node, otherwise false.
         */
        virtual bool IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const override;

    private:
        // Explicit private Constructor and Copy-Constructor, use Create() to create an instance of this object.
        FEATSTD_MAKE_CLASS_UNCOPYABLE(PointSprite);
        PointSprite();

        // Set up vertex buffer.
        bool Initialize();
        VertexBuffer::SharedPointer CreateVertexBuffer() const;

        Float m_pointSize;
        Float m_pointSizeScaleA;
        Float m_pointSizeScaleB;
        Float m_pointSizeScaleC;

        CdaDynamicProperties(Candera::PointSprite, Candera::Node);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_PointSprite_H
