//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_PLANAR_SHADOW_H)
#define CANDERA_PLANAR_SHADOW_H

#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/System/Mathematics/Plane.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */
    class VertexBuffer;
    class Light;

/**
 * @brief PlanarShadow implements a simple technique to simulate shadow casting.
 *        Basically a PlanarShadow node is drawn using its world transformation matrix multiplied
 *        with a projection matrix, that projects it's vertices onto a predefined plane with respect
 *        to an associated light source. Only point-, spot- and directional
 *        lights are considered as light sources, ambient lights don't throw shadows.
 *        Per default the vertex buffer of the parent node is taken (if there is any), but it's also
 *        possible to explicitly provide a dedicated vertex buffer which should be rendered as shadow.
 *
 *        To avoid double blending of semi transparent shadows stencil buffers can be used. The PlanarShadow's appearance
 *        already contains a pre-configured RenderMode with stencil buffers configured to only
 *        draw on fragments with a stencil value other than this PlanarShadows unique Id. After drawing, the value gets replaced to
 *        the current stencil reference value,
 *        such that the stencil test doesn't pass any more on future fragments to write.
 *        To get this mechanism running ensure that the camera always clears the stencil buffer to 0 every frame.
 *        Also the GraphicDeviceUnit has to be initialized accordingly to support stencil buffers.
 *        Alternatively also other stencil buffer configurations can be set, using the
 *        RenderMode's properties.
 *
 *        Note: PlanarShadows are an algorithmically very efficient and parameterizable way to simulate simple flat shadows, but they
 *        also have their limitations. Only flat planar surfaces are considered as area to draw the shadow on. So, shadowing on arbitrary objects cannot be
 *        achieved using this technique. Further, every object that intends to cast a shadow must have a dedicated shadow child node for each light source as well as for each plane acting as shadow receiver.
 *        This results from the fact that the shadow is simulated with adding shadow geometry
 *        as opposed to other shadowing techniques that reduce light.
 *        Additionally to consider shadows of the transformations done in the vertex shader, dedicated shader for the shadows have to be written.
 *        Such transformations include e.g. morphing meshes or displacement mapping.
 */
class PlanarShadow:  public Mesh
{
    FEATSTD_TYPEDEF_BASE(Mesh);

    public:

        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  The created PlanarShadow object already contains a predefined appearance with
         *  a pre-configured stencil buffer configuration to avoid double blending.
         *  @return Pointer to the created PlanarShadow object.
         */
        static PlanarShadow* Create();

        /**
         *  Destructor.
         */
        virtual ~PlanarShadow() override;

        /**
         *  Clones this Node only.
         *  Attached Node resources like Appearance or VertexBuffer are not deep-copied but referenced.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual PlanarShadow* Clone() const override;

        /**
         *  Retrieves the world transformation matrix of this Node concatenated with the shadow projection matrix.
         *  Multiplies local transformation with transformations of all ancestors to this Node. Additionally multiplies
         *  the planar projection matrix with respect to light source and the projection plane.
         *  World Transformation itself is cached as used from the base class. Calculation of shadow projection matrix
         *  and multiplication with world transformation is done every frame.
         *  @return World transformation matrix concatenated with planar projection matrix of this PlanarShadow.
         */
        virtual const Matrix4& GetWorldTransform() const override;

        /**
         * Sets Plane used to project the shadow onto.
         * @param plane Plane to set.
         */
        void SetPlane(const Plane& plane) { m_plane = plane; }
        /**
         * Gets Plane on which the shadow is projected onto.
         * @return Plane on which shadow is projected onto.
         */
        const Plane& GetPlane() const { return m_plane; }

        /**
         * Sets the light associated to this PlanarShadow object.
         * The shadow's projection is influenced by exactly one light.
         * Thus, multiple shadow casts of one geometry induced by multiple
         * lights consequently require multiple PlanarShadow objects.
         * @param light The associated light to be set.
         */
        void SetLight(Light* light) { m_associatedLight = light; }

        /**
         * Gets the light associated to this PlanarShadow object.
         * @return The associated light.
         */
        Light* GetLight() const { return m_associatedLight; }

        /**
         * Sets the Node to which the shadow's receiving Plane is aligned. 
         * If a node is set the Plane orientation is interpreted as being in object space.
         * If no node is set (node is null), the Plane orientation is interpreted as being in world space.
         * Per default no alignment node is set.
         *
         * @param alignmentNode The node to align the Plane to.
         */
        void SetAlignmentNode(Node* alignmentNode) { m_alignmentNode = alignmentNode; }

        /**
         * Get the Node to which the PlanarShadow's receiving Plane is aligned.
         * @return The Node to which the Plane is aligned.
         */
        Node* GetAlignmentNode() const { return m_alignmentNode; }

        /**
         * Enables or disables automatic parent vertex buffer usage.
         * If this property is enabled then every time an ancestor of this PlanarShadow is removed or added,
         * the vertex buffer of the parent node is taken for rendering, even if it was set manually in between.
         * If automatic setting of vertex buffer is disabled, then the vertex buffer is assumed to be set manually and will not be overwritten.
         * Note: In both cases, if a VertexBuffer object is set manually after Uploading the PlanarShadow, and if the parent vertex buffer
         * gets auto assigned, the caller has to take care that the PlanarShadow object gets unloaded and uploaded manually.
         * @param enable Enables/disables the auto vertex buffer mechanism.
         */
        void SetAutoVertexBufferEnabled(bool enable) { m_isAutoVertexBufferEnabled = enable;}
        /**
         * Gets whether the mechanism that automatically the parent vertex buffer is taken is enabled or disabled.
         * If this property is enabled then every time an ancestor of this PlanarShadow is removed or added,
         * the vertex buffer of the parent node is taken for rendering, even if it was set manually in between.
         * @return Returns whether the auto vertex buffer mechanism is enabled or disabled.
         */
        bool IsAutoVertexBufferEnabled() const { return m_isAutoVertexBufferEnabled; }


        /**
         * Sets Id of this PlanarShadow. By default a static counter is used to generate Id's for each
         * PlanarShadow object. This id is used to setup predefined stencil buffer settings, such
         * that every shadow can be drawn without double blending artifacts, but can blend with each other.
         * This id can be manually overwritten using this function. If the set id is higher than
         * the static counter, the static counter is set to this value, such that subsequently created
         * PlanarShadow objects also get an unique identifier.
         * The automatically set Id's start with 1, such that the stencil buffer can be cleared with value 0.
         * Note: Stencil buffer usage is only pre configured at creation time of a PlanarShadow instance, if id changes they have to be set
         * manually using the RenderMode's properties.
         * @param id Id to set.
         */
        void SetStencilBufferId(Int32 id);

        /**
         * Gets the id of this PlanarShadow.
         * @return Id of this PlanarShadow.
         */
        Int32 GetStencilBufferId() const { return m_Id; }

        /**
         *  Overrides Mesh::IsRenderPrerequisiteFulfilled.
         *  @return true if precondition to be rendered is fulfilled, false otherwise.
         *  The render preconditions consists of the mesh ones. Additionally
         *  an associated light that's no ambient light has to be set.
         */
        virtual bool IsRenderPrerequisiteFulfilled() const override;

        FEATSTD_RTTI_DECLARATION();
    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        PlanarShadow();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(PlanarShadow);

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

         /**
         *  After an ancestor is added, the parent's vertex buffer will be taken
         *  to render this PlanarShadow object, if IsAutoVertexBufferEnabled is true.
         *  Note: Upload and Unload has to be done explicitly by application code.
         *  @param scene The Scene node which became root of this node's scene graph.
         */
        virtual void OnAncestorAdded(Scene* scene) override;

        /**
         *  After removal of an ancestor the current vertex buffer will be unloaded and set to 0 if IsAutoVertexBufferEnabled
         *  is true.
         *  Note: Upload and Unload has to be done explicitly by application code.
         *  @param scene The Scene node which previously had been the root of the scene graph this node was
         *  connected with.
         */
        virtual void OnAncestorRemoved(Scene* scene) override;

    private:
        bool m_isAutoVertexBufferEnabled;
        Int32 m_Id;
        Light* m_associatedLight;
        Plane m_plane;
        Node* m_alignmentNode;

        mutable Matrix4 m_shadowTransformation;

        /**
         * Returns the vertex buffer of the parent node up in the scene graph hierarchy, if there is any.
         * Namely Billboards, LineLists, Meshes and their subclasses are supported.
         * @return The vertex buffer of the parent node up in the scene graph hierarchy, if there is any.
         */
        MemoryManagement::SharedPointer<VertexBuffer> GetParentVertexBuffer() const;

        CdaDynamicProperties(Candera::PlanarShadow, Candera::Mesh);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_PLANAR_SHADOW_H
